####################################################################################
##                                                                                ##
##  I N T E R S T E L L A R   C O N S O L E  A tool to correct Milky Way texture  ##
##                       Copyright (C) 2025- Misato Observatory, Wakayama, Japan  ##
##                                                                                ##
##  This software is available under the GPL (General Public License)             ##
##                                                                                ##
####################################################################################

# This is required to dump canvas widget
package require Img

# file name to read
set orig_file_name "milkyway.orig.png"
if { 0 < $argc } {
  set orig_file_name [regsub -all {\\} [lindex $argv 0] {/}]
}

# size of image (size will be rescaled to 2x)
set width  1024
set height 512

#
# Parameters "bottom,top,curve" for color corrections using +level option of
# ImageMagick.  See https://usage.imagemagick.org/color_mods/#level
#
# orig -(1)-> _tmp_color_img.tiff ---+
#                                 (lighten)-> _tmp_composite_img.tiff -(3)-> _tmp_result_img.png
# orig -(2)-> _tmp_mono_img.tiff ----+
#

# (1) for _tmp_color_img.tiff
set level_bright_r "0%,90%,0.54"
set level_bright_g "4%,80%,0.64"
set level_bright_b "0%,80%,0.89"

# (2) for _tmp_mono_img.tiff
set level_dark     "0%,50%,0.35"

# (3) for _tmp_result_img.png
set level_all      "-1%,100%,1.0"

#
# parameters for adjustment of local brightness
# (See the result: filter_data.png)
#
# _tmp_result_img.png -------+
#                        (multiply)-> milkyway.png
# (4)(5)-> filter_data.png --+
#

# (4) section-0 : Winter part

# filtering depth
# (max:1.0 disabled:0.0)
set section0_depth 0.3

# start posision of filtering
set section0_x_begin 632

# width of first gradient
set section0_width_grad0 340
# width of constant area
set section0_width_const 20
# width of last gradient
set section0_width_grad1 140

# (5) section-1 : Summer-Autumn part

set section1_depth 0.0
set section1_x_begin 235
set section1_width_grad0 110
set section1_width_const 90
set section1_width_grad1 60


####  P R O C E D U R E S  ####

proc save_my_image {} {

  global orig_file_name
  global level_bright_r
  global level_bright_g
  global level_bright_b
  global level_dark
  global level_all

  # dump .c0
  image create photo src -data .c0
  image create photo dest
  dest copy src -zoom 2
  dest write filter_data.png -format png

  # Create color bright part
  if { [catch {exec -ignorestderr magick $orig_file_name -channel R +level $level_bright_r -channel G +level $level_bright_g -channel B +level $level_bright_b _tmp_color_img.tiff} result] } {
    tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot exec magick!!"
  }

  # Create gray dark part
  if { [catch {exec -ignorestderr magick $orig_file_name -modulate 100,1 +level $level_dark _tmp_mono_img.tiff} result] } {
    tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot exec magick!!"
  }

  # Lighten Composition
  if { [catch {exec -ignorestderr magick composite -compose lighten _tmp_mono_img.tiff _tmp_color_img.tiff _tmp_composite_img.tiff} result] } {
    tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot exec magick!!"
  }

  # Adjust composite image
  if { [catch {exec -ignorestderr magick _tmp_composite_img.tiff +level $level_all _tmp_result_img.png} result] } {
    tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot exec magick!!"
  }

  # Composite using "multiply" mode
  if { [catch {exec -ignorestderr magick composite -compose multiply _tmp_result_img.png filter_data.png milkyway.png} result] } {
    tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot exec magick!!"
  } else {
    tk_messageBox -type ok -icon info -title "OK" -message "Processed $orig_file_name and saved milkyway.png"
  }

  #set im [image create photo -data .c0]
  #$im write filter_data.png -format png

  #image create photo .photo
  #.photo configure -format window -data $c
  #.photo write filter_data.png -format gif

  exit

}


####  M A I N  ####

wm title . "ESC: exit  Enter: Save and Exit"

# EBhD̐ݒ
wm resizable . 0 0

#wm overrideredirect . 1
#wm attributes . -topmost yes
wm geometry . ${width}x${height}
#wm minsize . $width $height
#wm maxsize . $width $height
#wm attributes . -alpha "1"
#wm attributes . -transparentcolor black

option add *background "#ffffff"
option add *foreground "#000000"
option add *highlightThickness "0"
option add *borderWidth "0"

canvas .c0 -width $width -height $height
pack .c0

#bind  . <Key-space> "wm overrideredirect . 0"
#bind  . <Return> "wm overrideredirect . 1"
bind  . <Escape> "exit"
bind  . <KeyPress-q> "exit"
#bind  . <KeyPress-s> "save_my_image"
bind  . <Return> "save_my_image"

#

set x 0
set rgb "#ffffff"

#
# section-0
#

# start pos
set x0 $section0_x_begin
# width of first gradient
set i_len $section0_width_grad0
for {set i 0} {$i < $i_len} {incr i} {
  set x [expr ($x0 + $i) % $width]
  set id [.c0 create line $x 0 $x $height]
  set v [expr round(255.0 - 255.0 * $section0_depth * (1.0 * $i / $i_len))]
  set rgb [format "#%02x%02x%02x" $v $v $v]
  .c0 itemconfigure $id -fill $rgb
}

set x0 $x
# width of constant area
set i_len $section0_width_const
for {set i 0} {$i < $i_len} {incr i} {
  set x [expr ($x0 + $i) % $width]
  set id [.c0 create line $x 0 $x $height]
  .c0 itemconfigure $id -fill $rgb
}

set x0 $x
# width of last gradient
set i_len $section0_width_grad1
for {set i 0} {$i < $i_len} {incr i} {
  set x [expr ($x0 + $i) % $width]
  set id [.c0 create line $x 0 $x $height]
  set v [expr round(255.0 - 255.0 * $section0_depth * (1.0 * ($i_len - 1 - $i) / $i_len))]
  set rgb [format "#%02x%02x%02x" $v $v $v]
  .c0 itemconfigure $id -fill $rgb
}

#
# section-1
#

# start pos
set x0 $section1_x_begin
# width of first gradient
set i_len $section1_width_grad0
for {set i 0} {$i < $i_len} {incr i} {
  set x [expr ($x0 + $i) % $width]
  set id [.c0 create line $x 0 $x $height]
  set v [expr round(255.0 - 255.0 * $section1_depth * (1.0 * $i / $i_len))]
  set rgb [format "#%02x%02x%02x" $v $v $v]
  .c0 itemconfigure $id -fill $rgb
}

set x0 $x
# width of constant area
set i_len $section1_width_const
for {set i 0} {$i < $i_len} {incr i} {
  set x [expr ($x0 + $i) % $width]
  set id [.c0 create line $x 0 $x $height]
  .c0 itemconfigure $id -fill $rgb
}

set x0 $x
# width of last gradient
set i_len $section1_width_grad1
for {set i 0} {$i < $i_len} {incr i} {
  set x [expr ($x0 + $i) % $width]
  set id [.c0 create line $x 0 $x $height]
  set v [expr round(255.0 - 255.0 * $section1_depth * (1.0 * ($i_len - 1 - $i) / $i_len))]
  set rgb [format "#%02x%02x%02x" $v $v $v]
  .c0 itemconfigure $id -fill $rgb
}


update

