#########################################################################
##                                                                     ##
##  I N T E R S T E L L A R   C O N S O L E  Main GUI                  ##
##            Copyright (C) 2025- Misato Observatory, Wakayama, Japan  ##
##                                                                     ##
##  This software is available under the GPL (General Public License)  ##
##                                                                     ##
##       Tcl/Tk References: http://www.nct9.ne.jp/m_hiroi/tcl_tk_doc/  ##
##                                                                     ##
#########################################################################


####################################################################
##                                                                ##
##                   L O C A L  S E T T I N G S                   ##
##                                                                ##
####################################################################

#
# Mode
#
set flag_initialize 0
set flag_dome_mode "-"
for {set i 0} {$i < $argc} {incr i} {
  if { [lindex $argv $i] eq "-dome" } {
    set flag_dome_mode 1
  } elseif { [string range [lindex $argv $i] 0 4] eq "-navi" } {
    set flag_dome_mode 0
  } elseif { [string range [lindex $argv $i] 0 4] eq "-init" } {
    set flag_initialize 1
  }
}

option add *font {"MS Gothic" 12}
option add *background "#000000"
option add *foreground "#ff5050"

if { $flag_dome_mode eq "-" } {
  set flag_dome_mode [tk_dialog .dlg_mode "Selection" "샂[hIĂ" "" 0 "Navigation" "Dome" "Cancel"]
}
if { $flag_dome_mode != 0 && $flag_dome_mode != 1 } {
  exit
}

#
# T[oURL
#
source local-config.tcl

#
# ݒł̌QON/OFF
#
set flag_light_pollution_after_initialization 1

#
#  tHg̐ݒ
#
if { $flag_dome_mode == 0 } {
  # Navigation Mode
  # ftHg
  option add *font {"MS Gothic" 12}
  set font_small {"MS Gothic" 10}
  set font_filename {"MS Gothic" 10}
  # xt[
  set font_labelframe {"MS Gothic" 10}
  # R[htB[h
  set font_codefield {"MS Gothic" 10}
} else {
  # Dome Mode
  # ftHg
  option add *font {"MS Gothic" 20}
  set font_small {"MS Gothic" 16}
  set font_filename {"MS Gothic" 14}
  # xt[
  set font_labelframe {"MS Gothic" 12}
  # R[htB[h
  set font_codefield {"MS Gothic" 14}
}

#
# eEBWFbg̃J[ݒ
#
option add *background "#000000"
option add *foreground "#ff5050"
set label_inverted_background "#d05050"
set label_inverted_foreground "#000000"
option add *activeBackground "#d08000"
option add *Button.background "#383838"
option add *Button.foreground "#ff6060"
option add *Button.padX "0"
option add *Button.padY "0"
option add *Entry.background "#303030"
option add *Entry.foreground "#ffe000"
option add *Entry.insertForeground "#000000"
option add *Entry.insertBackground "#00ff00"
option add *Entry.selectForeground "#ffffc0"
option add *Entry.selectBackground "#a01010"
option add *Text.foreground "#ff7030"
option add *Text.selectForeground "#ffc0c0"
option add *Text.selectBackground "#a01010"
option add *Dialog.msg.wrapLength "300p"
set tab_background "#383838"
set tab_foreground "#ff6060"
set tab_selected_background "#d08000"
set tab_selected_foreground "#000000"

# See https://wiki.tcl-lang.org/page/Changing+Widget+Colors
ttk::style theme use default
ttk::style configure TCombobox -fieldbackground "#303030" -foreground "#ff5050" -background "#505050" -arrowcolor "#ffe000" -selectforeground "#ff5050" -selectbackground "#000000"
ttk::style map TCombobox -fieldbackground [list disabled "#101010" readonly "#101010"]
option add *TCombobox*Listbox.selectBackground "#ff5050"
option add *TCombobox*Listbox.selectForeground "#000000"
option add *TCombobox*Listbox.font $font_small

#
# Do not change: Set stellarium_script_dir from localConfig.ssc
#
if { [catch {open "localConfig.ssc" r} fid] } {
  tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot open localConfig.ssc."
  exit
} else {
  set stellarium_script_dir "interstellar-console"
  while { [gets $fid line] >= 0 } {
    set status [regexp -- {^[ ]*var[ ]+([A-Za-z0-9_]+)[ ]*[=][ ]*["]([^"]+)["]} $line match s0 s1]
    if { $status == 1 && $s0 eq "stellarium_script_dir" } {
      set stellarium_script_dir $s1
    }
  }
  close $fid
}

#
# T[oURL (1)  T[oɕۑĂXNvg(staticȃR[h)
#                s鎞ɗp
# 
set stellarium_run_url "$stellarium_url/api/scripts/run"

#
# T[oURL (2)  NCAgŐXNvg(dynamicȃR[h)
#                𑗐M鎞ɗp
# 
#   ӁFdirect gꍇAsł̂͂Pŝ݂炵B
#         s̃R[h𑗐MƁAStellariumNbV
set stellarium_direct_url "$stellarium_url/api/scripts/direct"

#
# T[oURL (3)  擾p
# 
set stellarium_status_url "$stellarium_url/api/main/status"


####################################################################
##                                                                ##
##                           S P L A S H                          ##
##                                                                ##
####################################################################

set ret [exec -ignorestderr wish splash.tcl &]


####################################################################
##                                                                ##
##                       P R O C E D U R E S                      ##
##                                                                ##
####################################################################

#
# ėpFT[o̎w肳ꂽXNvg(foo.ssc)s (ψ)
#
proc exec_ssc_on_server { args } {

  global stellarium_run_url stellarium_script_dir
  global buffer_codefield

  foreach script_file $args {

    set buffer_codefield [string trim $script_file]
    set ret [exec -ignorestderr curl -s -d id=$stellarium_script_dir/$script_file $stellarium_run_url]
    if { $ret ne "ok" } {
      #tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR in $script_file: $ret"
      set buffer_codefield "$script_file / ERROR: $ret"
    }

  }

}

#
# ėpFstellarium XNvgns 𑗐M (ψ)
#
#  yӁz
#    Eargs[n] ɋ󔒂ꂽꍇ [~] A["] ꂽꍇ ['] ƂB
#      F "Ursa Major"  'Ursa~Major' Ƃ
#    EMł̂́A֐̗񂾂łB
#      StellariumNbV邽߁Afor()Ȃǂ̐\𑗂Ă͂ȂȂB
#
proc send_ssc_lines { args } {

  global stellarium_direct_url
  global buffer_codefield
  # Set debug=1 to write code to $code_file
  set debug 0
  set code_file "tmp.txt"

  foreach code1line $args {

    # replace [~] with [ ], and ['] with ["]
    set code_p0 [regsub -all {~} $code1line { }]
    set code_p1 [regsub -all {'} $code_p0 {"}]
    set buffer_codefield [string trim $code_p1]

    if { $debug == 0 } {
      set f [open "| curl -s -d @- $stellarium_direct_url" w]
        puts -nonewline $f "code="
        puts -nonewline $f "$buffer_codefield"
        puts $f ";"
      catch {close $f} std_err
    } else {
      set f [open $code_file w]
        puts -nonewline $f "code="
        puts -nonewline $f "$buffer_codefield"
        puts $f ";"
      catch {close $f} std_err
      set ret [exec -ignorestderr curl -s -d @$code_file $stellarium_direct_url]
    }

  }

}

#
# ėpFstellarium XNvgns 𑗐M
#         ܂
#         T[o̎w肳ꂽXNvg(foo.ssc)s (ψ)
#
proc send_or_exec_ssc { args } {

  foreach code $args {

    set code_case [string tolower $code]
    if { [string match "*.ssc" $code_case] } {
      exec_ssc_on_server $code
    } else {
      send_ssc_lines $code
    }

  }

}

#
# P̐̃ZNg/AZNg
#
#  yӁzname ɋ󔒂ꂽꍇ ~ Ƃ邱
#          F "Ursa Major"  "Ursa~Major" Ƃ
#
proc update_constellation { name flg } {

  global stellarium_direct_url
  global buffer_codefield
  # Set debug=1 to write code to $code_file
  set debug 0
  set code_file "tmp.txt"

  # replace '~' with ' '
  set arg_name [regsub -all {~} $name { }]

  if { $flg == "true" } {
    set buffer_codefield "ConstellationMgr.selectConstellation(\"$arg_name\")"
  } else {
    set buffer_codefield "ConstellationMgr.deselectConstellation(\"$arg_name\")"
  }

  if { $debug == 0 } {
    set f [open "| curl -s -d @- $stellarium_direct_url" w]
      puts -nonewline $f "code="
      puts -nonewline $f "$buffer_codefield"
      puts $f ";"
    catch {close $f} std_err
  } else {
    set f [open $code_file w]
      puts -nonewline $f "code="
      puts -nonewline $f "$buffer_codefield"
      puts $f ";"
    catch {close $f} std_err
    set ret [exec -ignorestderr curl -s -d @$code_file $stellarium_direct_url]
  }

}

#
# P̐xON/Off
#
proc set_star_label { args } {

  global asarr_star_label

  if { [array size asarr_star_label] == 0 } {
    initialize_star_labels
  }

  foreach name $args {
    send_ssc_lines "LabelMgr.setLabelShow($asarr_star_label($name),true)"
  }

}

proc unset_star_label { args } {

  global asarr_star_label

  if { [array size asarr_star_label] == 0 } {
    initialize_star_labels
  }

  foreach name $args {
    send_ssc_lines "LabelMgr.setLabelShow($asarr_star_label($name),false)"
  }

}

#
# ėpFfilename_text ǂݍ݁A1s=1vf̔z array_ret ɕԂ
#
proc get_array_from_text_lines { filename_text array_ret } {

  upvar $array_ret array_ref

  array set array_ref {}

  if { [catch {open $filename_text r} fid] } {
    # do nothing
  } else {
    set i 0
    while { [gets $fid line] >= 0 } {
      set s [string trim $line]
      # skip comments
      if { [string length $s] != 0 && [string index $s 0] ne "#" } {
        set array_ref($i) $s
        incr i
      }
    }
    close $fid
  }

}

#                                        #
# proc͓̃EBWFbgp #
#                                        #

#                                                        "spring" "p1"
proc build_dso_buttons { filename_text proc_name len_col sufx0    lf_sufx } {

  global font_small
  array set lines {}

  get_array_from_text_lines $filename_text lines

  set len_lines [array size lines]

  #set len_col 3
  set i 0
  set j 0
  for {set i 0} {$i < $len_lines} {incr i} {
    if { [expr $i % $len_col] == 0 } {
      frame .fr_dso_l$j$sufx0
    }
    set lst [split $lines($i) ","]
    if { 1 < [llength $lst] } {
      set s0 [string trim [lindex $lst 0]]
      set s1 [string trim [lindex $lst 1]]
      set s1 [regsub -all { } $s1 {~}]
      button .b_dso_$i$sufx0 -text "$s0" -font $font_small -command "$proc_name $s1"
      pack .b_dso_$i$sufx0 -side left -expand 1 -fill x -in .fr_dso_l$j$sufx0
    } else {
      if { $lines($i) ne "-" } {
        set s0 [string trim $lines($i)]
        set s1 [regsub -all { } $s0 {~}]
        button .b_dso_$i$sufx0 -text "$s0" -font $font_small -command "$proc_name $s1"
        pack .b_dso_$i$sufx0 -side left -expand 1 -fill x -in .fr_dso_l$j$sufx0
      }
    }
    if { [expr ($i + 1) % $len_col] == 0 } {
      pack .fr_dso_l$j$sufx0 -expand 0 -fill x -in .lf_dso_$lf_sufx
      incr j
    }
  }
  if { [expr $i % $len_col] != 0 } {
    pack .fr_dso_l$j$sufx0 -expand 0 -fill x -in .lf_dso_$lf_sufx
  }

}

# Info Server lGETAAzz(ėp)
# in: filename  out: asarr_ret
proc get_data_from_iscserv { filename asarr_ret } {

  global iscserv_url
  upvar $asarr_ret asarr_ref

  if { [catch {open "| curl -s $iscserv_url/$filename" r} fh] } {
    tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET $iscserv_url/$filename"
  } else {
    set flg_get 0
    while { [gets $fh txt_line] >= 0 } {
      set lst [split [string trim $txt_line] ","]
      if { [llength $lst] == 2 } {
        set k [string trim [lindex $lst 0]]
        set v [string trim [lindex $lst 1]]
        set asarr_ref($k) $v
        set flg_get 1
      }
    }
    catch {close $fh} std_err
    if { $flg_get == 0 } {
      tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET $iscserv_url/$filename"
    }
  }

}

# Info Server 琯AV̐쓙̒lGETAAzz
proc get_current_settings {} {

  global asarr_initial_settings

  # asarr_initial_settings is updated
  get_data_from_iscserv "_tmp_currentSettings.isc.txt" asarr_initial_settings

}

# Info Server 烉xIDGETAAzz
proc get_star_label_id {} {

  global asarr_star_label

  # asarr_star_label is updated
  get_data_from_iscserv "_tmp_starLabels.isc.txt" asarr_star_label

}

# x̏ (T[őNK{)
proc initialize_star_labels {} {

  send_ssc_lines "LabelMgr.deleteAllLabels()"
  exec_ssc_on_server createStarLabels.ssc
  after 500
  get_star_label_id

}

# GUI  (Configuration/Maintenance)
proc initialize_config_gui_params {} {

  send_ssc_of_projection_mode /
  send_ssc_of_display_gamma /
  send_ssc_of_view_fov /
  send_ssc_of_view_altitude /
  send_ssc_of_view_azimuth /
  send_ssc_of_viewport_x /
  send_ssc_of_viewport_y /
  send_ssc_of_disk_viewport /
  send_ssc_of_dso_fov_circular_marker /

}

# GUI
proc initialize_all_gui_params {} {

  send_ssc_of_timerate /
  send_ssc_of_milkyway /
  send_ssc_of_relative_star_scale /
  send_ssc_of_absolute_star_scale /
  send_ssc_of_landscape_transparency /
  send_ssc_of_meteor_zhr /
  send_ssc_of_min_fps /
  send_ssc_of_location_latitude /
  send_ssc_of_stellarlabels /
  send_ssc_of_solsyslabels /
  send_ssc_of_dsolabels /
  send_ssc_of_dsohints /

  initialize_config_gui_params

}

# r[
proc send_ssc_to_initialize_view {} {

  exec_ssc_on_server localInitView.ssc

  initialize_config_gui_params

}

# S
proc send_ssc_to_initialize_all {} {

  global flag_dome_mode
  global flag_light_pollution_after_initialization
  global asarr_initial_settings

  # To stop Scheduling in date-gui.tcl
  send_ssc_lines core.setTimeRate(0.0)

  # set mode
  if { $flag_dome_mode == 0 } {
    exec_ssc_on_server unsetFlagDomeMode.ssc
  } else {
    exec_ssc_on_server setFlagDomeMode.ssc
  }

  # Mask stars ...
  send_ssc_lines MilkyWay.setIntensity(0.1) StelSkyDrawer.setFlagStarMagnitudeLimit(true) StelSkyDrawer.setCustomStarMagnitudeLimit(1.0) StelSkyDrawer.setAbsoluteStarScale(0.1)

  # initialize and create lables
  initialize_star_labels

  # initialize location
  after 100
  exec_ssc_on_server localInitLocation.ssc

  # Mask stars ...
  send_ssc_lines StelSkyDrawer.setLightPollutionLuminance(0.01)

  after 100
  exec_ssc_on_server localInitView.ssc

  after 100
  exec_ssc_on_server localInitSettings.ssc

  after 200
  if { $flag_light_pollution_after_initialization != 0 } {
    exec_ssc_on_server setLightPollution.ssc
  } else {
    exec_ssc_on_server unsetLightPollution.ssc
  }

  # Recovering from view initialization failures
  after 500

  while { 1 } {

    exec_ssc_on_server localInitView.ssc

    # ̐ݒl擾
    after 200
    exec_ssc_on_server getCurrentSettings.ssc
    after 200
    get_current_settings

    set flg_found 0
    foreach {i} [array name asarr_initial_settings] {
      set v $asarr_initial_settings($i)
      #debug
      #if { $i eq "TestKey" } {
      #  tk_dialog .dlg_test "TEST" "i,v = ($i),($v)" "" 0 "Yes"
      #}
      if { [string tolower $v] eq "nan" } {
        set flg_found 1
      }
    }
    if { $flg_found == 0 } {
      break
    } else {
      set flg_retly [tk_dialog .dlg_retly_init "Question" "View̏Ɏs悤łBĎs܂H" "" 0 "Yes" "No"]
      if { $flg_retly != 0 } {
        break
      }
    }
  }

  initialize_all_gui_params

}

# SSCMR[h̓tB[hp
proc send_ssc_from_codefield {} {

  global buffer_codefield

  set buffer_codefield [string trim $buffer_codefield]

  if { ![string match "http*:*" $buffer_codefield] } {
    send_or_exec_ssc "$buffer_codefield"
  }

}

proc set_timerate { prm } {

  global buffer_timerate

  set buffer_timerate [format "%.1f" $prm]

  send_ssc_lines "core.setTimeRate($buffer_timerate)"

}

proc send_ssc_of_timerate { args } {

  global buffer_timerate

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 0.0
    } else {
      set v [expr $buffer_timerate + $prm]
    }
    set buffer_timerate $v
  }

  set v [string trim $buffer_timerate]
  set buffer_timerate [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "core.setTimeRate($buffer_timerate)"
  }

}

proc send_ssc_from_timearg { args } {

  global buffer_datearg

  set tm_strreal [clock format [clock seconds] -format {%Y-%m-%dT%H:%M:%S}]
  set tm_str0 [clock format [clock seconds] -format {%Y-%m-%dT}]

  set buffer_datearg $tm_strreal

  foreach prm $args {
    set buffer_datearg $tm_str0
    append buffer_datearg $prm
  }

  send_ssc_lines "core.setDate('$buffer_datearg','local')"

}

proc set_buffer_codefield_from_buffer_datearg {} {

  global buffer_codefield
  global buffer_datearg

  set buffer_codefield "core.setDate(\"$buffer_datearg\",\"local\")"

}

proc send_ssc_of_meteor_zhr { args } {

  global buffer_meteor_zhr

  set flg_show "true"
  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 0
      set flg_show "false"
    } else {
      set v [expr $buffer_meteor_zhr + $prm]
    }
    set buffer_meteor_zhr $v
  }

  set v [string trim $buffer_meteor_zhr]
  if { $v < 0.0 } {
    set v 0
  }
  set buffer_meteor_zhr [format "%.0f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "SporadicMeteorMgr.setFlagShow($flg_show)"
    send_ssc_lines "SporadicMeteorMgr.setZHR($buffer_meteor_zhr)"
  }

}

proc send_ssc_of_stellarlabels { args } {

  global buffer_stellarlabels

  set flg_labels "true"
  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 0.0
      set flg_labels "false"
    } else {
      set v [expr $buffer_stellarlabels + $prm]
    }
    set buffer_stellarlabels $v
  }

  set v [string trim $buffer_stellarlabels]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_stellarlabels [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StarMgr.setFlagLabels($flg_labels)"
    send_ssc_lines "StarMgr.setLabelsAmount($buffer_stellarlabels)"
  }

}

proc unset_solarsystem { args } {

  unset_star_label Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto

  set flg 0
  foreach prm $args {
    exec_ssc_on_server unsetSolarSystemLightPollution.ssc
    set flg 1
  }

  if { $flg == 0 } {
    exec_ssc_on_server unsetSolarSystem.ssc updateLightMoonPollution.ssc
  }

}

proc send_ssc_of_solsyslabels { args } {

  global buffer_solsyslabels

  set flg_labels "true"
  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 0.0
      set flg_labels "false"
    } else {
      set v [expr $buffer_solsyslabels + $prm]
    }
    set buffer_solsyslabels $v
  }

  set v [string trim $buffer_solsyslabels]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_solsyslabels [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "SolarSystem.setFlagLabels($flg_labels)"
    send_ssc_lines "SolarSystem.setLabelsAmount($buffer_solsyslabels)"
  }

}

proc send_ssc_of_milkyway { args } {

  global asarr_initial_settings
  global buffer_milkyway

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(MilkyWay.intensity)
    } else {
      set v [expr $buffer_milkyway + $prm]
    }
    set buffer_milkyway $v
  }

  set v [string trim $buffer_milkyway]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_milkyway [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "MilkyWay.setIntensity($buffer_milkyway)"
  }

}

proc send_ssc_of_relative_star_scale { args } {

  global asarr_initial_settings
  global buffer_relative_star_scale

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(StelSkyDrawer.relativeStarScale)
    } else {
      set v [expr $buffer_relative_star_scale + $prm]
    }
    set buffer_relative_star_scale $v
  }

  set v [string trim $buffer_relative_star_scale]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_relative_star_scale [format "%.2f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelSkyDrawer.setRelativeStarScale($buffer_relative_star_scale)"
  }

}

proc send_ssc_of_absolute_star_scale { args } {

  global asarr_initial_settings
  global buffer_absolute_star_scale

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(StelSkyDrawer.absoluteStarScale)
    } else {
      set v [expr $buffer_absolute_star_scale + $prm]
    }
    set buffer_absolute_star_scale $v
  }

  set v [string trim $buffer_absolute_star_scale]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_absolute_star_scale [format "%.2f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelSkyDrawer.setAbsoluteStarScale($buffer_absolute_star_scale)"
  }

}

proc send_ssc_of_landscape_transparency { args } {

  global asarr_initial_settings
  global buffer_landscape_transparency

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(LandscapeMgr.landscapeTransparency)
    } else {
      set v [expr $buffer_landscape_transparency + $prm]
    }
    set buffer_landscape_transparency $v
  }

  set v [string trim $buffer_landscape_transparency]
  if { $v < 0.0 } {
    set v 0.0
  } elseif { 1.0 < $v } {
    set v 1.0
  }
  set buffer_landscape_transparency [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "LandscapeMgr.setLandscapeTransparency($buffer_landscape_transparency)"
  }

}

proc update_azchange_duration { args } {

  global buffer_azchange_duration

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 5
    } else {
      set v [expr $buffer_azchange_duration + $prm]
    }
    set buffer_azchange_duration $v
  }

  set v [string trim $buffer_azchange_duration]
  if { 99 < $v } {
    set v 99
  } elseif { $v < 0 } {
    set v 0
  }
  set buffer_azchange_duration [format "%d" $v]

}

proc send_ssc_of_location_latitude { args } {

  global asarr_initial_settings
  global buffer_location_latitude

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(Latitude)
    } else {
      set v [expr $buffer_location_latitude + $prm]
    }
    set buffer_location_latitude $v
  }

  set v [string trim $buffer_location_latitude]
  if { 90.0 < $v } {
    set v 90.0
  } elseif { $v < -90.0 } {
    set v -90.0
  }
  set buffer_location_latitude [format "%.8f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "core.setObserverLocation($asarr_initial_settings(Longitude),$buffer_location_latitude,$asarr_initial_settings(Altitude),0.0,'Our Observatory','Earth')"
  }

}

proc send_ssc_of_min_fps { args } {

  global asarr_initial_settings
  global buffer_min_fps

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(MinFps)
    } else {
      if { 0 < $prm } {
        if { $buffer_min_fps < 2 } {
          set v [expr $buffer_min_fps + $prm]
        } else {
          set v [expr $buffer_min_fps + $prm + $prm]
        }
      } else {
        if { $buffer_min_fps < 3 } {
          set v [expr $buffer_min_fps + $prm]
        } else {
          set v [expr $buffer_min_fps + $prm + $prm]
        }
      }
    }
    set buffer_min_fps $v
  }

  set v [string trim $buffer_min_fps]
  if { $v < 1.0 } {
    set v 1.0
  }
  set buffer_min_fps [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "core.setMinFps($buffer_min_fps)"
  }

}

proc send_ssc_of_dsolabels { args } {

  global buffer_dsolabels

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 0.0
    } else {
      set v [expr $buffer_dsolabels + $prm]
    }
    set buffer_dsolabels $v
  }

  set v [string trim $buffer_dsolabels]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_dsolabels [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "NebulaMgr.setLabelsAmount($buffer_dsolabels)"
  }

}

proc send_ssc_of_dsohints { args } {

  global buffer_dsohints

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 0.0
    } else {
      set v [expr $buffer_dsohints + $prm]
    }
    set buffer_dsohints $v
  }

  set v [string trim $buffer_dsohints]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_dsohints [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "NebulaMgr.setHintsAmount($buffer_dsohints)"
  }

}

proc select_a_dso { args } {

  global buffer_selected_dsoname

  foreach name $args {
    set buffer_selected_dsoname [regsub -all {~} $name { }]
  }

  set buffer_selected_dsoname [string trim $buffer_selected_dsoname]
  send_ssc_lines core.selectObjectByName('$buffer_selected_dsoname',true)

  after 500 get_object_data_from_stellarium

}

proc send_ssc_of_dso_fov_circular_marker { args } {

  global buffer_dso_fov_circular_size

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 1.0
    } else {
      set v $prm
    }
    set buffer_dso_fov_circular_size $v
  }

  set v [string trim $buffer_dso_fov_circular_size]
  if { $v < 0.0 } {
    set v 0.0001
  }
  set buffer_dso_fov_circular_size [format "%.4f" $v]

  if { $prm ne "/" } {
    send_ssc_lines SpecialMarkersMgr.setFOVCircularMarkerSize($buffer_dso_fov_circular_size) SpecialMarkersMgr.setFlagFOVCircularMarker(true)
  }

}

proc get_object_data_from_stellarium {} {

  global stellarium_status_url

  # Set nkf's output option
  set v [info tclversion]
  if { 9.0 <= $v } {
    # output UTF-8
    set nkf_o "-w"
  } else {
    # output Shift-JIS
    set nkf_o "-s"
  }

  if { [catch {set line [exec -ignorestderr curl -s -G $stellarium_status_url | ./nkf -W $nkf_o]} result] } {
    set buffer_codefield "ERROR in get_object_data_from_stellarium: Cannot connect to Stellarium."
  } else {
    # Replace: \" -> \a
    set line [regsub -all {[\\]"} $line "\a"]
    set status [regexp -- {.*["]selectioninfo["]:["]([^"]+)["]} $line match info_str]
    if { $status == 1 } {
      # Restore: \a -> \"
      set info_str [string map {"\a" "\""} $info_str]
      set hr_str "-------------------------------------------------------------------"
      set info_str [regsub -all {<br[ ]*[/]*>} $info_str "\n"]
      set info_str [regsub -all {<h[0-9]>} $info_str "$hr_str\n"]
      set info_str [regsub -all {</h[0-9]>} $info_str "\n$hr_str\n"]
      set info_str [regsub -all {</sup>|</sub>|<[/]*b>|<[/]*strong>} $info_str ""]
      set info_str [string map {"<sup>" "^" "<sub>" "_" "&alpha;" "" "&delta;" "" "&phi;" ""} $info_str]
      # delete all in text area
      .t_objinfo delete 1.0 end
      .t_objinfo insert end "$info_str"
    }
  }

}

proc exec_date_gui {} {

  global stellarium_status_url
  global buffer_codefield
  global flag_dome_mode

  if { [catch {exec -ignorestderr curl -s -G $stellarium_status_url} result] } {
    set buffer_codefield "ERROR in exec_date_gui: Cannot connect to Stellarium."
  } else {
    if { $flag_dome_mode == 0 } {
      set ret [exec -ignorestderr wish date-gui.tcl &]
    } else {
      set ret [exec -ignorestderr wish date-gui.tcl -dome &]
    }
  }

}

proc exec_media_gui {} {

  global stellarium_status_url
  global buffer_codefield
  global flag_dome_mode

  if { [catch {exec -ignorestderr curl -s -G $stellarium_status_url} result] } {
    set buffer_codefield "ERROR in exec_date_gui: Cannot connect to Stellarium."
  } else {
    if { $flag_dome_mode == 0 } {
      set ret [exec -ignorestderr wish media-gui.tcl &]
    } else {
      set ret [exec -ignorestderr wish media-gui.tcl -dome &]
    }
  }

}

proc exec_date_monitor {} {

  global stellarium_status_url
  global buffer_codefield

  if { [catch {exec -ignorestderr curl -s -G $stellarium_status_url} result] } {
    set buffer_codefield "ERROR in exec_date_monitor: Cannot connect to Stellarium."
  } else {
    set ret [exec -ignorestderr exec_date-monitor.bat screen &]
  }

}

proc kill_date_monitor {} {

  set ret [exec -ignorestderr exec_date-monitor.bat kill &]

}

proc change_date_monitor {} {

  set ret [exec -ignorestderr exec_date-monitor.bat change &]

}

proc exit_or_reboot {} {

  global flag_dome_mode
  global buffer_timerate
  global buffer_min_fps

  set buffer_timerate 0.0
  send_ssc_lines "core.setTimeRate(0.0)"
  send_ssc_lines "core.setMinFps(0.2)"

  set flag [tk_dialog .dlg_mode "Standby mode: Exit?" "݁Aȓd͑ҋ@[hł\n\nȉIĂF" "" 0 "SI" "GUIċN" "ʏ탂[h֖߂"]

  send_ssc_lines "core.setMinFps($buffer_min_fps)"

  if { $flag == 0 } {
    exec isc_startup.bat -kill &
  } elseif { $flag == 1 } {
    if { $flag_dome_mode == 0 } {
      exec isc_startup.bat -gui-only -navi &
    } else {
      exec isc_startup.bat -gui-only -dome &
    }
  }

}

#
# For configuration/maintenance
#

proc send_ssc_of_projection_mode { args } {

  global asarr_initial_settings
  global buffer_projection_mode

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ProjectionMode)
    } else {
      set v $prm
    }
    set buffer_projection_mode $v
  }

  set v [string trim $buffer_projection_mode]
  set buffer_projection_mode $v

  if { $prm ne "/" } {
    send_ssc_lines "core.setProjectionMode('$buffer_projection_mode')"
  }

}

proc send_ssc_of_display_gamma { args } {

  global asarr_initial_settings
  global buffer_display_gamma

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(DisplayGamma)
    } else {
      set v $prm
    }
    set buffer_display_gamma $v
  }

  set v [string trim $buffer_display_gamma]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_display_gamma [format "%.2f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "core.setDisplayGamma($buffer_display_gamma)"
  }

}

proc send_ssc_of_view_fov { args } {

  global asarr_initial_settings
  global buffer_view_fov

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewFov)
    } else {
      set v $prm
    }
    set buffer_view_fov $v
  }

  set v [string trim $buffer_view_fov]
  if { $v < 0.1 } {
    set v 0.1
  }
  set buffer_view_fov [format "%.2f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelMovementMgr.zoomTo($buffer_view_fov,0.0)"
  }

}

proc send_ssc_of_view_altitude { args } {

  global asarr_initial_settings
  global buffer_view_altitude
  global buffer_view_azimuth

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewAltitude)
    } else {
      set v $prm
    }
    set buffer_view_altitude $v
  }

  set v [string trim $buffer_view_altitude]
  if { $v < 0.0 } {
    set v 0.0
  }
  while { 90.0 < $v } {
    set v 90.0
  }
  set buffer_view_altitude [format "%.3f" $v]

  if { $prm ne "/" } {
    if { 89.99999 < $buffer_view_altitude } {
      send_ssc_lines "core.moveToAltAzi('89.99999','$buffer_view_azimuth',0.0)"
    } else {
      send_ssc_lines "core.moveToAltAzi('$buffer_view_altitude','$buffer_view_azimuth',0.0)"
    }
  }

}

proc send_ssc_of_view_azimuth { args } {

  global asarr_initial_settings
  global buffer_view_altitude
  global buffer_view_azimuth

  global buffer_azchange_duration

  set prm ""
  set duration_tm 0.0
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewAzimuth)
    } else {
      set v [expr $asarr_initial_settings(ViewAzimuth) + $prm]
      set duration_tm $buffer_azchange_duration
    }
    set buffer_view_azimuth $v
  }

  set v [format "%.2f" [string trim $buffer_view_azimuth]]
  while { $v < 0.0 } {
    set v [format "%.2f" [expr $v + 360.0]]
  }
  while { 360.0 <= $v } {
    set v [format "%.2f" [expr $v - 360.0]]
  }
  set buffer_view_azimuth $v

  if { $prm ne "/" } {
    if { 89.99999 < $buffer_view_altitude } {
      send_ssc_lines "core.moveToAltAzi('89.99999','$buffer_view_azimuth',$duration_tm)"
    } else {
      send_ssc_lines "core.moveToAltAzi('$buffer_view_altitude','$buffer_view_azimuth',$duration_tm)"
    }
  }

}

proc send_ssc_of_viewport_x { args } {

  global asarr_initial_settings
  global buffer_viewport_x
  global buffer_viewport_y

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewportX)
    } else {
      set v $prm
    }
    set buffer_viewport_x $v
  }

  set v [string trim $buffer_viewport_x]
  set buffer_viewport_x [format "%.3f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelMovementMgr.moveViewport($buffer_viewport_x,$buffer_viewport_y,0.0)"
  }

}

proc send_ssc_of_viewport_y { args } {

  global asarr_initial_settings
  global buffer_viewport_x
  global buffer_viewport_y

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewportY)
    } else {
      set v $prm
    }
    set buffer_viewport_y $v
  }

  set v [string trim $buffer_viewport_y]
  set buffer_viewport_y [format "%.3f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelMovementMgr.moveViewport($buffer_viewport_x,$buffer_viewport_y,0.0)"
  }

}

proc send_ssc_of_disk_viewport { prm } {

  global asarr_initial_settings

  set v "false"
  if { $prm eq "-" || $prm eq "/" } {
    set v $asarr_initial_settings(DiskViewport)
  } else {
    set v $prm
  }

  if { $prm ne "/" } {
    send_ssc_lines "core.setDiskViewport($v)"
  }

}


#
# ^u̐ݒ
#

proc deselect_all {} {
  global tab_background
  global tab_foreground
  .pw forget .lf_general
  .pw forget .lf_constel
  .pw forget .lf_spring
  .pw forget .lf_summer
  .pw forget .lf_autumn
  .pw forget .lf_winter
  .pw forget .lf_spring_asterism_star
  .pw forget .lf_summer_asterism_star
  .pw forget .lf_autumn_asterism_star
  .pw forget .lf_winter_asterism_star
  .pw forget .lf_summer_star
  .pw forget .lf_autumn_star
  .pw forget .lf_winter_star
  .pw forget .lf_solarsystem
  .pw forget .lf_dso_p0
  .pw forget .lf_dso_p1
  .pw forget .lf_dso_p2
  .pw forget .lf_gridlines
  .pw forget .lf_objinfo
  .pw forget .lf_config
  for {set i 0} {$i < 8} {incr i} {
    .b_tab$i configure -bg $tab_background -fg $tab_foreground -relief raised
  }
}

proc select_general {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_solarsystem
  .pw add    .lf_gridlines
  .b_tab0 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
}

proc select_spring {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_spring
  .pw add    .lf_spring_asterism_star
  .b_tab1 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
}

proc select_summer {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_summer
  .pw add    .lf_summer_asterism_star
  .b_tab2 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
}

proc select_autumn {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_autumn
  .pw add    .lf_autumn_asterism_star
  .b_tab3 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
}

proc select_winter {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_winter
  .pw add    .lf_winter_asterism_star
  .b_tab4 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
}

proc select_dso {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  #.pw add    .lf_solarsystem
  .pw add    .lf_dso_p0
  #.pw add    .lf_gridlines
  .pw add    .lf_dso_p1
  .pw add    .lf_dso_p2
  .b_tab5 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
}

proc select_objinfo {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_dso_p0
  .pw add    .lf_objinfo
  .b_tab6 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
}

proc select_config {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_solarsystem
  .pw add    .lf_gridlines
  .pw add    .lf_config
  .b_tab7 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
}


####################################################################
##                                                                ##
##                             M A I N                            ##
##                                                                ##
####################################################################

# EBhD,̐ݒ
if { $flag_dome_mode == 0 } {
  wm title . "Intersteller Console on $os_name :: main (Navigation Mode)"
} else {
  wm title . "Intersteller Console on $os_name :: main (Dome Mode)"
}
wm resizable . 0 0

# EBWFbg̐ݒ
option add *Labelframe.padX "2"
option add *Labelframe.padY "2"

# pύXɂb̏l
set buffer_azchange_duration 5

# DSȌl
set buffer_selected_dsoname "M42"

# ŌGUIŐݒ肳ꂽtB̓tB[h[DATE]{^ŌĂяo
set buffer_datearg [clock format [clock seconds] -format {%Y-%m-%dT%H:%M:%S}]

# xpAzz
array set asarr_star_label {}

# AV̖̐邳 lۑp Azz
array set asarr_initial_settings {
  StelSkyDrawer.relativeStarScale 1.0
  StelSkyDrawer.absoluteStarScale 1.0
  MilkyWay.intensity 1.0
  ZodiacalLight.intensity 1.0
  LandscapeMgr.landscapeTransparency 0.0
  Altitude 0
  Longitude 135.18989563
  Latitude 34.19189835
  ProjectionMode ProjectionStereographic
  DisplayGamma 2.2
  ViewFov 190.0
  ViewAltitude 90.0
  ViewAzimuth 180.0
  ViewportX 0.0
  ViewportY 0.0
  DiskViewport false
  MinFps 18.0
}

########################################################################

frame .fr_rootwin

########################################################################

frame .fr_guitab

labelframe .lf_gui -text "GUI" -font $font_labelframe

button .b_exitall -text "I" -background "#500000" -font $font_small -command "exit_or_reboot"
button .b_dategui -text "" -command "exec_date_gui"
button .b_mediagui -text "Media" -command "exec_media_gui"
pack .b_exitall .b_dategui .b_mediagui -side left -expand 0 -fill y -in .lf_gui

labelframe .lf_selection -text "Selection" -font $font_labelframe

button .b_tab0 -text "" -command "select_general"
button .b_tab1 -text " t" -command "select_spring"
button .b_tab2 -text " " -command "select_summer"
button .b_tab3 -text " H" -command "select_autumn"
button .b_tab4 -text " ~" -command "select_winter"
button .b_tab5 -text "V" -command "select_dso"
button .b_tab6 -text "" -command "select_objinfo"
button .b_tab7 -text "ݒ" -font $font_small -command "select_config"

pack .b_tab0 .b_tab1 .b_tab2 .b_tab3 .b_tab4 .b_tab5 .b_tab6 -side left -expand 1 -fill x -in .lf_selection
pack .b_tab7 -side left -expand 0 -fill y -in .lf_selection

pack .lf_gui -side left -expand 0 -padx 3 -pady 2 -in .fr_guitab
pack .lf_selection -side left -expand 1 -fill x  -padx 3 -pady 2 -in .fr_guitab

pack .fr_guitab -expand 1 -fill x -padx 1 -pady 1 -in .fr_rootwin


################################################################

panedwindow .pw -sashwidth 4
pack .pw -expand 1 -fill both  -padx 3 -pady 2 -in .fr_rootwin

################################################################

labelframe .lf_general -text "General" -font $font_labelframe

frame .fr_init
label .l_init -text "ݒ"
button .b_init0 -text "20" -command "send_ssc_from_timearg 20:00:00"
button .b_init1 -text "ꏊ" -command "send_ssc_of_location_latitude -"
button .b_init2 -text "VIEW" -command "send_ssc_to_initialize_view"
# Note: Light Pollution settings are disabled by localInitLocation.ssc.
button .b_init3 -text "S" -command "send_ssc_to_initialize_all"
pack .l_init -in .fr_init
pack .b_init0 .b_init1 .b_init2 .b_init3 -side left -expand 1 -fill both -in .fr_init

frame .fr_tmrate
frame .fr_tmrate_l
label .l_tmrate -text "[g" -padx 0
label .l_tmrate_b -text "(^)" -font $font_small -padx 0
pack .l_tmrate .l_tmrate_b -side left -padx 0 -in .fr_tmrate_l
button .b_tmrate0 -text "0" -command "set_timerate 0.0"
button .b_tmrate1 -text "" -command "set_timerate 1.0"
entry .e_tmrate -width 5 -textvariable buffer_timerate
set buffer_timerate "0.0"
button .b_tmrate2 -text "<" -command "send_ssc_of_timerate -20.0"
button .b_tmrate3 -text ">" -command "send_ssc_of_timerate +20.0"
bind .e_tmrate <Return> "send_ssc_of_timerate"
pack .fr_tmrate_l -in .fr_tmrate
pack .b_tmrate0 .b_tmrate1 -side left -fill both -in .fr_tmrate
pack .e_tmrate -expand 1 -side left -fill both -in .fr_tmrate
pack .b_tmrate2 .b_tmrate3 -side left -fill both -in .fr_tmrate

frame .fr_illumi
label .l_illumi -text "znEQ"
button .b_illumi2 -text "ON" -command "exec_ssc_on_server setLightPollution.ssc"
button .b_illumi1 -text "Off" -command "exec_ssc_on_server unsetLightPollution.ssc"
button .b_illumi0 -text "znOff" -command "unset_solarsystem -"
pack .l_illumi -in .fr_illumi
pack .b_illumi0 .b_illumi1 .b_illumi2 -side left -expand 1 -fill both -in .fr_illumi

frame .fr_atmos
label .l_atmos -text "C" -font $font_small
button .b_atmos2 -text "FOG" -font $font_small -command "send_ssc_lines LandscapeMgr.setFlagFog(true) LandscapeMgr.setFlagAtmosphere(true)"
button .b_atmos1 -text "ON" -font $font_small -command "send_ssc_lines LandscapeMgr.setFlagFog(false) LandscapeMgr.setFlagAtmosphere(true)"
button .b_atmos0 -text "Off" -font $font_small -command "send_ssc_lines LandscapeMgr.setFlagFog(false) LandscapeMgr.setFlagAtmosphere(false)"
pack .l_atmos -in .fr_atmos
pack .b_atmos0 .b_atmos1 .b_atmos2 -side left -expand 1 -fill both -in .fr_atmos

set buffer_milkyway $asarr_initial_settings(MilkyWay.intensity)
frame .fr_milky
label .l_milky -text "V̐ " -font $font_small
button .b_milky0 -text "l" -font $font_small -command "send_ssc_of_milkyway -"
entry .e_milky -width 5 -textvariable buffer_milkyway -font $font_small
button .b_milky1 -text "<" -font $font_small -command "send_ssc_of_milkyway -0.1"
button .b_milky2 -text ">" -font $font_small -command "send_ssc_of_milkyway +0.1"
bind .e_milky <Return> "send_ssc_of_milkyway"
pack .l_milky -in .fr_milky
pack .b_milky0 -side left -fill both -in .fr_milky
pack .e_milky -expand 1 -side left -fill both -in .fr_milky
pack .b_milky1 .b_milky2 -side left -fill both -in .fr_milky

#

frame .fr_gen_lbl01

set buffer_absolute_star_scale $asarr_initial_settings(StelSkyDrawer.absoluteStarScale)
frame .fr_gen_label001
label .l_gen_label001 -text "\:" -font $font_small
button .b_absstar0 -text "l" -font $font_small -command "send_ssc_of_absolute_star_scale -"
entry .e_absstar -width 4 -font $font_small -textvariable buffer_absolute_star_scale
bind .e_absstar <Return> "send_ssc_of_absolute_star_scale"
pack .l_gen_label001 -in .fr_gen_label001
pack .b_absstar0 .e_absstar -side left -expand 1 -fill both -in .fr_gen_label001

set buffer_relative_star_scale $asarr_initial_settings(StelSkyDrawer.relativeStarScale)
frame .fr_gen_label002
label .l_gen_label002 -text "\:" -font $font_small
button .b_relstar0 -text "l" -font $font_small -command "send_ssc_of_relative_star_scale -"
entry .e_relstar -width 4 -font $font_small -textvariable buffer_relative_star_scale
bind .e_relstar <Return> "send_ssc_of_relative_star_scale"
pack .l_gen_label002 -in .fr_gen_label002
pack .b_relstar0 .e_relstar -side left -expand 1 -fill both -in .fr_gen_label002

pack .fr_gen_label001 .fr_gen_label002 -side left -expand 1 -fill both -in .fr_gen_lbl01

#

frame .fr_gen_lbl02

frame .fr_gen_label003
label .l_gen_label003 -text "Q" -font $font_small
button .b_metlbl0 -text "Off" -font $font_small -command "send_ssc_lines MeteorShowers.setEnableMarker(false)"
button .b_metlbl1 -text "ON" -font $font_small -command "send_ssc_lines MeteorShowers.setEnableMarker(true)"
pack .l_gen_label003 -in .fr_gen_label003
pack .b_metlbl0 .b_metlbl1 -side left -expand 1 -fill both -in .fr_gen_label003

set buffer_meteor_zhr "0"
frame .fr_gen_label004
label .l_gen_label004 -text "ZHR" -font $font_small
button .b_metzhr0 -text "0" -font $font_small -command "send_ssc_of_meteor_zhr -"
entry .e_metzhr -width 6 -font $font_small -textvariable buffer_meteor_zhr
button .b_metzhr1 -text "<" -font $font_small -command "send_ssc_of_meteor_zhr -1000"
button .b_metzhr2 -text ">" -font $font_small -command "send_ssc_of_meteor_zhr +1000"
bind .e_metzhr <Return> "send_ssc_of_meteor_zhr"
pack .l_gen_label004 -in .fr_gen_label004
pack .b_metzhr0 -side left -expand 0 -fill both -in .fr_gen_label004
pack .e_metzhr  -side left -expand 1 -fill both -in .fr_gen_label004
pack .b_metzhr1 .b_metzhr2 -side left -expand 0 -fill both -in .fr_gen_label004

pack .fr_gen_label003 .fr_gen_label004 -side left -expand 1 -fill both -in .fr_gen_lbl02

#

frame .fr_gen_lbl03

frame .fr_gen_label005
label .l_gen_label005 -text "p" -font $font_small
button .b_gen_label005_1 -text "ON" -font $font_small -command "exec_ssc_on_server setCardinal.ssc"
button .b_gen_label005_0 -text "Off" -font $font_small -command "exec_ssc_on_server unsetCardinal.ssc"
pack .l_gen_label005 -expand 1 -fill y -in .fr_gen_label005
pack .b_gen_label005_0 .b_gen_label005_1 -side left -expand 1 -fill x -in .fr_gen_label005

set buffer_landscape_transparency $asarr_initial_settings(LandscapeMgr.landscapeTransparency)
frame .fr_gen_label006
frame .fr_gen_label006u
label .l_gen_label006 -text "i" -font $font_small
label .l_gen_label006_0 -text "" -font $font_small
button .b_gen_label006_1 -text "" -font $font_small -command "send_ssc_of_landscape_transparency -"
entry .e_land_trans -width 3 -font $font_small -textvariable buffer_landscape_transparency
button .b_gen_label006_3 -text "ON" -font $font_small -command "send_ssc_lines LandscapeMgr.setFlagLandscape(true)"
button .b_gen_label006_2 -text "Off" -font $font_small -command "send_ssc_lines LandscapeMgr.setFlagLandscape(false)"
bind .e_land_trans <Return> "send_ssc_of_landscape_transparency"
pack .l_gen_label006 -side left -expand 1 -fill both -in .fr_gen_label006u
pack .l_gen_label006_0 .b_gen_label006_1 .e_land_trans -side left -expand 0 -fill both -in .fr_gen_label006u
pack .fr_gen_label006u -expand 1 -fill x -in .fr_gen_label006
pack .b_gen_label006_2 .b_gen_label006_3 -side left -expand 1 -fill x -in .fr_gen_label006

pack .fr_gen_label005 .fr_gen_label006 -side left -expand 1 -fill both -in .fr_gen_lbl03

#

frame .fr_gen_lbl04

label .l_gen_label007 -text "pؑ" -font $font_small
entry .e_azch_dt -width 2 -font $font_small -textvariable buffer_azchange_duration
label .l_gen_label008 -text "b" -font $font_small
button .b_gen_label0080 -text "<" -font $font_small -command "update_azchange_duration -1"
button .b_gen_label0081 -text ">" -font $font_small -command "update_azchange_duration +1"
bind .e_azch_dt <Return> "update_azchange_duration"
pack .l_gen_label007 -side left -expand 1 -fill both -in .fr_gen_lbl04
pack .e_azch_dt .l_gen_label008 .b_gen_label0080 .b_gen_label0081 -side left -expand 0 -fill both -in .fr_gen_lbl04

frame .fr_gen_lbl05

button .b_gen_label0090 -text "  0" -font $font_small -command "send_ssc_of_view_azimuth 0"
button .b_gen_label0091 -text " 90" -font $font_small -command "send_ssc_of_view_azimuth 90"
button .b_gen_label0092 -text "180" -font $font_small -command "send_ssc_of_view_azimuth 180"
button .b_gen_label0093 -text "270" -font $font_small -command "send_ssc_of_view_azimuth 270"
pack .b_gen_label0090 .b_gen_label0091 .b_gen_label0092 .b_gen_label0093 -side left -expand 1 -fill both -in .fr_gen_lbl05

#

#frame .fr_land
#label .l_land -text "i"
#button .b_land1 -text "ON" -command "send_ssc_lines LandscapeMgr.setFlagLandscape(true)"
#button .b_land0 -text "Off" -command "send_ssc_lines LandscapeMgr.setFlagLandscape(false)"
#pack .l_land -in .fr_land
#pack .b_land0 .b_land1 -side left -expand 1 -fill both -in .fr_land

#frame .fr_ordinal
#label .l_ordinal -text "p"
#button .b_ordinal1 -text "ON" -command "exec_ssc_on_server setCardinal.ssc"
#button .b_ordinal0 -text "Off" -command "exec_ssc_on_server unsetCardinal.ssc"
#pack .l_ordinal -in .fr_ordinal
#pack .b_ordinal0 .b_ordinal1 -side left -expand 1 -fill both -in .fr_ordinal

#

set buffer_location_latitude $asarr_initial_settings(Latitude)
frame .fr_loclat
label .l_loclat -text "ϑn ܓx" -font $font_small
button .b_loclat0 -text "l" -font $font_small -command "send_ssc_of_location_latitude -"
entry .e_loclat -width 10 -textvariable buffer_location_latitude -font $font_small
button .b_loclat1 -text "<" -font $font_small -command "send_ssc_of_location_latitude -5.0"
button .b_loclat2 -text ">" -font $font_small -command "send_ssc_of_location_latitude +5.0"
bind .e_loclat <Return> "send_ssc_of_location_latitude"
pack .l_loclat -in .fr_loclat
pack .b_loclat0 -side left -fill both -in .fr_loclat
pack .e_loclat -expand 1 -side left -fill both -in .fr_loclat
pack .b_loclat1 .b_loclat2 -side left -fill both -in .fr_loclat

#

set buffer_min_fps $asarr_initial_settings(MinFps)
frame .fr_minfps
label .l_minfps -text "ŏFPS" -font $font_small
button .b_minfps0 -text "l" -font $font_small -command "send_ssc_of_min_fps -"
entry .e_minfps -width 10 -textvariable buffer_min_fps -font $font_small
button .b_minfps1 -text "<" -font $font_small -command "send_ssc_of_min_fps -1.0"
button .b_minfps2 -text ">" -font $font_small -command "send_ssc_of_min_fps +1.0"
bind .e_minfps <Return> "send_ssc_of_min_fps"
pack .l_minfps -in .fr_minfps
pack .b_minfps0 -side left -fill both -in .fr_minfps
pack .e_minfps -expand 1 -side left -fill both -in .fr_minfps
pack .b_minfps1 .b_minfps2 -side left -fill both -in .fr_minfps

#

frame .fr_datetm
label .l_datetm -text "v on Screen" -font $font_small
button .b_datetm0 -text "Off" -font $font_small -command "kill_date_monitor"
button .b_datetm1 -text "ON" -font $font_small -command "exec_date_monitor"
button .b_datetm2 -text "p" -font $font_small -command "change_date_monitor"
pack .l_datetm -in .fr_datetm
pack .b_datetm0 .b_datetm1 .b_datetm2 -side left -expand 1 -fill both -in .fr_datetm

pack .fr_init .fr_tmrate .fr_illumi .fr_atmos .fr_milky .fr_gen_lbl01 .fr_gen_lbl02 .fr_gen_lbl03 -expand 0 -fill both -in .lf_general
if { $flag_dome_mode != 0 } {
  pack .fr_gen_lbl04 .fr_gen_lbl05 -expand 0 -fill both -in .lf_general
}
pack .fr_loclat .fr_minfps -expand 0 -fill both -in .lf_general
if { $flag_dome_mode != 0 } {
  pack .fr_datetm -expand 0 -fill both -in .lf_general
}

.pw add .lf_general


################################################################

labelframe .lf_constel -text "Constellation" -font $font_labelframe

frame .fr_name
label .l_name -text ""
button .b_name1 -text "ON" -command "send_ssc_lines ConstellationMgr.setFlagLabels(true)"
button .b_name0 -text "Off" -command "send_ssc_lines ConstellationMgr.setFlagLabels(false)"
pack .l_name -in .fr_name
pack .b_name0 .b_name1 -side left -expand 1 -fill both -in .fr_name

frame .fr_line
label .l_line -text ""
button .b_line1 -text "ON" -command "send_ssc_lines ConstellationMgr.setFlagLines(true)"
button .b_line0 -text "Off" -command "send_ssc_lines ConstellationMgr.setFlagLines(false)"
pack .l_line -in .fr_line
pack .b_line0 .b_line1 -side left -expand 1 -fill both -in .fr_line

frame .fr_art
label .l_art -text "G"
button .b_art0 -text "Off" -command "send_ssc_lines ConstellationMgr.setFlagArt(false)"
button .b_art1 -text "1" -command "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.1)"
button .b_art2 -text "2" -command "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.2)"
button .b_art3 -text "3" -command "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.3)"
button .b_art4 -text "4" -command "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.4)"
button .b_art5 -text "5" -command "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.5)"
pack .l_art -in .fr_art
pack .b_art0 .b_art1 .b_art2 .b_art3 .b_art4 .b_art5 -side left -expand 1 -fill both -in .fr_art

frame .fr_boun
label .l_boun -text "E"
button .b_boun1 -text "ON" -command "send_ssc_lines ConstellationMgr.setFlagBoundaries(true)"
button .b_boun0 -text "Off" -command "send_ssc_lines ConstellationMgr.setFlagBoundaries(false)"
pack .l_boun -in .fr_boun
pack .b_boun0 .b_boun1 -side left -expand 1 -fill both -in .fr_boun

frame .fr_astername
label .l_astername -text "AXeY"
button .b_astername0 -text " Off " -command "send_ssc_lines AsterismMgr.setFlagLabels(false)"
button .b_astername1 -text " ON " -command "send_ssc_lines AsterismMgr.setFlagLabels(true)"
pack .l_astername -in .fr_astername
pack .b_astername0 .b_astername1 -side left -expand 1 -fill both -in .fr_astername

frame .fr_aster
label .l_aster -text "AXeY"
button .b_aster0 -text "Off" -command "send_ssc_lines AsterismMgr.deselectAsterisms() AsterismMgr.deselectAsterisms()"
pack .l_aster -in .fr_aster
pack .b_aster0 -side left -expand 1 -fill both -in .fr_aster

frame .fr_all
label .l_all -text " I"
button .b_all2 -text "S88" -command "exec_ssc_on_server setConstellationAll.ssc"
button .b_all1 -text "12" -command "exec_ssc_on_server setEclipticConstellationAll.ssc"
#button .b_all0 -text "Unset" -command "send_ssc_lines ConstellationMgr.deselectConstellations()"
button .b_all0 -text "Unset" -command "exec_ssc_on_server unsetConstellationAll.ssc"
pack .l_all -in .fr_all
pack .b_all0 .b_all1 .b_all2 -side left -expand 1 -fill both -in .fr_all

frame .fr_strlabel
label .l_strlabel -text "P "
button .b_strlabel0 -text "Off" -command "send_ssc_of_stellarlabels -"
entry .e_strlabel -width 4 -textvariable buffer_stellarlabels
set buffer_stellarlabels "0.0"
button .b_strlabel1 -text "<" -command "send_ssc_of_stellarlabels -0.1"
button .b_strlabel2 -text ">" -command "send_ssc_of_stellarlabels +0.1"
bind .e_strlabel <Return> "send_ssc_of_stellarlabels"
pack .l_strlabel -in .fr_strlabel
pack .b_strlabel0 -side left -fill both -in .fr_strlabel
pack .e_strlabel -expand 1 -side left -fill both -in .fr_strlabel
pack .b_strlabel1 .b_strlabel2 -side left -fill both -in .fr_strlabel

pack .fr_name .fr_line .fr_art .fr_boun .fr_astername .fr_aster .fr_all .fr_strlabel -expand 0 -fill both -in .lf_constel

.pw add .lf_constel


################################################################
#  t

labelframe .lf_spring -text "Spring" -font $font_labelframe

frame .fr_Leo
label .l_Leo -text ""
button .b_Leo1 -text " SET " -command "update_constellation Leo true"
button .b_Leo0 -text "Unset" -command "update_constellation Leo false"
pack .l_Leo -in .fr_Leo
pack .b_Leo0 .b_Leo1 -side left -expand 1 -fill both -in .fr_Leo

frame .fr_Vir
label .l_Vir -text "Ƃߍ"
button .b_Vir1 -text " SET " -command "update_constellation Virgo true"
button .b_Vir0 -text "Unset" -command "update_constellation Virgo false"
pack .l_Vir -in .fr_Vir
pack .b_Vir0 .b_Vir1 -side left -expand 1 -fill both -in .fr_Vir

frame .fr_Boo
label .l_Boo -text ""
button .b_Boo1 -text " SET " -command "update_constellation Bootes true"
button .b_Boo0 -text "Unset" -command "update_constellation Bootes false"
pack .l_Boo -in .fr_Boo
pack .b_Boo0 .b_Boo1 -side left -expand 1 -fill both -in .fr_Boo

frame .fr_UMa
label .l_UMa -text "܍"
button .b_UMa1 -text " SET " -command "update_constellation Ursa~Major true"
button .b_UMa0 -text "Unset" -command "update_constellation Ursa~Major false"
pack .l_UMa -in .fr_UMa
pack .b_UMa0 .b_UMa1 -side left -expand 1 -fill both -in .fr_UMa

frame .fr_UMi
label .l_UMi -text "܍"
button .b_UMi1 -text " SET " -command "update_constellation Ursa~Minor true"
button .b_UMi0 -text "Unset" -command "update_constellation Ursa~Minor false"
pack .l_UMi -in .fr_UMi
pack .b_UMi0 .b_UMi1 -side left -expand 1 -fill both -in .fr_UMi

frame .fr_Crv
label .l_Crv -text "炷"
button .b_Crv1 -text " SET " -command "update_constellation Corvus true"
button .b_Crv0 -text "Unset" -command "update_constellation Corvus false"
pack .l_Crv -in .fr_Crv
pack .b_Crv0 .b_Crv1 -side left -expand 1 -fill both -in .fr_Crv

frame .fr_Cnc
label .l_Cnc -text "ɍ"
button .b_Cnc1 -text " SET " -command "update_constellation Cancer true"
button .b_Cnc0 -text "Unset" -command "update_constellation Cancer false"
pack .l_Cnc -in .fr_Cnc
pack .b_Cnc0 .b_Cnc1 -side left -expand 1 -fill both -in .fr_Cnc

frame .fr_Com
label .l_Com -text "݂̂" -font $font_small
button .b_Com1 -text " SET " -font $font_small -command "update_constellation Coma~Berenices true"
button .b_Com0 -text "Unset" -font $font_small -command "update_constellation Coma~Berenices false"
pack .l_Com -in .fr_Com
pack .b_Com0 .b_Com1 -side left -expand 1 -fill both -in .fr_Com

frame .fr_Hya
label .l_Hya -text "݂ւэ" -font $font_small
button .b_Hya1 -text " SET " -font $font_small -command "update_constellation Hydra true"
button .b_Hya0 -text "Unset" -font $font_small -command "update_constellation Hydra false"
pack .l_Hya -in .fr_Hya
pack .b_Hya0 .b_Hya1 -side left -expand 1 -fill both -in .fr_Hya

frame .fr_CVn
label .l_CVn -text "傤" -font $font_small
button .b_CVn1 -text " SET " -font $font_small -command "update_constellation Canes~Venatici true"
button .b_CVn0 -text "Unset" -font $font_small -command "update_constellation Canes~Venatici false"
pack .l_CVn -in .fr_CVn
pack .b_CVn0 .b_CVn1 -side left -expand 1 -fill both -in .fr_CVn

frame .fr_Cra
label .l_Cra -text "Rbv" -font $font_small
button .b_Cra1 -text " SET " -font $font_small -command "update_constellation Crater true"
button .b_Cra0 -text "Unset" -font $font_small -command "update_constellation Crater false"
pack .l_Cra -in .fr_Cra
pack .b_Cra0 .b_Cra1 -side left -expand 1 -fill both -in .fr_Cra

pack .fr_Leo .fr_Vir .fr_Boo .fr_UMa .fr_UMi .fr_Crv .fr_Cnc .fr_Com .fr_Hya .fr_CVn .fr_Cra -expand 0 -fill both -in .lf_spring

.pw add .lf_spring

################################################################
#  

labelframe .lf_summer -text "Summer" -font $font_labelframe

frame .fr_lyr
label .l_lyr -text "ƍ"
button .b_lyr1 -text " SET " -command "update_constellation Lyra true"
button .b_lyr0 -text "Unset" -command "update_constellation Lyra false"
pack .l_lyr -in .fr_lyr
pack .b_lyr0 .b_lyr1 -side left -expand 1 -fill both -in .fr_lyr

frame .fr_cyg
label .l_cyg -text "͂傤"
button .b_cyg1 -text " SET " -command "update_constellation Cygnus true"
button .b_cyg0 -text "Unset" -command "update_constellation Cygnus false"
pack .l_cyg -in .fr_cyg
pack .b_cyg0 .b_cyg1 -side left -expand 1 -fill both -in .fr_cyg

frame .fr_aqu
label .l_aqu -text "킵"
button .b_aqu1 -text " SET " -command "update_constellation Aquila true"
button .b_aqu0 -text "Unset" -command "update_constellation Aquila false"
pack .l_aqu -in .fr_aqu
pack .b_aqu0 .b_aqu1 -side left -expand 1 -fill both -in .fr_aqu

frame .fr_Sco
label .l_Sco -text ""
button .b_Sco1 -text " SET " -command "update_constellation Scorpius true"
button .b_Sco0 -text "Unset" -command "update_constellation Scorpius false"
pack .l_Sco -in .fr_Sco
pack .b_Sco0 .b_Sco1 -side left -expand 1 -fill both -in .fr_Sco

frame .fr_Sgr
label .l_Sgr -text "č"
button .b_Sgr1 -text " SET " -command "update_constellation Sagittarius true"
button .b_Sgr0 -text "Unset" -command "update_constellation Sagittarius false"
pack .l_Sgr -in .fr_Sgr
pack .b_Sgr0 .b_Sgr1 -side left -expand 1 -fill both -in .fr_Sgr

frame .fr_Lib
label .l_Lib -text "Ăт"
button .b_Lib1 -text " SET " -command "update_constellation Libra true"
button .b_Lib0 -text "Unset" -command "update_constellation Libra false"
pack .l_Lib -in .fr_Lib
pack .b_Lib0 .b_Lib1 -side left -expand 1 -fill both -in .fr_Lib

frame .fr_Her
label .l_Her -text "wNX"
button .b_Her1 -text " SET " -command "update_constellation Hercules true"
button .b_Her0 -text "Unset" -command "update_constellation Hercules false"
pack .l_Her -in .fr_Her
pack .b_Her0 .b_Her1 -side left -expand 1 -fill both -in .fr_Her

frame .fr_CrB
label .l_CrB -text "ނ" -font $font_small
button .b_CrB1 -text " SET " -font $font_small -command "update_constellation Corona~Borealis true"
button .b_CrB0 -text "Unset" -font $font_small -command "update_constellation Corona~Borealis false"
pack .l_CrB -in .fr_CrB
pack .b_CrB0 .b_CrB1 -side left -expand 1 -fill both -in .fr_CrB

frame .fr_Oph
label .l_Oph -text "ւт" -font $font_small
button .b_Oph1 -text " SET " -font $font_small -command "update_constellation Ophiuchus true"
button .b_Oph0 -text "Unset" -font $font_small -command "update_constellation Ophiuchus false"
pack .l_Oph -in .fr_Oph
pack .b_Oph0 .b_Oph1 -side left -expand 1 -fill both -in .fr_Oph

frame .fr_Ser
label .l_Ser -text "ւэ" -font $font_small
button .b_Ser1 -text " SET " -font $font_small -command "update_constellation Serpens true"
button .b_Ser0 -text "Unset" -font $font_small -command "update_constellation Serpens false"
pack .l_Ser -in .fr_Ser
pack .b_Ser0 .b_Ser1 -side left -expand 1 -fill both -in .fr_Ser

#frame .fr_Del
#label .l_Del -text "邩" -font $font_small
#button .b_Del1 -text " SET " -font $font_small -command "update_constellation Delphinus true"
#button .b_Del0 -text "Unset" -font $font_small -command "update_constellation Delphinus false"
#pack .l_Del -in .fr_Del
#pack .b_Del0 .b_Del1 -side left -expand 1 -fill both -in .fr_Del

#frame .fr_Dra
#label .l_Dra -text "イ" -font $font_small
#button .b_Dra1 -text " SET " -font $font_small -command "update_constellation Draco true"
#button .b_Dra0 -text "Unset" -font $font_small -command "update_constellation Draco false"
#pack .l_Dra -in .fr_Dra
#pack .b_Dra0 .b_Dra1 -side left -expand 1 -fill both -in .fr_Dra

#

frame .fr_cnst_lbl11

frame .fr_c_101
label .l_c_101 -text "イ" -font $font_small
button .b_c_1011 -text "SET" -font $font_small -command "update_constellation Draco true"
button .b_c_1010 -text "Unset" -font $font_small -command "update_constellation Draco false"
pack .l_c_101 -in .fr_c_101
pack .b_c_1010 .b_c_1011 -side left -expand 1 -fill both -in .fr_c_101

frame .fr_c_102
label .l_c_102 -text "邩" -font $font_small
button .b_c_1021 -text "SET" -font $font_small -command "update_constellation Delphinus true"
button .b_c_1020 -text "Unset" -font $font_small -command "update_constellation Delphinus false"
pack .l_c_102 -in .fr_c_102
pack .b_c_1020 .b_c_1021 -side left -expand 1 -fill both -in .fr_c_102

pack .fr_c_101 .fr_c_102 -side left -expand 1 -fill both -in .fr_cnst_lbl11

#

pack .fr_lyr .fr_cyg .fr_aqu .fr_Sco .fr_Sgr .fr_Lib .fr_Her .fr_CrB .fr_Oph .fr_Ser .fr_cnst_lbl11 -expand 0 -fill both -in .lf_summer

.pw add .lf_summer

################################################################
#  H

labelframe .lf_autumn -text "Autumn" -font $font_labelframe

frame .fr_Cas
label .l_Cas -text "JVIyA"
button .b_Cas1 -text " SET " -command "update_constellation Cassiopeia true"
button .b_Cas0 -text "Unset" -command "update_constellation Cassiopeia false"
pack .l_Cas -in .fr_Cas
pack .b_Cas0 .b_Cas1 -side left -expand 1 -fill both -in .fr_Cas

frame .fr_Per
label .l_Per -text "yZEX"
button .b_Per1 -text " SET " -command "update_constellation Perseus true"
button .b_Per0 -text "Unset" -command "update_constellation Perseus false"
pack .l_Per -in .fr_Per
pack .b_Per0 .b_Per1 -side left -expand 1 -fill both -in .fr_Per

frame .fr_And
label .l_And -text "Ah_"
button .b_And1 -text " SET " -command "update_constellation Andromeda true"
button .b_And0 -text "Unset" -command "update_constellation Andromeda false"
pack .l_And -in .fr_And
pack .b_And0 .b_And1 -side left -expand 1 -fill both -in .fr_And

frame .fr_Peg
label .l_Peg -text "yKXX"
button .b_Peg1 -text " SET " -command "update_constellation Pegasus true"
button .b_Peg0 -text "Unset" -command "update_constellation Pegasus false"
pack .l_Peg -in .fr_Peg
pack .b_Peg0 .b_Peg1 -side left -expand 1 -fill both -in .fr_Peg

frame .fr_Psc
label .l_Psc -text ""
button .b_Psc1 -text " SET " -command "update_constellation Pisces true"
button .b_Psc0 -text "Unset" -command "update_constellation Pisces false"
pack .l_Psc -in .fr_Psc
pack .b_Psc0 .b_Psc1 -side left -expand 1 -fill both -in .fr_Psc

frame .fr_Cet
label .l_Cet -text ""
button .b_Cet1 -text " SET " -command "update_constellation Cetus true"
button .b_Cet0 -text "Unset" -command "update_constellation Cetus false"
pack .l_Cet -in .fr_Cet
pack .b_Cet0 .b_Cet1 -side left -expand 1 -fill both -in .fr_Cet

frame .fr_Cep
label .l_Cep -text "PtFEX"
button .b_Cep1 -text " SET " -command "update_constellation Cepheus true"
button .b_Cep0 -text "Unset" -command "update_constellation Cepheus false"
pack .l_Cep -in .fr_Cep
pack .b_Cep0 .b_Cep1 -side left -expand 1 -fill both -in .fr_Cep

frame .fr_PsA
label .l_PsA -text "݂Ȃ݂̂" -font $font_small
button .b_PsA1 -text " SET " -font $font_small -command "update_constellation Piscis~Austrinus true"
button .b_PsA0 -text "Unset" -font $font_small -command "update_constellation Piscis~Austrinus false"
pack .l_PsA -in .fr_PsA
pack .b_PsA0 .b_PsA1 -side left -expand 1 -fill both -in .fr_PsA

frame .fr_Aqr
label .l_Aqr -text "݂ߍ" -font $font_small
button .b_Aqr1 -text " SET " -font $font_small -command "update_constellation Aquarius true"
button .b_Aqr0 -text "Unset" -font $font_small -command "update_constellation Aquarius false"
pack .l_Aqr -in .fr_Aqr
pack .b_Aqr0 .b_Aqr1 -side left -expand 1 -fill both -in .fr_Aqr

#frame .fr_Cap
#label .l_Cap -text "€" -font $font_small
#button .b_Cap1 -text " SET " -font $font_small -command "update_constellation Capricornus true"
#button .b_Cap0 -text "Unset" -font $font_small -command "update_constellation Capricornus false"
#pack .l_Cap -in .fr_Cap
#pack .b_Cap0 .b_Cap1 -side left -expand 1 -fill both -in .fr_Cap

#frame .fr_Ari
#label .l_Ari -text "Ђ" -font $font_small
#button .b_Ari1 -text " SET " -font $font_small -command "update_constellation Aries true"
#button .b_Ari0 -text "Unset" -font $font_small -command "update_constellation Aries false"
#pack .l_Ari -in .fr_Ari
#pack .b_Ari0 .b_Ari1 -side left -expand 1 -fill both -in .fr_Ari

#

frame .fr_cnst_lbl21

frame .fr_c_201
label .l_c_201 -text "Ђ" -font $font_small
button .b_c_2011 -text "SET" -font $font_small -command "update_constellation Aries true"
button .b_c_2010 -text "Unset" -font $font_small -command "update_constellation Aries false"
pack .l_c_201 -in .fr_c_201
pack .b_c_2010 .b_c_2011 -side left -expand 1 -fill both -in .fr_c_201

frame .fr_c_202
label .l_c_202 -text "€" -font $font_small
button .b_c_2021 -text "SET" -font $font_small -command "update_constellation Capricornus true"
button .b_c_2020 -text "Unset" -font $font_small -command "update_constellation Capricornus false"
pack .l_c_202 -in .fr_c_202
pack .b_c_2020 .b_c_2021 -side left -expand 1 -fill both -in .fr_c_202

pack .fr_c_201 .fr_c_202 -side left -expand 1 -fill both -in .fr_cnst_lbl21

#

frame .fr_Scu
label .l_Scu -text "傤" -font $font_small
button .b_Scu1 -text " SET " -font $font_small -command "update_constellation Sculptor true"
button .b_Scu0 -text "Unset" -font $font_small -command "update_constellation Sculptor false"
pack .l_Scu -in .fr_Scu
pack .b_Scu0 .b_Scu1 -side left -expand 1 -fill both -in .fr_Scu

pack .fr_Cas .fr_Per .fr_And .fr_Peg .fr_Psc .fr_Cet .fr_Cep .fr_PsA .fr_Aqr .fr_cnst_lbl21 .fr_Scu -expand 0 -fill both -in .lf_autumn

.pw add .lf_autumn

################################################################
#  ~
labelframe .lf_winter -text "Winter" -font $font_labelframe

frame .fr_Ori
label .l_Ori -text "II"
button .b_Ori1 -text " SET " -command "update_constellation Orion true"
button .b_Ori0 -text "Unset" -command "update_constellation Orion false"
pack .l_Ori -in .fr_Ori
pack .b_Ori0 .b_Ori1 -side left -expand 1 -fill both -in .fr_Ori

frame .fr_CMa
label .l_CMa -text "ʍ"
button .b_CMa1 -text " SET " -command "update_constellation Canis~Major true"
button .b_CMa0 -text "Unset" -command "update_constellation Canis~Major false"
pack .l_CMa -in .fr_CMa
pack .b_CMa0 .b_CMa1 -side left -expand 1 -fill both -in .fr_CMa

frame .fr_CMi
label .l_CMi -text "ʍ"
button .b_CMi1 -text " SET " -command "update_constellation Canis~Minor true"
button .b_CMi0 -text "Unset" -command "update_constellation Canis~Minor false"
pack .l_CMi -in .fr_CMi
pack .b_CMi0 .b_CMi1 -side left -expand 1 -fill both -in .fr_CMi

frame .fr_Gem
label .l_Gem -text "ӂ"
button .b_Gem1 -text " SET " -command "update_constellation Gemini true"
button .b_Gem0 -text "Unset" -command "update_constellation Gemini false"
pack .l_Gem -in .fr_Gem
pack .b_Gem0 .b_Gem1 -side left -expand 1 -fill both -in .fr_Gem

frame .fr_Tau
label .l_Tau -text ""
button .b_Tau1 -text " SET " -command "update_constellation Taurus true"
button .b_Tau0 -text "Unset" -command "update_constellation Taurus false"
pack .l_Tau -in .fr_Tau
pack .b_Tau0 .b_Tau1 -side left -expand 1 -fill both -in .fr_Tau

frame .fr_Aur
label .l_Aur -text "債"
button .b_Aur1 -text " SET " -command "update_constellation Auriga true"
button .b_Aur0 -text "Unset" -command "update_constellation Auriga false"
pack .l_Aur -in .fr_Aur
pack .b_Aur0 .b_Aur1 -side left -expand 1 -fill both -in .fr_Aur

#frame .fr_Ari_w
#label .l_Ari_w -text "Ђ" -font $font_small
#button .b_Ari_w1 -text " SET " -font $font_small -command "update_constellation Aries true"
#button .b_Ari_w0 -text "Unset" -font $font_small -command "update_constellation Aries false"
#pack .l_Ari_w -in .fr_Ari_w
#pack .b_Ari_w0 .b_Ari_w1 -side left -expand 1 -fill both -in .fr_Ari_w

frame .fr_Lep
label .l_Lep -text "" -font $font_small
button .b_Lep1 -text " SET " -font $font_small -command "update_constellation Lepus true"
button .b_Lep0 -text "Unset" -font $font_small -command "update_constellation Lepus false"
pack .l_Lep -in .fr_Lep
pack .b_Lep0 .b_Lep1 -side left -expand 1 -fill both -in .fr_Lep

frame .fr_Col
label .l_Col -text "͂ƍ" -font $font_small
button .b_Col1 -text " SET " -font $font_small -command "update_constellation Columba true"
button .b_Col0 -text "Unset" -font $font_small -command "update_constellation Columba false"
pack .l_Col -in .fr_Col
pack .b_Col0 .b_Col1 -side left -expand 1 -fill both -in .fr_Col

frame .fr_Mon
label .l_Mon -text "イ" -font $font_small
button .b_Mon1 -text " SET " -font $font_small -command "update_constellation Monoceros true"
button .b_Mon0 -text "Unset" -font $font_small -command "update_constellation Monoceros false"
pack .l_Mon -in .fr_Mon
pack .b_Mon0 .b_Mon1 -side left -expand 1 -fill both -in .fr_Mon

frame .fr_Lyn
label .l_Lyn -text "܂˂" -font $font_small
button .b_Lyn1 -text " SET " -font $font_small -command "update_constellation Lynx true"
button .b_Lyn0 -text "Unset" -font $font_small -command "update_constellation Lynx false"
pack .l_Lyn -in .fr_Lyn
pack .b_Lyn0 .b_Lyn1 -side left -expand 1 -fill both -in .fr_Lyn

frame .fr_Cam
label .l_Cam -text "" -font $font_small
button .b_Cam1 -text " SET " -font $font_small -command "update_constellation Camelopardalis true"
button .b_Cam0 -text "Unset" -font $font_small -command "update_constellation Camelopardalis false"
pack .l_Cam -in .fr_Cam
pack .b_Cam0 .b_Cam1 -side left -expand 1 -fill both -in .fr_Cam

pack .fr_Ori .fr_CMa .fr_CMi .fr_Gem .fr_Tau .fr_Aur .fr_Lep .fr_Col .fr_Mon .fr_Lyn .fr_Cam -expand 0 -fill both -in .lf_winter
.pw add .lf_winter

################################################################

#
# See https://github.com/Stellarium/stellarium/blob/master/skycultures/modern/asterism_names.eng.fab
#

labelframe .lf_spring_asterism_star -text "Spring Asterism/Stars" -font $font_labelframe

frame .fr_001
label .l_001 -text "kl"
button .b_0011 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~(Plough)')"
button .b_0010 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~(Plough)')"
pack .l_001 -in .fr_001
pack .b_0010 .b_0011 -side left -expand 1 -fill both -in .fr_001

frame .fr_002
label .l_002 -text "kl5{"
button .b_0021 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~5times')"
button .b_0020 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~5times')"
pack .l_002 -in .fr_002
pack .b_0020 .b_0021 -side left -expand 1 -fill both -in .fr_002

frame .fr_003
label .l_003 -text "t̑Op"
button .b_0031 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Spring~Triangle')"
button .b_0030 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Spring~Triangle')"
pack .l_003 -in .fr_003
pack .b_0030 .b_0031 -side left -expand 1 -fill both -in .fr_003

frame .fr_004
label .l_004 -text "t̃_Ch"
button .b_0041 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Great~Diamond')"
button .b_0040 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Great~Diamond')"
pack .l_004 -in .fr_004
pack .b_0040 .b_0041 -side left -expand 1 -fill both -in .fr_004

frame .fr_005
label .l_005 -text "t̑Ȑ"
button .b_0051 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Spring~Curve')"
button .b_0050 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Spring~Curve')"
pack .l_005 -in .fr_005
pack .b_0050 .b_0051 -side left -expand 1 -fill both -in .fr_005

#frame .fr_006
#label .l_006 -text "kɐ"
#button .b_0061 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Polaris')"
#button .b_0060 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Polaris')"
#pack .l_006 -in .fr_006
#pack .b_0060 .b_0061 -side left -expand 1 -fill both -in .fr_006

frame .fr_007
label .l_007 -text "JVIy5{" -font $font_small
button .b_0071 -text " SET " -font $font_small -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Cassio~5times')"
button .b_0070 -text "Unset" -font $font_small -command "send_ssc_lines AsterismMgr.deselectAsterism('Cassio~5times')"
pack .l_007 -in .fr_007
pack .b_0070 .b_0071 -side left -expand 1 -fill both -in .fr_007

#

frame .fr_astar_lbl00 -height 12

#

frame .fr_astar_lbl01

label .l_astar_eachlbl02 -text "ʕ\" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
button .b_astar_off02 -text "SOff" -font $font_small -command "unset_star_label Polaris Spica Arcturus Regulus Denebola HIP63125A"
pack .l_astar_eachlbl02 .b_astar_off02 -side left -expand 1 -fill both -in .fr_astar_lbl01

#

frame .fr_astar_lbl02

frame .fr_021
label .l_021 -text "kɐ" -font $font_small
button .b_0211 -text "ON" -font $font_small -command "set_star_label Polaris"
button .b_0210 -text "Off" -font $font_small -command "unset_star_label Polaris"
pack .l_021 -in .fr_021
pack .b_0210 .b_0211 -side left -expand 1 -fill both -in .fr_021

pack .fr_021 -side left -expand 1 -fill both -in .fr_astar_lbl02

#

frame .fr_astar_lbl03

frame .fr_022
label .l_022 -text "A[NgDX" -font $font_small
button .b_0221 -text "ON" -font $font_small -command "set_star_label Arcturus"
button .b_0220 -text "Off" -font $font_small -command "unset_star_label Arcturus"
pack .l_022 -in .fr_022
pack .b_0220 .b_0221 -side left -expand 1 -fill both -in .fr_022

pack .fr_022 -side left -expand 1 -fill both -in .fr_astar_lbl03

#

frame .fr_astar_lbl04

frame .fr_023
label .l_023 -text "XsJ" -font $font_small
button .b_0231 -text "ON" -font $font_small -command "set_star_label Spica"
button .b_0230 -text "Off" -font $font_small -command "unset_star_label Spica"
pack .l_023 -in .fr_023
pack .b_0230 .b_0231 -side left -expand 1 -fill both -in .fr_023

frame .fr_024
label .l_024 -text "OX" -font $font_small
button .b_0241 -text "ON" -font $font_small -command "set_star_label Regulus"
button .b_0240 -text "Off" -font $font_small -command "unset_star_label Regulus"
pack .l_024 -in .fr_024
pack .b_0240 .b_0241 -side left -expand 1 -fill both -in .fr_024

pack .fr_023 .fr_024 -side left -expand 1 -fill both -in .fr_astar_lbl04

#

frame .fr_astar_lbl05

frame .fr_025
label .l_025 -text "fl{" -font $font_small
button .b_0251 -text "ON" -font $font_small -command "set_star_label Denebola"
button .b_0250 -text "Off" -font $font_small -command "unset_star_label Denebola"
pack .l_025 -in .fr_025
pack .b_0250 .b_0251 -side left -expand 1 -fill both -in .fr_025

frame .fr_026
label .l_026 -text "REJ" -font $font_small
button .b_0261 -text "ON" -font $font_small -command "set_star_label HIP63125A"
button .b_0260 -text "Off" -font $font_small -command "unset_star_label HIP63125A"
pack .l_026 -in .fr_026
pack .b_0260 .b_0261 -side left -expand 1 -fill both -in .fr_026

pack .fr_025 .fr_026 -side left -expand 1 -fill both -in .fr_astar_lbl05

#

pack .fr_001 .fr_002 .fr_003 .fr_004 .fr_005 .fr_007 .fr_astar_lbl00 .fr_astar_lbl01 .fr_astar_lbl02 .fr_astar_lbl03 .fr_astar_lbl04 .fr_astar_lbl05 -expand 0 -fill both -in .lf_spring_asterism_star

.pw add .lf_spring_asterism_star

################################################################

#
# See https://github.com/Stellarium/stellarium/blob/master/skycultures/modern/asterism_names.eng.fab
#

labelframe .lf_summer_asterism_star -text "Summer Asterism/Stars" -font $font_labelframe


frame .fr_101
label .l_101 -text "kl"
button .b_1011 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~(Plough)')"
button .b_1010 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~(Plough)')"
pack .l_101 -in .fr_101
pack .b_1010 .b_1011 -side left -expand 1 -fill both -in .fr_101

frame .fr_102
label .l_102 -text "kl5{"
button .b_1021 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~5times')"
button .b_1020 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~5times')"
pack .l_102 -in .fr_102
pack .b_1020 .b_1021 -side left -expand 1 -fill both -in .fr_102

frame .fr_103
label .l_103 -text "Ă̑Op"
button .b_1031 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Summer~Triangle')"
button .b_1030 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Summer~Triangle')"
pack .l_103 -in .fr_103
pack .b_1030 .b_1031 -side left -expand 1 -fill both -in .fr_103

frame .fr_104
label .l_104 -text "eB[|bg"
button .b_1041 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Teapot')"
button .b_1040 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Teapot')"
pack .l_104 -in .fr_104
pack .b_1040 .b_1041 -side left -expand 1 -fill both -in .fr_104

frame .fr_105
label .l_105 -text "lZ"
button .b_1051 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Milk~Dipper')"
button .b_1050 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Milk~Dipper')"
pack .l_105 -in .fr_105
pack .b_1050 .b_1051 -side left -expand 1 -fill both -in .fr_105

#frame .fr_106
#label .l_106 -text "kɐ"
#button .b_1061 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Polaris')"
#button .b_1060 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Polaris')"
#pack .l_106 -in .fr_106
#pack .b_1060 .b_1061 -side left -expand 1 -fill both -in .fr_106

frame .fr_107
label .l_107 -text "JVIy5{"
button .b_1071 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Cassio~5times')"
button .b_1070 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Cassio~5times')"
pack .l_107 -in .fr_107
pack .b_1070 .b_1071 -side left -expand 1 -fill both -in .fr_107

#

frame .fr_astar_lbl10 -height 12

#

frame .fr_astar_lbl11

label .l_astar_eachlbl12 -text "ʕ\" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
button .b_astar_off12 -text "SOff" -font $font_small -command "unset_star_label Polaris Vega Altair Deneb Albireo Antares"
pack .l_astar_eachlbl12 .b_astar_off12 -side left -expand 1 -fill both -in .fr_astar_lbl11

#

frame .fr_astar_lbl12

frame .fr_121
label .l_121 -text "kɐ" -font $font_small
button .b_1211 -text "ON" -font $font_small -command "set_star_label Polaris"
button .b_1210 -text "Off" -font $font_small -command "unset_star_label Polaris"
pack .l_121 -in .fr_121
pack .b_1210 .b_1211 -side left -expand 1 -fill both -in .fr_121

pack .fr_121 -side left -expand 1 -fill both -in .fr_astar_lbl12

#

frame .fr_astar_lbl13

frame .fr_122
label .l_122 -text "xK" -font $font_small
button .b_1221 -text "ON" -font $font_small -command "set_star_label Vega"
button .b_1220 -text "Off" -font $font_small -command "unset_star_label Vega"
pack .l_122 -in .fr_122
pack .b_1220 .b_1221 -side left -expand 1 -fill both -in .fr_122

frame .fr_123
label .l_123 -text "A^C" -font $font_small
button .b_1231 -text "ON" -font $font_small -command "set_star_label Altair"
button .b_1230 -text "Off" -font $font_small -command "unset_star_label Altair"
pack .l_123 -in .fr_123
pack .b_1230 .b_1231 -side left -expand 1 -fill both -in .fr_123

pack .fr_122 .fr_123 -side left -expand 1 -fill both -in .fr_astar_lbl13

#

frame .fr_astar_lbl14

frame .fr_124
label .l_124 -text "flu" -font $font_small
button .b_1241 -text "ON" -font $font_small -command "set_star_label Deneb"
button .b_1240 -text "Off" -font $font_small -command "unset_star_label Deneb"
pack .l_124 -in .fr_124
pack .b_1240 .b_1241 -side left -expand 1 -fill both -in .fr_124

frame .fr_125
label .l_125 -text "ArI" -font $font_small
button .b_1251 -text "ON" -font $font_small -command "set_star_label Albireo"
button .b_1250 -text "Off" -font $font_small -command "unset_star_label Albireo"
pack .l_125 -in .fr_125
pack .b_1250 .b_1251 -side left -expand 1 -fill both -in .fr_125

pack .fr_124 .fr_125 -side left -expand 1 -fill both -in .fr_astar_lbl14

#

frame .fr_astar_lbl15

frame .fr_126
label .l_126 -text "A^X" -font $font_small
button .b_1261 -text "ON" -font $font_small -command "set_star_label Antares"
button .b_1260 -text "Off" -font $font_small -command "unset_star_label Antares"
pack .l_126 -in .fr_126
pack .b_1260 .b_1261 -side left -expand 1 -fill both -in .fr_126

pack .fr_126 -side left -expand 1 -fill both -in .fr_astar_lbl15

#

pack .fr_101 .fr_102 .fr_103 .fr_104 .fr_105 .fr_107 .fr_astar_lbl10 .fr_astar_lbl11 .fr_astar_lbl12 .fr_astar_lbl13 .fr_astar_lbl14 .fr_astar_lbl15 -expand 0 -fill both -in .lf_summer_asterism_star

.pw add .lf_summer_asterism_star

################################################################

#
# See https://github.com/Stellarium/stellarium/blob/master/skycultures/modern/asterism_names.eng.fab
#

labelframe .lf_autumn_asterism_star -text "Autumn Asterism/Stars" -font $font_labelframe

frame .fr_201
label .l_201 -text "kl"
button .b_2011 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~(Plough)')"
button .b_2010 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~(Plough)')"
pack .l_201 -in .fr_201
pack .b_2010 .b_2011 -side left -expand 1 -fill both -in .fr_201

frame .fr_202
label .l_202 -text "kl5{"
button .b_2021 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~5times')"
button .b_2020 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~5times')"
pack .l_202 -in .fr_202
pack .b_2020 .b_2021 -side left -expand 1 -fill both -in .fr_202

frame .fr_203
label .l_203 -text "H̎lӌ`"
button .b_2031 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Great~Square~of~Pegasus')"
button .b_2030 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Great~Square~of~Pegasus')"
pack .l_203 -in .fr_203
pack .b_2030 .b_2031 -side left -expand 1 -fill both -in .fr_203

#frame .fr_204
#label .l_204 -text "kɐ"
#button .b_2041 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Polaris')"
#button .b_2040 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Polaris')"
#pack .l_204 -in .fr_204
#pack .b_2040 .b_2041 -side left -expand 1 -fill both -in .fr_204

frame .fr_205
label .l_205 -text "JVIy5{"
button .b_2051 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Cassio~5times')"
button .b_2050 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Cassio~5times')"
pack .l_205 -in .fr_205
pack .b_2050 .b_2051 -side left -expand 1 -fill both -in .fr_205

frame .fr_206
label .l_206 -text "lӌ`kɐ"
button .b_2061 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Square~Lines')"
button .b_2060 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Square~Lines')"
pack .l_206 -in .fr_206
pack .b_2060 .b_2061 -side left -expand 1 -fill both -in .fr_206

#

frame .fr_astar_lbl20 -height 12

#

frame .fr_astar_lbl21

label .l_astar_eachlbl22 -text "ʕ\" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
button .b_astar_off22 -text "SOff" -font $font_small -command "unset_star_label Polaris Fomalhaut HIP3419 HIP677 HIP10826"
pack .l_astar_eachlbl22 .b_astar_off22 -side left -expand 1 -fill both -in .fr_astar_lbl21

#

frame .fr_astar_lbl22

frame .fr_221
label .l_221 -text "kɐ" -font $font_small
button .b_2211 -text "ON" -font $font_small -command "set_star_label Polaris"
button .b_2210 -text "Off" -font $font_small -command "unset_star_label Polaris"
pack .l_221 -in .fr_221
pack .b_2210 .b_2211 -side left -expand 1 -fill both -in .fr_221

frame .fr_222
label .l_222 -text "~" -font $font_small
button .b_2221 -text "ON" -font $font_small -command "set_star_label HIP10826"
button .b_2220 -text "Off" -font $font_small -command "unset_star_label HIP10826"
pack .l_222 -in .fr_222
pack .b_2220 .b_2221 -side left -expand 1 -fill both -in .fr_222

pack .fr_221 .fr_222 -side left -expand 1 -fill both -in .fr_astar_lbl22

#

frame .fr_astar_lbl23
label .l_223 -text "tH[}nEg" -font $font_small
button .b_2231 -text "ON" -font $font_small -command "set_star_label Fomalhaut"
button .b_2230 -text "Off" -font $font_small -command "unset_star_label Fomalhaut"
pack .l_223 -in .fr_astar_lbl23
pack .b_2230 .b_2231 -side left -expand 1 -fill both -in .fr_astar_lbl23

frame .fr_astar_lbl24
label .l_224 -text "fluEJCgX" -font $font_small
button .b_2241 -text "ON" -font $font_small -command "set_star_label HIP3419"
button .b_2240 -text "Off" -font $font_small -command "unset_star_label HIP3419"
pack .l_224 -in .fr_astar_lbl24
pack .b_2240 .b_2241 -side left -expand 1 -fill both -in .fr_astar_lbl24

frame .fr_astar_lbl25
label .l_225 -text "AtFbc" -font $font_small
button .b_2251 -text "ON" -font $font_small -command "set_star_label HIP677"
button .b_2250 -text "Off" -font $font_small -command "unset_star_label HIP677"
pack .l_225 -in .fr_astar_lbl25
pack .b_2250 .b_2251 -side left -expand 1 -fill both -in .fr_astar_lbl25

pack .fr_201 .fr_202 .fr_203 .fr_205 .fr_206 .fr_astar_lbl20 .fr_astar_lbl21 .fr_astar_lbl22 .fr_astar_lbl23 .fr_astar_lbl24 .fr_astar_lbl25 -expand 0 -fill both -in .lf_autumn_asterism_star

.pw add .lf_autumn_asterism_star

################################################################

#
# See https://github.com/Stellarium/stellarium/blob/master/skycultures/modern/asterism_names.eng.fab
#

labelframe .lf_winter_asterism_star -text "Winter Asterism/Stars" -font $font_labelframe

frame .fr_301
label .l_301 -text "kl"
button .b_3011 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~(Plough)')"
button .b_3010 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~(Plough)')"
pack .l_301 -in .fr_301
pack .b_3010 .b_3011 -side left -expand 1 -fill both -in .fr_301

frame .fr_302
label .l_302 -text "kl5{"
button .b_3021 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~5times')"
button .b_3020 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~5times')"
pack .l_302 -in .fr_302
pack .b_3020 .b_3021 -side left -expand 1 -fill both -in .fr_302

frame .fr_303
label .l_303 -text "~̑Op"
button .b_3031 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Winter~Triangle')"
button .b_3030 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Winter~Triangle')"
pack .l_303 -in .fr_303
pack .b_3030 .b_3031 -side left -expand 1 -fill both -in .fr_303

frame .fr_304
label .l_304 -text "~̃_Ch"
button .b_3041 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Winter~Hexagon~(Winter~Circle)')"
button .b_3040 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Winter~Hexagon~(Winter~Circle)')"
pack .l_304 -in .fr_304
pack .b_3040 .b_3041 -side left -expand 1 -fill both -in .fr_304

#frame .fr_305
#label .l_305 -text "kɐ"
#button .b_3051 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Polaris')"
#button .b_3050 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Polaris')"
#pack .l_305 -in .fr_305
#pack .b_3050 .b_3051 -side left -expand 1 -fill both -in .fr_305

#frame .fr_306
#label .l_306 -text "΂" -font $font_small
#button .b_3061 -text " SET " -font $font_small -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Preiades')"
#button .b_3060 -text "Unset" -font $font_small -command "send_ssc_lines AsterismMgr.deselectAsterism('Preiades')"
#pack .l_306 -in .fr_306
#pack .b_3060 .b_3061 -side left -expand 1 -fill both -in .fr_306

frame .fr_307
label .l_307 -text "JVIy5{"
button .b_3071 -text " SET " -command "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Cassio~5times')"
button .b_3070 -text "Unset" -command "send_ssc_lines AsterismMgr.deselectAsterism('Cassio~5times')"
pack .l_307 -in .fr_307
pack .b_3070 .b_3071 -side left -expand 1 -fill both -in .fr_307

#

frame .fr_astar_lbl30 -height 12

#

frame .fr_astar_lbl31

label .l_astar_eachlbl32 -text "ʕ\" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
button .b_astar_off32 -text "SOff" -font $font_small -command "unset_star_label Polaris M45 Betelgeuse Rigel Sirius Procyon Pollux Castor Capella Aldebaran"
pack .l_astar_eachlbl32 .b_astar_off32 -side left -expand 1 -fill both -in .fr_astar_lbl31

#

frame .fr_astar_lbl32

frame .fr_321
label .l_321 -text "kɐ" -font $font_small
button .b_3211 -text "ON" -font $font_small -command "set_star_label Polaris"
button .b_3210 -text "Off" -font $font_small -command "unset_star_label Polaris"
pack .l_321 -in .fr_321
pack .b_3210 .b_3211 -side left -expand 1 -fill both -in .fr_321

frame .fr_322
label .l_322 -text "΂" -font $font_small
button .b_3221 -text "ON" -font $font_small -command "set_star_label M45"
button .b_3220 -text "Off" -font $font_small -command "unset_star_label M45"
pack .l_322 -in .fr_322
pack .b_3220 .b_3221 -side left -expand 1 -fill both -in .fr_322

pack .fr_321 .fr_322 -side left -expand 1 -fill both -in .fr_astar_lbl32

#

frame .fr_astar_lbl33

frame .fr_323
label .l_323 -text "xeMEX" -font $font_small
button .b_3231 -text "ON" -font $font_small -command "set_star_label Betelgeuse"
button .b_3230 -text "Off" -font $font_small -command "unset_star_label Betelgeuse"
pack .l_323 -in .fr_323
pack .b_3230 .b_3231 -side left -expand 1 -fill both -in .fr_323

frame .fr_324
label .l_324 -text "Q" -font $font_small
button .b_3241 -text "ON" -font $font_small -command "set_star_label Rigel"
button .b_3240 -text "Off" -font $font_small -command "unset_star_label Rigel"
pack .l_324 -in .fr_324
pack .b_3240 .b_3241 -side left -expand 1 -fill both -in .fr_324

pack .fr_323 .fr_324 -side left -expand 1 -fill both -in .fr_astar_lbl33

#

frame .fr_astar_lbl34

frame .fr_325
label .l_325 -text "VEX" -font $font_small
button .b_3251 -text "ON" -font $font_small -command "set_star_label Sirius"
button .b_3250 -text "Off" -font $font_small -command "unset_star_label Sirius"
pack .l_325 -in .fr_325
pack .b_3250 .b_3251 -side left -expand 1 -fill both -in .fr_325

frame .fr_326
label .l_326 -text "vLI" -font $font_small
button .b_3261 -text "ON" -font $font_small -command "set_star_label Procyon"
button .b_3260 -text "Off" -font $font_small -command "unset_star_label Procyon"
pack .l_326 -in .fr_326
pack .b_3260 .b_3261 -side left -expand 1 -fill both -in .fr_326

pack .fr_325 .fr_326 -side left -expand 1 -fill both -in .fr_astar_lbl34

#

frame .fr_astar_lbl35

frame .fr_327
label .l_327 -text "|bNX" -font $font_small
button .b_3271 -text "ON" -font $font_small -command "set_star_label Pollux"
button .b_3270 -text "Off" -font $font_small -command "unset_star_label Pollux"
pack .l_327 -in .fr_327
pack .b_3270 .b_3271 -side left -expand 1 -fill both -in .fr_327

frame .fr_328
label .l_328 -text "JXg" -font $font_small
button .b_3281 -text "ON" -font $font_small -command "set_star_label Castor"
button .b_3280 -text "Off" -font $font_small -command "unset_star_label Castor"
pack .l_328 -in .fr_328
pack .b_3280 .b_3281 -side left -expand 1 -fill both -in .fr_328

pack .fr_327 .fr_328 -side left -expand 1 -fill both -in .fr_astar_lbl35

#

frame .fr_astar_lbl36

frame .fr_329
label .l_329 -text "Jy" -font $font_small
button .b_3291 -text "ON" -font $font_small -command "set_star_label Capella"
button .b_3290 -text "Off" -font $font_small -command "unset_star_label Capella"
pack .l_329 -in .fr_329
pack .b_3290 .b_3291 -side left -expand 1 -fill both -in .fr_329

frame .fr_330
label .l_330 -text "Afo" -font $font_small
button .b_3301 -text "ON" -font $font_small -command "set_star_label Aldebaran"
button .b_3300 -text "Off" -font $font_small -command "unset_star_label Aldebaran"
pack .l_330 -in .fr_330
pack .b_3300 .b_3301 -side left -expand 1 -fill both -in .fr_330

pack .fr_329 .fr_330 -side left -expand 1 -fill both -in .fr_astar_lbl36

#

pack .fr_301 .fr_302 .fr_303 .fr_304 .fr_307 .fr_astar_lbl30 .fr_astar_lbl31 .fr_astar_lbl32 .fr_astar_lbl33 .fr_astar_lbl34 .fr_astar_lbl35 .fr_astar_lbl36 -expand 0 -fill both -in .lf_winter_asterism_star

.pw add .lf_winter_asterism_star

################################################################

labelframe .lf_solarsystem -text "Solar System" -font $font_labelframe

frame .fr_sol_disp
label .l_sol_disp -text "zn \"
button .b_sol_disp0 -text "Off" -command "unset_solarsystem"
button .b_sol_disp1 -text "ON" -command "exec_ssc_on_server setSolarSystem.ssc updateLightMoonPollution.ssc"
pack .l_sol_disp -in .fr_sol_disp
pack .b_sol_disp0 .b_sol_disp1 -side left -expand 1 -fill both -in .fr_sol_disp

frame .fr_sol_label
label .l_sol_label -text "zn V̖"
button .b_sol_label0 -text "Unset" -command "send_ssc_of_solsyslabels -"
entry .e_sol_label -width 4 -textvariable buffer_solsyslabels
set buffer_solsyslabels "0.0"
button .b_sol_label1 -text "<" -command "send_ssc_of_solsyslabels -0.5"
button .b_sol_label2 -text ">" -command "send_ssc_of_solsyslabels +0.5"
bind .e_sol_label <Return> "send_ssc_of_solsyslabels"
pack .l_sol_label -in .fr_sol_label
pack .b_sol_label0 -side left -fill both -in .fr_sol_label
pack .e_sol_label -expand 1 -side left -fill both -in .fr_sol_label
pack .b_sol_label1 .b_sol_label2 -side left -fill both -in .fr_sol_label

frame .fr_sol_porbit
label .l_sol_porbit -text "f O"
button .b_sol_porbit0 -text "Unset" -command "send_ssc_lines SolarSystem.setFlagOrbits(false)"
button .b_sol_porbit1 -text "SET" -command "send_or_exec_ssc SolarSystem.setOrbitColorStyle('one_color') setPlanetaryOrbits.ssc"
button .b_sol_porbit2 -text "F" -command "send_or_exec_ssc setPlanetaryOrbits.ssc SolarSystem.setOrbitColorStyle('major_planets_minor_types')"
button .b_sol_porbit3 -text "" -command "send_or_exec_ssc SolarSystem.setOrbitColorStyle('major_planets_minor_types') setMoonPlanetaryOrbits.ssc"
pack .l_sol_porbit -in .fr_sol_porbit
pack .b_sol_porbit0 .b_sol_porbit1 .b_sol_porbit2 .b_sol_porbit3 -side left -expand 1 -fill both -in .fr_sol_porbit

frame .fr_sol_eorbit
label .l_sol_eorbit -text "n O"
button .b_sol_eorbit0 -text "Unset" -command "exec_ssc_on_server unsetEarthOrbitLabels.ssc unsetEarthOrbit.ssc"
button .b_sol_eorbit1 -text "SET" -command "exec_ssc_on_server unsetEarthOrbitLabels.ssc setEarthOrbit.ssc"
button .b_sol_eorbit2 -text "DATE" -command "exec_ssc_on_server setEarthOrbit.ssc setEarthOrbitLabels.ssc"
pack .l_sol_eorbit -in .fr_sol_eorbit
pack .b_sol_eorbit0 .b_sol_eorbit1 .b_sol_eorbit2 -side left -expand 1 -fill both -in .fr_sol_eorbit

frame .fr_sol_pscale
label .l_sol_pscale -text "V \TCY"
button .b_sol_pscale0 -text "f1x" -command "send_ssc_lines SolarSystem.setPlanetScale(1) SolarSystem.setFlagPlanetScale(false)"
button .b_sol_pscale1 -text "400x" -command "send_ssc_lines SolarSystem.setFlagPlanetScale(true) SolarSystem.setPlanetScale(400)"
button .b_sol_pscale2 -text "999x" -command "send_ssc_lines SolarSystem.setFlagPlanetScale(true) SolarSystem.setPlanetScale(999)"
pack .l_sol_pscale -in .fr_sol_pscale
pack .b_sol_pscale0 .b_sol_pscale1 .b_sol_pscale2 -side left -expand 1 -fill both -in .fr_sol_pscale

frame .fr_sol_mscale
#label .l_sol_mscale -text " \TCY" -font $font_small
button .b_sol_mscale0 -text "1x" -font $font_small -command "send_ssc_lines SolarSystem.setMoonScale(1) SolarSystem.setFlagMoonScale(false)"
button .b_sol_mscale1 -text "l" -font $font_small -command "exec_ssc_on_server localInitMoonScale.ssc"
button .b_sol_mscale2 -text "32x" -font $font_small -command "send_ssc_lines SolarSystem.setFlagMoonScale(true) SolarSystem.setMoonScale(32)"
#pack .l_sol_mscale -in .fr_sol_mscale
pack .b_sol_mscale0 .b_sol_mscale1 .b_sol_mscale2 -side left -expand 1 -fill both -in .fr_sol_mscale

frame .fr_sol_sscale
#label .l_sol_sscale -text "z \TCY" -font $font_small
button .b_sol_sscale0 -text "z1x" -font $font_small -command "send_ssc_lines SolarSystem.setSunScale(1) SolarSystem.setFlagSunScale(false)"
button .b_sol_sscale1 -text "l" -font $font_small -command "exec_ssc_on_server localInitSunScale.ssc"
button .b_sol_sscale2 -text "32x" -font $font_small -command "send_ssc_lines SolarSystem.setFlagSunScale(true) SolarSystem.setSunScale(32)"
#pack .l_sol_sscale -in .fr_sol_sscale
pack .b_sol_sscale0 .b_sol_sscale1 .b_sol_sscale2 -side left -expand 1 -fill both -in .fr_sol_sscale

#

frame .fr_sol_lbl00

label .l_sol_eachlabel -text " ʕ\" -foreground $label_inverted_foreground -background $label_inverted_background
#button .b_sol_label000_0 -text "Off" -command "unset_star_label Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto"
pack .l_sol_eachlabel -side left -expand 1 -fill both -in .fr_sol_lbl00

#

frame .fr_sol_lbl01

frame .fr_sol_label000
label .l_sol_label000 -text "@S@" -font $font_small
button .b_sol_label000_1 -text "ON" -font $font_small -command "set_star_label Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto"
button .b_sol_label000_0 -text "Off" -font $font_small -command "unset_star_label Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto"
pack .l_sol_label000 -in .fr_sol_label000
pack .b_sol_label000_0 .b_sol_label000_1 -side left -expand 1 -fill both -in .fr_sol_label000

frame .fr_sol_label001
label .l_sol_label001 -text "z" -font $font_small
button .b_sol_label001_1 -text "ON" -font $font_small -command "set_star_label Sun"
button .b_sol_label001_0 -text "Off" -font $font_small -command "unset_star_label Sun"
pack .l_sol_label001 -in .fr_sol_label001
pack .b_sol_label001_0 .b_sol_label001_1 -side left -expand 1 -fill both -in .fr_sol_label001

pack .fr_sol_label000 .fr_sol_label001 -side left -expand 1 -fill both -in .fr_sol_lbl01

#

frame .fr_sol_lbl02

frame .fr_sol_label002
label .l_sol_label002 -text "" -font $font_small
button .b_sol_label002_1 -text "ON" -font $font_small -command "set_star_label Moon"
button .b_sol_label002_0 -text "Off" -font $font_small -command "unset_star_label Moon"
pack .l_sol_label002 -in .fr_sol_label002
pack .b_sol_label002_0 .b_sol_label002_1 -side left -expand 1 -fill both -in .fr_sol_label002

frame .fr_sol_label003
label .l_sol_label003 -text "" -font $font_small
button .b_sol_label003_1 -text "ON" -font $font_small -command "set_star_label Mercury"
button .b_sol_label003_0 -text "Off" -font $font_small -command "unset_star_label Mercury"
pack .l_sol_label003 -in .fr_sol_label003
pack .b_sol_label003_0 .b_sol_label003_1 -side left -expand 1 -fill both -in .fr_sol_label003

pack .fr_sol_label002 .fr_sol_label003 -side left -expand 1 -fill both -in .fr_sol_lbl02

#

frame .fr_sol_lbl03

frame .fr_sol_label004
label .l_sol_label004 -text "" -font $font_small
button .b_sol_label004_1 -text "ON" -font $font_small -command "set_star_label Venus"
button .b_sol_label004_0 -text "Off" -font $font_small -command "unset_star_label Venus"
pack .l_sol_label004 -in .fr_sol_label004
pack .b_sol_label004_0 .b_sol_label004_1 -side left -expand 1 -fill both -in .fr_sol_label004

frame .fr_sol_label005
label .l_sol_label005 -text "ΐ" -font $font_small
button .b_sol_label005_1 -text "ON" -font $font_small -command "set_star_label Mars"
button .b_sol_label005_0 -text "Off" -font $font_small -command "unset_star_label Mars"
pack .l_sol_label005 -in .fr_sol_label005
pack .b_sol_label005_0 .b_sol_label005_1 -side left -expand 1 -fill both -in .fr_sol_label005

pack .fr_sol_label004 .fr_sol_label005 -side left -expand 1 -fill both -in .fr_sol_lbl03

#

frame .fr_sol_lbl04

frame .fr_sol_label006
label .l_sol_label006 -text "ؐ" -font $font_small
button .b_sol_label006_1 -text "ON" -font $font_small -command "set_star_label Jupiter"
button .b_sol_label006_0 -text "Off" -font $font_small -command "unset_star_label Jupiter"
pack .l_sol_label006 -in .fr_sol_label006
pack .b_sol_label006_0 .b_sol_label006_1 -side left -expand 1 -fill both -in .fr_sol_label006

frame .fr_sol_label007
label .l_sol_label007 -text "y" -font $font_small
button .b_sol_label007_1 -text "ON" -font $font_small -command "set_star_label Saturn"
button .b_sol_label007_0 -text "Off" -font $font_small -command "unset_star_label Saturn"
pack .l_sol_label007 -in .fr_sol_label007
pack .b_sol_label007_0 .b_sol_label007_1 -side left -expand 1 -fill both -in .fr_sol_label007

pack .fr_sol_label006 .fr_sol_label007 -side left -expand 1 -fill both -in .fr_sol_lbl04

#

frame .fr_sol_lbl05

frame .fr_sol_label008
label .l_sol_label008 -text "V" -font $font_small
button .b_sol_label008_1 -text "ON" -font $font_small -command "set_star_label Uranus"
button .b_sol_label008_0 -text "Off" -font $font_small -command "unset_star_label Uranus"
pack .l_sol_label008 -in .fr_sol_label008
pack .b_sol_label008_0 .b_sol_label008_1 -side left -expand 1 -fill both -in .fr_sol_label008

frame .fr_sol_label009
label .l_sol_label009 -text "C" -font $font_small
button .b_sol_label009_1 -text "ON" -font $font_small -command "set_star_label Neptune"
button .b_sol_label009_0 -text "Off" -font $font_small -command "unset_star_label Neptune"
pack .l_sol_label009 -in .fr_sol_label009
pack .b_sol_label009_0 .b_sol_label009_1 -side left -expand 1 -fill both -in .fr_sol_label009

pack .fr_sol_label008 .fr_sol_label009 -side left -expand 1 -fill both -in .fr_sol_lbl05

#

pack .fr_sol_disp .fr_sol_label .fr_sol_porbit .fr_sol_eorbit .fr_sol_pscale .fr_sol_mscale .fr_sol_sscale -expand 0 -fill both -in .lf_solarsystem
pack .fr_sol_lbl00 .fr_sol_lbl01 .fr_sol_lbl02 .fr_sol_lbl03 .fr_sol_lbl04 .fr_sol_lbl05 -expand 0 -fill both -in .lf_solarsystem

.pw add .lf_solarsystem


################################################################

labelframe .lf_dso_p0 -text "Objects" -font $font_labelframe

frame .fr_nebula
label .l_dso_list -text "ꗗ" -foreground $label_inverted_foreground -background $label_inverted_background
label .l_nebula -text "_Ec \"
button .b_nebula1 -text "ON" -command "send_ssc_lines NebulaMgr.setFlagHints(true) NebulaMgr.setFlagShow(true)"
button .b_nebula0 -text "Off" -command "send_ssc_lines NebulaMgr.setFlagHints(false) NebulaMgr.setFlagShow(false)"
pack .l_dso_list .l_nebula -fill x -in .fr_nebula
pack .b_nebula0 .b_nebula1 -side left -expand 1 -fill both -in .fr_nebula

frame .fr_dsotype
label .l_dsotype -text "J^OI"
button .b_dsotype1 -text "NGC" -command "send_ssc_lines NebulaMgr.setCatalogFilters(1)"
button .b_dsotype2 -text "IC" -command  "send_ssc_lines NebulaMgr.setCatalogFilters(2)"
button .b_dsotype4 -text " M " -command "send_ssc_lines NebulaMgr.setCatalogFilters(4)"
pack .l_dsotype -in .fr_dsotype
pack .b_dsotype4 .b_dsotype1 .b_dsotype2 -side left -expand 1 -fill both -in .fr_dsotype

frame .fr_dsohint
label .l_dsohint -text "qg"
button .b_dsohint0 -text "0" -command "send_ssc_of_dsohints -"
entry .e_dsohint -width 5 -textvariable buffer_dsohints
set buffer_dsohints "0.0"
button .b_dsohint1 -text "<" -command "send_ssc_of_dsohints -0.5"
button .b_dsohint2 -text ">" -command "send_ssc_of_dsohints +0.5"
bind .e_dsohint <Return> "send_ssc_of_dsohints"
pack .l_dsohint -in .fr_dsohint
pack .b_dsohint0 -side left -fill both -in .fr_dsohint
pack .e_dsohint -expand 1 -side left -fill both -in .fr_dsohint
pack .b_dsohint1 .b_dsohint2 -side left -fill both -in .fr_dsohint

frame .fr_dsolbl
label .l_dsolbl -text ""
button .b_dsolbl0 -text "0" -command "send_ssc_of_dsolabels -"
entry .e_dsolbl -width 5 -textvariable buffer_dsolabels
set buffer_dsolabels "0.0"
button .b_dsolbl1 -text "<" -command "send_ssc_of_dsolabels -0.5"
button .b_dsolbl2 -text ">" -command "send_ssc_of_dsolabels +0.5"
bind .e_dsolbl <Return> "send_ssc_of_dsolabels"
pack .l_dsolbl -in .fr_dsolbl
pack .b_dsolbl0 -side left -fill both -in .fr_dsolbl
pack .e_dsolbl -expand 1 -side left -fill both -in .fr_dsolbl
pack .b_dsolbl1 .b_dsolbl2 -side left -fill both -in .fr_dsolbl

frame .fr_dsoinit_p0
label .l_dso_each -text "" -foreground $label_inverted_foreground -background $label_inverted_background
label .l_dsoinit_p0 -text "Zbg"
button .b_dsoinit_p0m -text "}[J[" -command "send_ssc_lines core.setSelectedObjectMarkerVisible(false)"
button .b_dsoinit_p0v -text "VIEW" -command "exec_ssc_on_server localInitView.ssc"
pack .l_dso_each .l_dsoinit_p0 -fill x -in .fr_dsoinit_p0
if { $flag_dome_mode != 0 } {
  pack .b_dsoinit_p0m -side left -expand 1 -fill both -in .fr_dsoinit_p0
}
pack .b_dsoinit_p0v -side left -expand 1 -fill both -in .fr_dsoinit_p0

frame .fr_dsomarker
label .l_dsomarker -text "V̑I"
entry .e_dsomarker -width 8 -textvariable buffer_selected_dsoname
bind .e_dsomarker <Return> "select_a_dso"
button .b_dsomarker -text "SET" -command "select_a_dso"
pack .l_dsomarker -in .fr_dsomarker
pack .e_dsomarker .b_dsomarker -side left -expand 1 -fill both -in .fr_dsomarker

frame .fr_dsozoom
label .l_dsozoom -text "ړ&Y[(x)"
frame .fr_dsozoom_b0
button .b_dsozoom32 -text "32" -command "exec_ssc_on_server moveZoomToSelectedFov32.ssc"
button .b_dsozoom16 -text "16" -command "exec_ssc_on_server moveZoomToSelectedFov16.ssc"
button .b_dsozoom8 -text "8" -command "exec_ssc_on_server moveZoomToSelectedFov8.ssc"
button .b_dsozoom4 -text "4" -command "exec_ssc_on_server moveZoomToSelectedFov4.ssc"
button .b_dsozoom2 -text "2" -command "exec_ssc_on_server moveZoomToSelectedFov2.ssc"
button .b_dsozoom1 -text "1" -command "exec_ssc_on_server moveZoomToSelectedFov1.ssc"
pack .b_dsozoom32 .b_dsozoom16 .b_dsozoom8 .b_dsozoom4 .b_dsozoom2 .b_dsozoom1 -side left -expand 1 -fill both -in .fr_dsozoom_b0
frame .fr_dsozoom_b1
button .b_dsozoom18m -text "0.3" -font $font_small -command "exec_ssc_on_server moveZoomToSelectedFov0.3.ssc"
button .b_dsozoom6m -text "0.1" -font $font_small -command "exec_ssc_on_server moveZoomToSelectedFov0.1.ssc"
button .b_dsozoom2m -text "0.03" -font $font_small -command "exec_ssc_on_server moveZoomToSelectedFov0.03.ssc"
button .b_dsozoom06m -text "0.01" -font $font_small -command "exec_ssc_on_server moveZoomToSelectedFov0.01.ssc"
pack .b_dsozoom18m .b_dsozoom6m .b_dsozoom2m .b_dsozoom06m -side left -expand 1 -fill both -in .fr_dsozoom_b1
pack .l_dsozoom -in .fr_dsozoom
pack .fr_dsozoom_b0 .fr_dsozoom_b1 -expand 1 -fill both -in .fr_dsozoom

set buffer_dso_fov_circular_size "1.0000"
frame .fr_dsofovc

frame .fr_dsofovc_b0
label .l_dsofovc -text "~(x)" -font $font_small
button .b_dso_ed_fovc -text "ҏW" -font $font_small -command "exec $text_editor list_fov-circular.txt &"
pack .l_dsofovc -expand 1 -side left -fill x -in .fr_dsofovc_b0
pack .b_dso_ed_fovc -side left -fill x -in .fr_dsofovc_b0

frame .fr_dsofovc_b1
button .b_dsofovc0 -text "Off" -font $font_small -command "send_ssc_lines SpecialMarkersMgr.setFlagFOVCircularMarker(false)"
entry .e_dsofovc -width 5 -textvariable buffer_dso_fov_circular_size -font $font_small
button .b_dsofovc1 -text "ON" -font $font_small -command "send_ssc_of_dso_fov_circular_marker"
bind .e_dsofovc <Return> "send_ssc_of_dso_fov_circular_marker"
pack .b_dsofovc0 -side left -fill x -in .fr_dsofovc_b1
pack .e_dsofovc -expand 1 -side left -fill both -in .fr_dsofovc_b1
pack .b_dsofovc1 -side left -fill x -in .fr_dsofovc_b1

#frame .fr_dsofovc_b2
#button .b_dsofovc_t0 -text "6x" -font $font_small -command "send_ssc_of_dso_fov_circular_marker 8.0"
#button .b_dsofovc_t1 -text "45x" -font $font_small -command "send_ssc_of_dso_fov_circular_marker 1.0"
#button .b_dsofovc_t2 -text "100x" -font $font_small -command "send_ssc_of_dso_fov_circular_marker 0.45"
#button .b_dsofovc_t3 -text "200x" -font $font_small -command "send_ssc_of_dso_fov_circular_marker 0.225"
#pack .b_dsofovc_t0 .b_dsofovc_t1 .b_dsofovc_t2 .b_dsofovc_t3 -expand 1 -side left -fill x -in .fr_dsofovc_b2

pack .fr_dsofovc_b0 .fr_dsofovc_b1 -expand 1 -fill x -in .fr_dsofovc

pack .fr_nebula .fr_dsotype .fr_dsohint .fr_dsolbl .fr_dsoinit_p0 .fr_dsomarker .fr_dsozoom .fr_dsofovc -expand 0 -fill both -in .lf_dso_p0

build_dso_buttons "list_fov-circular.txt" "send_ssc_of_dso_fov_circular_marker" 4 "fovc" "p0"

.pw add .lf_dso_p0


################################################################

labelframe .lf_dso_p1 -text "Objects 1" -font $font_labelframe

frame .fr_dso_spring
label .l_dso_spring -text "t"
button .b_dso_ed_spring -text "ҏW" -font $font_small -command "exec $text_editor objects_spring.txt &"
pack .l_dso_spring -side left -expand 1 -fill x -in .fr_dso_spring
pack .b_dso_ed_spring -side left -expand 0 -fill x -in .fr_dso_spring
pack .fr_dso_spring -expand 0 -fill x -in .lf_dso_p1

build_dso_buttons "objects_spring.txt" "select_a_dso" 3 "spring" "p1"

frame .fr_dso_p1_spc00 -height 3
pack .fr_dso_p1_spc00 -expand 0 -fill x -in .lf_dso_p1

frame .fr_dso_summer
label .l_dso_summer -text ""
button .b_dso_ed_summer -text "ҏW" -font $font_small -command "exec $text_editor objects_summer.txt &"
pack .l_dso_summer -side left -expand 1 -fill x -in .fr_dso_summer
pack .b_dso_ed_summer -side left -expand 0 -fill x -in .fr_dso_summer
pack .fr_dso_summer -expand 0 -fill x -in .lf_dso_p1

build_dso_buttons "objects_summer.txt" "select_a_dso" 3 "summer" "p1"

frame .fr_dso_p1_spc01 -height 3
pack .fr_dso_p1_spc01 -expand 0 -fill x -in .lf_dso_p1

frame .fr_dso_etc1
label .l_dso_etc1 -text "etc.1"
button .b_dso_ed_etc1 -text "ҏW" -font $font_small -command "exec $text_editor objects_etc1.txt &"
pack .l_dso_etc1 -side left -expand 1 -fill x -in .fr_dso_etc1
pack .b_dso_ed_etc1 -side left -expand 0 -fill x -in .fr_dso_etc1
pack .fr_dso_etc1 -expand 0 -fill x -in .lf_dso_p1

build_dso_buttons "objects_etc1.txt" "select_a_dso" 3 "etc1_" "p1"

.pw add .lf_dso_p1


################################################################

labelframe .lf_dso_p2 -text "Objects 2" -font $font_labelframe

frame .fr_dso_autumn
label .l_dso_autumn -text "H"
button .b_dso_ed_autumn -text "ҏW" -font $font_small -command "exec $text_editor objects_autumn.txt &"
pack .l_dso_autumn -side left -expand 1 -fill x -in .fr_dso_autumn
pack .b_dso_ed_autumn -side left -expand 0 -fill x -in .fr_dso_autumn
pack .fr_dso_autumn -expand 0 -fill x -in .lf_dso_p2

build_dso_buttons "objects_autumn.txt" "select_a_dso" 3 "autumn" "p2"

frame .fr_dso_p2_spc00 -height 3
pack .fr_dso_p2_spc00 -expand 0 -fill x -in .lf_dso_p2

frame .fr_dso_winter
label .l_dso_winter -text "~"
button .b_dso_ed_winter -text "ҏW" -font $font_small -command "exec $text_editor objects_winter.txt &"
pack .l_dso_winter -side left -expand 1 -fill x -in .fr_dso_winter
pack .b_dso_ed_winter -side left -expand 0 -fill x -in .fr_dso_winter
pack .fr_dso_winter -expand 0 -fill x -in .lf_dso_p2

build_dso_buttons "objects_winter.txt" "select_a_dso" 3 "winter" "p2"

frame .fr_dso_p2_spc01 -height 3
pack .fr_dso_p2_spc01 -expand 0 -fill x -in .lf_dso_p2

frame .fr_dso_etc2
label .l_dso_etc2 -text "etc.2"
button .b_dso_ed_etc2 -text "ҏW" -font $font_small -command "exec $text_editor objects_etc2.txt &"
pack .l_dso_etc2 -side left -expand 1 -fill x -in .fr_dso_etc2
pack .b_dso_ed_etc2 -side left -expand 0 -fill x -in .fr_dso_etc2
pack .fr_dso_etc2 -expand 0 -fill x -in .lf_dso_p2

build_dso_buttons "objects_etc2.txt" "select_a_dso" 3 "etc2_" "p2"

.pw add .lf_dso_p2


################################################################

labelframe .lf_gridlines -text "Grid Lines" -font $font_labelframe

frame .fr_grid_disp
label .l_grid_disp -text "W \"
button .b_grid_disp1 -text "ON" -command "exec_ssc_on_server setFlagGridLines.ssc"
button .b_grid_disp0 -text "Off" -command "exec_ssc_on_server unsetFlagGridLines.ssc"
pack .l_grid_disp -in .fr_grid_disp
pack .b_grid_disp0 .b_grid_disp1 -side left -expand 1 -fill both -in .fr_grid_disp

frame .fr_grid_azim
label .l_grid_azim -text "x"
button .b_grid_azim1 -text " SET " -command "send_ssc_lines GridLinesMgr.setFlagAzimuthalGrid(true)"
button .b_grid_azim0 -text "Unset" -command "send_ssc_lines GridLinesMgr.setFlagAzimuthalGrid(false)"
pack .l_grid_azim -in .fr_grid_azim
pack .b_grid_azim0 .b_grid_azim1 -side left -expand 1 -fill both -in .fr_grid_azim

frame .fr_grid_eclip
label .l_grid_eclip -text "o/"
button .b_grid_eclip1 -text " SET " -command "send_ssc_lines GridLinesMgr.setFlagEclipticGrid(true)"
button .b_grid_eclip0 -text "Unset" -command "send_ssc_lines GridLinesMgr.setFlagEclipticGrid(false)"
pack .l_grid_eclip -in .fr_grid_eclip
pack .b_grid_eclip0 .b_grid_eclip1 -side left -expand 1 -fill both -in .fr_grid_eclip

frame .fr_grid_eclipl
label .l_grid_eclipl -text ""
button .b_grid_eclipl2 -text "DATE" -command "send_or_exec_ssc GridLinesMgr.setFlagEclipticLine(true) setEclipticDateLabels.ssc"
button .b_grid_eclipl1 -text "SET" -command "send_or_exec_ssc unsetEclipticDateLabels.ssc GridLinesMgr.setFlagEclipticLine(true)"
button .b_grid_eclipl0 -text "Unset" -command "send_or_exec_ssc unsetEclipticDateLabels.ssc GridLinesMgr.setFlagEclipticLine(false)"
pack .l_grid_eclipl -in .fr_grid_eclipl
pack .b_grid_eclipl0 .b_grid_eclipl1 .b_grid_eclipl2 -side left -expand 1 -fill both -in .fr_grid_eclipl

frame .fr_grid_equat
label .l_grid_equat -text "Ԍo/Ԉ"
button .b_grid_equat1 -text " SET " -command "send_ssc_lines GridLinesMgr.setFlagEquatorGrid(true)"
button .b_grid_equat0 -text "Unset" -command "send_ssc_lines GridLinesMgr.setFlagEquatorGrid(false)"
pack .l_grid_equat -in .fr_grid_equat
pack .b_grid_equat0 .b_grid_equat1 -side left -expand 1 -fill both -in .fr_grid_equat

frame .fr_grid_equatl
label .l_grid_equatl -text "V̐ԓ"
button .b_grid_equatl1 -text " SET " -command "send_ssc_lines GridLinesMgr.setFlagEquatorLine(true)"
button .b_grid_equatl0 -text "Unset" -command "send_ssc_lines GridLinesMgr.setFlagEquatorLine(false)"
pack .l_grid_equatl -in .fr_grid_equatl
pack .b_grid_equatl0 .b_grid_equatl1 -side left -expand 1 -fill both -in .fr_grid_equatl

frame .fr_grid_sundnl
label .l_grid_sundnl -text "z ^"
button .b_grid_sundnl1 -text " SET " -command "exec_ssc_on_server setSunDiurnalMotion.ssc"
button .b_grid_sundnl0 -text "Unset" -command "exec_ssc_on_server unsetSunDiurnalMotion.ssc"
pack .l_grid_sundnl -in .fr_grid_sundnl
pack .b_grid_sundnl0 .b_grid_sundnl1 -side left -expand 1 -fill both -in .fr_grid_sundnl

frame .fr_grid_galact
label .l_grid_galact -text "o/"
button .b_grid_galact1 -text " SET " -command "send_ssc_lines GridLinesMgr.setFlagGalacticGrid(true)"
button .b_grid_galact0 -text "Unset" -command "send_ssc_lines GridLinesMgr.setFlagGalacticGrid(false)"
pack .l_grid_galact -in .fr_grid_galact
pack .b_grid_galact0 .b_grid_galact1 -side left -expand 1 -fill both -in .fr_grid_galact

frame .fr_grid_galactl
label .l_grid_galactl -text "͖"
button .b_grid_galactl1 -text " SET " -command "send_ssc_lines GridLinesMgr.setFlagGalacticEquatorLine(true)"
button .b_grid_galactl0 -text "Unset" -command "send_ssc_lines GridLinesMgr.setFlagGalacticEquatorLine(false)"
pack .l_grid_galactl -in .fr_grid_galactl
pack .b_grid_galactl0 .b_grid_galactl1 -side left -expand 1 -fill both -in .fr_grid_galactl

pack .fr_grid_disp .fr_grid_azim .fr_grid_eclip .fr_grid_eclipl .fr_grid_equat .fr_grid_equatl .fr_grid_sundnl .fr_grid_galact .fr_grid_galactl -expand 0 -fill both -in .lf_gridlines

.pw add .lf_gridlines


################################################################

labelframe .lf_config -text "Configuration" -font $font_labelframe

set buffer_projection_mode $asarr_initial_settings(ProjectionMode)
set pjmod {
  "ProjectionPerspective"
  "ProjectionEqualArea"
  "ProjectionStereographic"
  "ProjectionFisheye"
  "ProjectionHammer"
  "ProjectionCylinder"
  "ProjectionMercator"
  "ProjectionOrthographic"
  "ProjectionSinusoidal"
  "ProjectionMiller"
}
frame .fr_mnt_pjmod
label .l_mnt_pjmod -text "Projection Mode" -font $font_small
ttk::combobox .cb_mnt_pjmod -textvariable buffer_projection_mode -value $pjmod -font $font_small -state readonly
bind .cb_mnt_pjmod <<ComboboxSelected>> { send_ssc_of_projection_mode }
pack .l_mnt_pjmod -in .fr_mnt_pjmod
pack .cb_mnt_pjmod -side left -expand 1 -fill both -in .fr_mnt_pjmod

frame .fr_mnt_gui
label .l_mnt_gui -text "GUI of Main Window" -font $font_small
button .b_mnt_gui0 -text "Off" -font $font_small -command "send_ssc_lines StelMovementMgr.setFlagEnableMouseZooming(false) StelMovementMgr.setFlagEnableMouseNavigation(false) core.setSelectedObjectMarkerVisible(false) core.setGuiVisible(false) core.setSelectedObjectInfo('None')"
button .b_mnt_gui1 -text "ON" -font $font_small -command "send_ssc_lines StelMovementMgr.setFlagEnableMouseZooming(true) StelMovementMgr.setFlagEnableMouseNavigation(true) core.setSelectedObjectMarkerVisible(true) core.setGuiVisible(true) core.setSelectedObjectInfo('AllInfo')"
pack .l_mnt_gui -in .fr_mnt_gui
pack .b_mnt_gui0 .b_mnt_gui1 -side left -expand 1 -fill both -in .fr_mnt_gui

#frame .fr_mnt_msz
#label .l_mnt_msz -text "Mouse Zoom & Navi" -font $font_small
#button .b_mnt_msz0 -text "Off" -font $font_small -command "send_ssc_lines StelMovementMgr.setFlagEnableMouseZooming(false) StelMovementMgr.setFlagEnableMouseNavigation(false)"
#button .b_mnt_msz1 -text "ON" -font $font_small -command "send_ssc_lines StelMovementMgr.setFlagEnableMouseZooming(true) StelMovementMgr.setFlagEnableMouseNavigation(true)"
#pack .l_mnt_msz -in .fr_mnt_msz
#pack .b_mnt_msz0 .b_mnt_msz1 -side left -expand 1 -fill both -in .fr_mnt_msz

set buffer_display_gamma $asarr_initial_settings(DisplayGamma)
frame .fr_mnt_gamma
label .l_mnt_gamma -text "Display Gamma" -font $font_small
button .b_mnt_gamma0 -text "l" -font $font_small -command "send_ssc_of_display_gamma -"
entry .e_mnt_gamma -width 7 -textvariable buffer_display_gamma -font $font_small
button .b_mnt_gamma1 -text "SET" -font $font_small -command "send_ssc_of_display_gamma"
bind .e_mnt_gamma <Return> "send_ssc_of_display_gamma"
pack .l_mnt_gamma -in .fr_mnt_gamma
pack .b_mnt_gamma0 -side left -fill both -in .fr_mnt_gamma
pack .e_mnt_gamma -expand 1 -side left -fill both -in .fr_mnt_gamma
pack .b_mnt_gamma1 -side left -fill both -in .fr_mnt_gamma

set buffer_disk_viewport $asarr_initial_settings(DiskViewport)
frame .fr_mnt_diskview
label .l_mnt_diskview -text "Disk Viewport" -font $font_small
button .b_mnt_diskview0 -text "l" -font $font_small -command "send_ssc_of_disk_viewport -"
button .b_mnt_diskview1 -text "Off" -font $font_small -command "send_ssc_of_disk_viewport false"
button .b_mnt_diskview2 -text "ON" -font $font_small -command "send_ssc_of_disk_viewport true"
pack .l_mnt_diskview -in .fr_mnt_diskview
pack .b_mnt_diskview0 -side left -fill both -in .fr_mnt_diskview
pack .b_mnt_diskview1 .b_mnt_diskview2 -expand 1 -side left -fill both -in .fr_mnt_diskview

set buffer_viewport_x $asarr_initial_settings(ViewportX)
frame .fr_mnt_vpx
label .l_mnt_vpx -text "Viewport X" -font $font_small
button .b_mnt_vpx0 -text "l" -font $font_small -command "send_ssc_of_viewport_x -"
entry .e_mnt_vpx -width 7 -textvariable buffer_viewport_x -font $font_small
button .b_mnt_vpx1 -text "SET" -font $font_small -command "send_ssc_of_viewport_x"
bind .e_mnt_vpx <Return> "send_ssc_of_viewport_x"
pack .l_mnt_vpx -in .fr_mnt_vpx
pack .b_mnt_vpx0 -side left -fill both -in .fr_mnt_vpx
pack .e_mnt_vpx -expand 1 -side left -fill both -in .fr_mnt_vpx
pack .b_mnt_vpx1 -side left -fill both -in .fr_mnt_vpx

set buffer_view_fov $asarr_initial_settings(ViewFov)
frame .fr_mnt_fov
label .l_mnt_fov -text "View FOV" -font $font_small
button .b_mnt_fov0 -text "l" -font $font_small -command "send_ssc_of_view_fov -"
entry .e_mnt_fov -width 7 -textvariable buffer_view_fov -font $font_small
button .b_mnt_fov1 -text "SET" -font $font_small -command "send_ssc_of_view_fov"
bind .e_mnt_fov <Return> "send_ssc_of_view_fov"
pack .l_mnt_fov -in .fr_mnt_fov
pack .b_mnt_fov0 -side left -fill both -in .fr_mnt_fov
pack .e_mnt_fov -expand 1 -side left -fill both -in .fr_mnt_fov
pack .b_mnt_fov1 -side left -fill both -in .fr_mnt_fov

set buffer_viewport_y $asarr_initial_settings(ViewportY)
frame .fr_mnt_vpy
label .l_mnt_vpy -text "Viewport Y" -font $font_small
button .b_mnt_vpy0 -text "l" -font $font_small -command "send_ssc_of_viewport_y -"
entry .e_mnt_vpy -width 7 -textvariable buffer_viewport_y -font $font_small
button .b_mnt_vpy1 -text "SET" -font $font_small -command "send_ssc_of_viewport_y"
bind .e_mnt_vpy <Return> "send_ssc_of_viewport_y"
pack .l_mnt_vpy -in .fr_mnt_vpy
pack .b_mnt_vpy0 -side left -fill both -in .fr_mnt_vpy
pack .e_mnt_vpy -expand 1 -side left -fill both -in .fr_mnt_vpy
pack .b_mnt_vpy1 -side left -fill both -in .fr_mnt_vpy

set buffer_view_altitude $asarr_initial_settings(ViewAltitude)
frame .fr_mnt_altitude
label .l_mnt_altitude -text "View Altitude" -font $font_small
button .b_mnt_altitude0 -text "l" -font $font_small -command "send_ssc_of_view_altitude -"
entry .e_mnt_altitude -width 7 -textvariable buffer_view_altitude -font $font_small
button .b_mnt_altitude1 -text "SET" -font $font_small -command "send_ssc_of_view_altitude"
bind .e_mnt_altitude <Return> "send_ssc_of_view_altitude"
pack .l_mnt_altitude -in .fr_mnt_altitude
pack .b_mnt_altitude0 -side left -fill both -in .fr_mnt_altitude
pack .e_mnt_altitude -expand 1 -side left -fill both -in .fr_mnt_altitude
pack .b_mnt_altitude1 -side left -fill both -in .fr_mnt_altitude

set buffer_view_azimuth $asarr_initial_settings(ViewAzimuth)
frame .fr_mnt_azimuth
label .l_mnt_azimuth -text "View Azimuth" -font $font_small
button .b_mnt_azimuth0 -text "l" -font $font_small -command "send_ssc_of_view_azimuth -"
entry .e_mnt_azimuth -width 7 -textvariable buffer_view_azimuth -font $font_small
button .b_mnt_azimuth1 -text "SET" -font $font_small -command "send_ssc_of_view_azimuth"
bind .e_mnt_azimuth <Return> "send_ssc_of_view_azimuth"
pack .l_mnt_azimuth -in .fr_mnt_azimuth
pack .b_mnt_azimuth0 -side left -fill both -in .fr_mnt_azimuth
pack .e_mnt_azimuth -expand 1 -side left -fill both -in .fr_mnt_azimuth
pack .b_mnt_azimuth1 -side left -fill both -in .fr_mnt_azimuth

frame .fr_mnt_edlbl

label .l_mnt_edlbl -text "ݒt@CҏW" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
pack .l_mnt_edlbl -side left -expand 1 -fill both -in .fr_mnt_edlbl

button .b_mnt_ed00 -text "localConfig.ssc" -font $font_filename -command "exec -ignorestderr $text_editor localConfig.ssc &"
button .b_mnt_ed01 -text "localInitView.ssc" -font $font_filename -command "exec -ignorestderr $text_editor localInitView.ssc &"
button .b_mnt_ed02 -text "localInitLocation.ssc" -font $font_filename -command "exec -ignorestderr $text_editor localInitLocation.ssc &"
button .b_mnt_ed03 -text "localInitSettings.ssc" -font $font_filename -command "exec -ignorestderr $text_editor localInitSettings.ssc &"
button .b_mnt_ed04 -text "localInitSunScale.ssc" -font $font_filename -command "exec -ignorestderr $text_editor localInitSunScale.ssc &"
button .b_mnt_ed05 -text "localInitMoonScale.ssc" -font $font_filename -command "exec -ignorestderr $text_editor localInitMoonScale.ssc &"
button .b_mnt_ed06 -text "exec_date-monitor.bat" -font $font_filename -command "exec -ignorestderr $text_editor exec_date-monitor.bat &"

pack .fr_mnt_pjmod .fr_mnt_gui .fr_mnt_gamma .fr_mnt_diskview .fr_mnt_fov .fr_mnt_vpx .fr_mnt_vpy .fr_mnt_altitude .fr_mnt_azimuth .fr_mnt_edlbl -expand 0 -fill both -in .lf_config
pack .b_mnt_ed00 .b_mnt_ed01 .b_mnt_ed02 .b_mnt_ed03 .b_mnt_ed04 .b_mnt_ed05 -expand 0 -fill both -in .lf_config
if { $flag_dome_mode != 0 } {
  pack .b_mnt_ed06 -expand 0 -fill both -in .lf_config
}

.pw add .lf_config


################################################################

labelframe .lf_objinfo -text "Object Info" -font $font_labelframe

set buffer_object_info ""

#get_object_data_from_stellarium

text .t_objinfo -font $font_small -width 68 -yscrollcommand ".sb_objinfo set"
scrollbar .sb_objinfo -command ".t_objinfo yview"

pack .t_objinfo .sb_objinfo -side left -expand 1 -fill both -in .lf_objinfo

.pw add .lf_objinfo


################################################################

labelframe .lf_codefield -text "Log / Send or Exec an SSC" -font $font_labelframe

entry .e_code -textvariable buffer_codefield -font $font_codefield
bind .e_code <Return> "send_ssc_from_codefield"
button .b_code0 -text "SEND/EXEC" -font $font_codefield -command "send_ssc_from_codefield"
button .b_code1 -text "Clr" -font $font_codefield -command "set buffer_codefield \"\""
button .b_code2 -text "Doc" -font $font_codefield -command "set buffer_codefield https://stellarium.org/doc/head/scripting.html"
button .b_code3 -text "Tcl/Tk" -font $font_codefield -command "set buffer_codefield http://www.nct9.ne.jp/m_hiroi/tcl_tk_doc/"
button .b_code4 -text "Date" -font $font_codefield -command "set_buffer_codefield_from_buffer_datearg"
pack .b_code1 .b_code4 .b_code2 .b_code3 -side left -in .lf_codefield
pack .e_code -side left -expand 1 -fill both -in .lf_codefield
pack .b_code0 -side left -in .lf_codefield

pack .lf_codefield -expand 0 -fill both  -padx 4 -pady 3 -in .fr_rootwin


########################################################################

pack .fr_rootwin

########################################################################

deselect_all
select_general

#######################################################################

#
# tEBhEN
#

exec_date_gui

#######################################################################

if { $flag_initialize != 0 } {
  send_ssc_to_initialize_all
} else {
  if { $flag_dome_mode == 0 } {
    exec_ssc_on_server unsetFlagDomeMode.ssc
    exec_ssc_on_server unsetCardinal.ssc setCardinal.ssc
  } else {
    exec_ssc_on_server setFlagDomeMode.ssc
    exec_ssc_on_server localInitView.ssc
    exec_ssc_on_server unsetCardinal.ssc setCardinal.ssc
  }
  # ݒl擾
  exec_ssc_on_server getCurrentSettings.ssc
  after 500
  get_current_settings
  initialize_all_gui_params
}

