#########################################################################
##                                                                     ##
##  I N T E R S T E L L A R   C O N S O L E  Date Time GUI             ##
##            Copyright (C) 2025- Misato Observatory, Wakayama, Japan  ##
##                                                                     ##
##  This software is available under the GPL (General Public License)  ##
##                                                                     ##
##       Tcl/Tk References: http://www.nct9.ne.jp/m_hiroi/tcl_tk_doc/  ##
##                                                                     ##
#########################################################################

#  【注意】この GUI は、グリッドラインの更新や光害の制御を行っている。
#          Host用 GUI ではない場合、flag_host_gui = 0 とする事でこの制御
#          が無効化される。

#
# サーバURL, OS依存の定数, GUIの基本設定(フォント,色)をセット
#
source local-config.tcl

#
# 基本的procをインポートする
#
source base-proc.tcl


####################################################################
##                                                                ##
##              G E N E R I C   P R O C E D U R E S               ##
##                                                                ##
####################################################################

#
# ISC用 基本的procをインポートする
#
source isc-proc.tcl

#
# 閏年かどうか
#
#proc is_leapyear { year } {
#
#  if { [catch "clock scan 2/29/$year"] } {
#    return 0
#  } else {
#    return 1
#  }
#
#}

#
# 汎用: stellariumサーバ上の指定されたスクリプト(foo.ssc)を実行 (可変引数)
#
proc exec_ssc_on_server_dg { args } {

  global stellarium_run_url stellarium_script_dir
  global buffer_internal_info_dg

  foreach script_file $args {

    set buffer_internal_info_dg "$script_file"

    do_http_post $stellarium_run_url "id=$stellarium_script_dir/$script_file" ret buffer_internal_info_dg
    set status [string trim $ret]

    if { $status ne "ok" } {
      #wm state . normal
      #tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR in $script_file: $status"
      set buffer_internal_info_dg "$script_file / ERROR: $status"
    }

  }

}

#
# 汎用: ISC Server からファイル filename にある値をGETし、
#       連想配列を作って asarr_ret に返す
#
proc get_data_from_iscserv_dg { filename asarr_ret } {

  global iscserv_url
  global buffer_internal_info_dg
  upvar $asarr_ret asarr_ref

  do_http_get $iscserv_url/$filename ret_string buffer_internal_info_dg

  if { $ret_string eq "" } {
    wm state . normal
    tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET $iscserv_url/$filename"
  } else {
    set lines [split $ret_string "\n"]
    set len_lines [llength $lines]
    set flg_get 0
    for {set i 0} {$i < $len_lines} {incr i} {
      set txt_line [lindex $lines $i]
      set lst [split [string trim $txt_line] ","]
      if { [llength $lst] == 2 } {
        set k [string trim [lindex $lst 0]]
        set v [string trim [lindex $lst 1]]
        set asarr_ref($k) $v
        set flg_get 1
      }
    }
    if { $flg_get == 0 } {
      wm state . normal
      tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET $iscserv_url/$filename"
    }
  }

}


####################################################################
##                                                                ##
##                      P R O C E D U R E S                       ##
##                                                                ##
####################################################################

#
# 設定された時刻を stellariumサーバへ送信
#
proc send_ssc_ldatetime {} {

  global stellarium_direct_url
  global buffer_ldate_label
  global buffer_ltime_label
  global flag_during_input_ldatetime
  global flag_needs_update_datetime_info
  global half_update_cycle
  global flag_light_moon_pollution

  if { $flag_during_input_ldatetime != 0 } {

    global buffer_internal_info_dg

    set buffer_ldate_label [regsub -all {[/]} $buffer_ldate_label {-}]
    set buffer_ldate_label [regsub -all {[^-0-9]} $buffer_ldate_label {}]
    set date_str_in $buffer_ldate_label

    set buffer_ltime_label [regsub -all {[-]} $buffer_ltime_label {:}]
    set buffer_ltime_label [regsub -all {[^:0-9]} $buffer_ltime_label {}]
    set time_str_in $buffer_ltime_label

    set s [format "core.setDate(\"%sT%s\",\"local\")" $date_str_in $time_str_in]

    # send a code using POST
    set buffer_internal_info_dg $s
    do_http_post $stellarium_direct_url "code=$s" ret buffer_internal_info_dg

    # To non-input mode
    .e_ldate configure -state disable
    .e_ltime configure -state disable
    set flag_during_input_ldatetime 0

    # Force an update
    # (not requied)
    ##set flag_needs_update_datetime_info 1

    # 月明かり/光害を制御
    if { $flag_light_moon_pollution != 0 } {
      after $half_update_cycle exec_ssc_on_server_dg updateLightMoonPollution.ssc
    }

  }

}

#
# ISC Server から太陽・月情報をGETし、連想配列 asarr_sunmoon_info を作る
#
#proc get_sunmoon_info {} {
#
#  global asarr_sunmoon_info
#
#  get_data_from_iscserv_dg "_tmp_SunMoonTime.isc.txt" asarr_sunmoon_info
#
#}

#
# ISC Server から太陽・惑星情報をGETし、連想配列 asarr_sunplanets_info を作る
#  返り値: 有効値の個数
#  (Stellariumは日付をまたぐと、rise や set に "-" を返す。"-" はカウントしない)
#
proc get_sunplanets_info {} {

  global asarr_sunplanets_info

  get_data_from_iscserv_dg "_tmp_SunPlanetsTime.isc.txt" asarr_sunplanets_info

  set n_valid 0
  foreach {index value} [array get asarr_sunplanets_info *] {
    if { $value ne "" && $value ne "-" } {
      incr n_valid
    }
  }

  return $n_valid

}

#
# ISC Server から月情報をGETし、連想配列 asarr_moon_info を作る
#
proc get_moon_info {} {

  global asarr_moon_info

  get_data_from_iscserv_dg "_tmp_MoonTime.isc.txt" asarr_moon_info

}

#
# オフセット値から日付時刻文字列を生成し
# buffer_ldate_label, buffer_ltime_label へ入れる
#
proc set_ldatetime_by_ymds { off_year off_month off_day off_sec } {

  global buffer_ldate_label
  global buffer_ltime_label

  global flag_during_input_ldatetime
  global flag_immediate_input

  # To input mode
  .e_ldate configure -state normal
  .e_ltime configure -state normal
  set flag_during_input_ldatetime 1

  # filtering
  set date_str_in [regsub {^[^-0-9]+} $buffer_ldate_label {}]
  set time_str_in [regsub {^[^0-9]+} $buffer_ltime_label {}]

  # Ignore Year <= 0000
  if { [string match "-*" $buffer_ldate_label] } {
    set date_str_in "0001-01-01"
  }
  if { [string match "0000*" $buffer_ldate_label] } {
    set date_str_in "0001-01-01"
  }

  # "0000-01-01" -> "0-1-1"
  set date_str [regsub -all {0*(\d+)} $date_str_in {\1}]
  set time_str [regsub -all {0*(\d+)} $time_str_in {\1}]

  # split
  set date_list [split $date_str "-"]
  set date_y [lindex $date_list 0]
  set date_m [lindex $date_list 1]
  set date_d [lindex $date_list 2]
  set time_list [split $time_str ":"]
  set time_h [lindex $time_list 0]
  set time_m [lindex $time_list 1]
  set time_s [lindex $time_list 2]

  # Fix year < 100 problem ...
  # i.e. [clock scan {0099-01-01 -1 year}] returns wrong value ...
  if { $date_y < 101 } {
    set date_y [expr $date_y + 101]
    set off_year [expr $off_year - 101]
    # overwrite...
    set date_str_in [format "%04d-%02d-%02d" $date_y $date_m $date_d]
  }

  if { $off_sec != 0 } {

    set tm_s_raw [expr $time_s + $off_sec]

    set time_s [expr $tm_s_raw % 60]
    if { 0 <= $tm_s_raw } {
      set tm_m_raw [expr $time_m + ($tm_s_raw / 60)]
    } else {
      set tm_m_raw [expr $time_m + (($tm_s_raw - 1) / 60)]
    }

    set time_m [expr $tm_m_raw % 60]
    if { 0 <= $tm_m_raw } {
      set tm_h_raw [expr $time_h + ($tm_m_raw / 60)]
    } else {
      set tm_h_raw [expr $time_h + (($tm_m_raw - 1) / 60)]
    }

    set time_h [expr $tm_h_raw % 24]
    if { 0 <= $tm_h_raw } {
      set tm_d_raw [expr 0 + ($tm_h_raw / 24)]
    } else {
      set tm_d_raw [expr 0 + (($tm_h_raw - 1) / 24)]
    }

    set off_day [expr $off_day + $tm_d_raw]

    # SET!!
    set buffer_ltime_label [format "%02d:%02d:%02d" $time_h $time_m $time_s]

  }

  if { $off_day != 0 || $off_month != 0 || $off_year != 0 } {

    # initial value
    set date_str_in_updated $date_str_in

    if { $off_day != 0 } {
      set lst [list $date_str_in $off_day "day"]
      set date_raw [clock scan $lst]
      set date_str_in_updated [clock format $date_raw -format {%Y-%m-%d}]
    }

    if { $off_month != 0 } {
      set lst [list $date_str_in_updated $off_month "month"]
      set date_raw [clock scan $lst]
      set date_str_in_updated [clock format $date_raw -format {%Y-%m-%d}]
    }

    set date_str [regsub -all {0*(\d+)} $date_str_in_updated {\1}]
    set date_list [split $date_str "-"]
    set date_y [lindex $date_list 0]
    set date_m [lindex $date_list 1]
    set date_d [lindex $date_list 2]

    # do not handle BC
    if { $date_y + $off_year < 1 } {
      set off_year [expr 1 - $date_y]
    }
    if { 9999 < $date_y + $off_year } {
      set off_year [expr 9999 - $date_y]
    }

    # Fix year < 1900 problem ...
    set flag_force_simple_yearcalc 1

    # Fix year < 100 problem ...
    if { $flag_force_simple_yearcalc == 0 && 100 <= $date_y + $off_year } {
      set lst [list $date_str_in_updated $off_year "year"]
      set date_raw [clock scan $lst]
      set result_str [clock format $date_raw -format {%Y-%m-%d}]
    } else {
      set status [regexp -- {([-]*[0-9]+)[-]([0-9]+[-][0-9]+)} $date_str_in_updated match s_yyyy s_mmdd]
      if { $status == 1 } {
        # "0012" -> "12"
        set s_yyyy [regsub -all {0*(\d+)} $s_yyyy {\1}]
        set s_yyyy [expr $s_yyyy + $off_year]
      }
      set result_str [format "%04d-%s" $s_yyyy $s_mmdd]
    }
    # SET!!
    set buffer_ldate_label $result_str

  }

  if { $flag_immediate_input != 0 } {
    send_ssc_ldatetime
  }

}

#
# ローカル日付時刻への入力Enable/Disableの切り替え
#
proc setup_ldatetime_field { args } {

  global flag_during_input_ldatetime
  global buffer_internal_info_dg

  foreach flg $args {
    set flag_during_input_ldatetime $flg
  }

  if { $flag_during_input_ldatetime == 0 } {
    .e_ldate configure -state disable
    .e_ltime configure -state disable
  } else {
    .e_ldate configure -state normal
    .e_ltime configure -state normal
  }

  #set buffer_internal_info_dg $flag_during_input_ldatetime

}

#
# PC時刻をセット
#
proc set_pcdatetime {} {

  global buffer_ldate_label
  global buffer_ltime_label
  global flag_immediate_input

  # To input mode
  setup_ldatetime_field 1

  #### Current date time ####

  set pc_time_raw [clock seconds]
  set buffer_ldate_label [clock format $pc_time_raw -format {%Y-%m-%d}]
  set buffer_ltime_label [clock format $pc_time_raw -format {%H:%M:%S}]

  if { $flag_immediate_input != 0 } {
    send_ssc_ldatetime
  }

}

#
# ユーザ時刻をセット
#
proc set_start_time { arg_ltime } {

  global buffer_ldate_label
  global buffer_ltime_label
  global flag_send_today_with_users_time

  # To input mode
  setup_ldatetime_field 1

  #### Current date time ####

  set pc_time_raw [clock seconds]
  if { $flag_send_today_with_users_time != 0 } {
    set buffer_ldate_label [clock format $pc_time_raw -format {%Y-%m-%d}]
  }
  if { $arg_ltime eq "-" } {
    set buffer_ltime_label [clock format $pc_time_raw -format {%H:%M:%S}]
  } else {
    set buffer_ltime_label [format "%s:00" $arg_ltime]
  }

  send_ssc_ldatetime

}

#set counter_debug1 0
#set counter_debug2 0

#
# スケジューリング用proc
#  flag_half_cycle: 1サイクルにつき 2回実行する場合の 2回目なら 1 がセットされる
#
proc do_scheduling { flag_half_cycle } {

  global current_ltime
  global buffer_nightend
  global buffer_sunrise
  global buffer_sunset
  global flag_light_moon_pollution
  global flag_astronomical_dawn
  global buffer_internal_info_dg

  #global counter_debug1
  #global counter_debug2

  # いずれかのスケジューリング設定があれば実行
  if { $flag_light_moon_pollution != 0 || $flag_astronomical_dawn != 0 } {

    # "00:01:01" -> "0:01:01"
    set ltime_str [regsub -all {0*(\d+)} $current_ltime {\1}]
    set sunrise_str [regsub -all {0*(\d+)} $buffer_sunrise {\1}]
    set sunset_str [regsub -all {0*(\d+)} $buffer_sunset {\1}]

    # split
    set ltime_list [split $ltime_str ":"]
    set sunrise_list [split $sunrise_str ":"]
    set sunset_list [split $sunset_str ":"]

    if { [llength $ltime_list] == 3 && [llength $sunrise_list] == 2 && [llength $sunset_list] == 2 } {

      set ltime_h [lindex $ltime_list 0]
      set ltime_m [lindex $ltime_list 1]
      set ltime_s [lindex $ltime_list 2]
      set sunrise_h [lindex $sunrise_list 0]
      set sunrise_m [lindex $sunrise_list 1]
      set sunset_h [lindex $sunset_list 0]
      set sunset_m [lindex $sunset_list 1]

      set ltime [expr ($ltime_h * 3600) + ($ltime_m * 60) + $ltime_s]

      #set buffer_internal_info_dg [format "debug '%s' '%s' '%s'" $ltime_h $ltime_m $ltime_s]

      if { $sunrise_str ne "--:--" } {
        set sunrise [expr ($sunrise_h * 3600) + ($sunrise_m * 60)]
      } else {
        set sunrise "-"
      }
      if { $sunset_str ne "--:--" } {
        set sunset [expr ($sunset_h * 3600) + ($sunset_m * 60)]
      } else {
        set sunset "-"
      }

      # 月明かり/光害を制御
      if { $flag_light_moon_pollution != 0 } {
        set flg 0
        if { $sunset ne "-" } {
          if { $sunset < $ltime } {
            set flg 1
          }
        }
        if { $sunrise ne "-" } {
          if { $ltime < $sunrise } {
            set flg 1
          }
        }
        if { $flg == 1 } {
          exec_ssc_on_server_dg updateLightMoonPollution.ssc
          #incr counter_debug1
        }
      }

      # 薄明中に太陽系ON
      if { $flag_half_cycle == 0 && $flag_astronomical_dawn != 0 } {

        # "00:01:01" -> "0:01:01"
        set nightend_str [regsub -all {0*(\d+)} $buffer_nightend {\1}]

        # split
        set nightend_list [split $nightend_str ":"]

        if { [llength $nightend_list] == 2 } {

          set nightend_h [lindex $nightend_list 0]
          set nightend_m [lindex $nightend_list 1]

          if { $nightend_str ne "--:--" } {
            set nightend [expr ($nightend_h * 3600) + ($nightend_m * 60)]
          } else {
            set nightend "-"
          }

          #set buffer_internal_info_dg [format "debug '%s' '%s' '%s'" $ltime $nightend $sunrise]

          if { $nightend ne "-" && $sunrise ne "-" } {
            if { $nightend <= $ltime && $ltime <= $sunrise } {
              exec_ssc_on_server_dg setSolarSystem.ssc
              #incr counter_debug2
            }
          }

        }

      }

    }

  }

  # debug
  #set buffer_internal_info_dg "counter_debug = $counter_debug1, $counter_debug2"

}


#
# 日付/時刻 薄明時刻など 表示用 無限ループ
#
proc show_date_time_info {} {

  global flag_host_gui
  global iscserv_root_directory
  global iscserv_url

  global base_update_cycle
  global half_update_cycle

  global stellarium_status_url
  global current_ldate
  global current_ltime
  global buffer_ldate_label
  global buffer_ltime_label
  global buffer_date_label
  global buffer_time_label
  global buffer_alt_label
  global buffer_lon_label
  global buffer_lat_label
  global buffer_sunrise
  global buffer_sunset
  global buffer_night
  global buffer_nightend
  global buffer_moonrise
  global buffer_moonset
  global buffer_moonage

  global buffer_mercuryrise
  global buffer_mercuryset
  global buffer_mercurytransit
  global buffer_venusrise
  global buffer_venusset
  global buffer_venustransit
  global buffer_marsrise
  global buffer_marsset
  global buffer_marstransit
  global buffer_jupiterrise
  global buffer_jupiterset
  global buffer_jupitertransit
  global buffer_saturnrise
  global buffer_saturnset
  global buffer_saturntransit
  global buffer_uranusrise
  global buffer_uranusset
  global buffer_uranustransit
  global buffer_neptunerise
  global buffer_neptuneset
  global buffer_neptunetransit

  global flag_during_input_ldatetime
  global flag_needs_update_datetime_info
  global counter_for_delayed_computing

  #global asarr_sunmoon_info
  global asarr_sunplanets_info
  global asarr_moon_info

  global counter_sunplanets_info
  global counter_moon_info

  global buffer_pcdate_label
  global buffer_pctime_label

  global buffer_internal_info_dg

  # flag to use suncalc (JavaScript)
  set flg_use_suncalc 0

  # Set debug=1 to write code to $code_file
  set debug 0
  set code_file "tmp_calc_sun.txt"

  #### Current date time ####

  set pc_time_raw [clock seconds]
  set buffer_pcdate_label [clock format $pc_time_raw -format {%Y-%m-%d}]
  set buffer_pctime_label [clock format $pc_time_raw -format {%H:%M:%S}]

  #### Planetarium date time ####

  set diff_sec 0

  if { 1 } {

    # get status from stellarium
    do_http_get $stellarium_status_url line buffer_internal_info_dg
    if { $line eq "" } {
      exit
    }

    set flg_date_changed 0
    set flg_hour_changed 0
    set flg_time_changed 0
    set flg_site_changed 0

    set status [regexp -- {.*["]local["]:["]([^"T]+)T([^".]+)} $line match s_ldate s_ltime]
    if { $status == 1 } {
      if { $s_ltime ne $current_ltime } {
        scan $current_ltime "%d:%d:%d" hh0 mm0 ss0
        set v0sec [expr $ss0 + 60 * $mm0 + 3600 * $hh0]
        scan $s_ltime "%d:%d:%d" hh1 mm1 ss1
        set v1sec [expr $ss1 + 60 * $mm1 + 3600 * $hh1]
        #
        #set v $current_ltime
        #set v0sec [clock scan $v]
        #set v $s_ltime
        #set v1sec [clock scan $v]
        #
        set diff_sec [expr abs($v1sec - $v0sec)]
        #
        if { $hh0 != $hh1 } {
          set flg_hour_changed 1
        }
        #
        set current_ltime $s_ltime
        set flg_time_changed 1
      }
      if { $s_ldate ne $current_ldate } {
        set current_ldate $s_ldate
        set flg_date_changed 1
      }
    }
    set status [regexp -- {.*["]utc["]:["]([^"T]+)T([^".]+)} $line match s_date s_time]
    if { $status == 1 } {
      set buffer_date_label $s_date
      set buffer_time_label $s_time
    }
    set status [regexp -- {.*["]altitude["]:([^,]+).*["]latitude["]:([^,]+).*["]longitude["]:([^,]+)} $line match s_alt0 s_lat0 s_lon0]
    if { $status == 1 } {
      set s_alt [format "%.0fm" $s_alt0]
      set s_lon [format "%+.4f" $s_lon0]
      set s_lat [format "%+.4f" $s_lat0]
      if { $s_alt != $buffer_alt_label || $s_lon != $buffer_lon_label || $s_lat != $buffer_lat_label } {
        set buffer_alt_label $s_alt
        set buffer_lon_label $s_lon
        set buffer_lat_label $s_lat
        set flg_site_changed 1
      }
    }

    #
    # カウントダウンのスタート値を設定
    #
    # for delayed processing (at HOST)
    #  103->102 : compute moon info only
    #    3->2   : compute sun+planets/moon info and update additional grid lines, etc.
    #
    if { $flg_hour_changed != 0 } {
      # avoid transition of 3 -> 103, etc.
      if { 0 < $counter_for_delayed_computing && $counter_for_delayed_computing < 100 } {
        set counter_for_delayed_computing 3
      } else {
        set counter_for_delayed_computing 103
      }
    }
    if { $flg_date_changed != 0 || $flg_site_changed != 0 || $flag_needs_update_datetime_info != 0 } {
      set counter_for_delayed_computing 3
    }

    # Compute Sun/Moon set,rise, etc. and update additional grid lines, etc.
    #
    if { $counter_for_delayed_computing == 2 } {

      if { $flag_host_gui != 0 && $flg_use_suncalc == 0 } {
        exec_ssc_on_server_dg getSunPlanetsTime.ssc getMoonTime.ssc
      }

      if { $flag_host_gui != 0 } {
        # グリッドライン類のアップデート
        exec_ssc_on_server_dg updateEclipticDateLabels.ssc updateSunDiurnalMotion.ssc updateEarthOrbit.ssc updateEarthOrbitLabels.ssc
      }

    } elseif { $counter_for_delayed_computing == 102 } {

      if { $flag_host_gui != 0 && $flg_use_suncalc == 0 } {
        exec_ssc_on_server_dg getMoonTime.ssc
      }

    }

    # host, remote 間の排他処理
    # host では、$counter_for_delayed_computing をみて表示更新を行い、
    # remote では、$counter_{sunplanets or moon}_info をみて表示更新を行う

    set flag_get_sunplanets_info 0
    set flag_get_moon_info 0
    if { $flag_host_gui != 0 } {
      #
      # host側
      #
      if { $counter_for_delayed_computing == 1 } {
        set flag_get_sunplanets_info 1
        incr counter_sunplanets_info
        save_text_file "$counter_sunplanets_info\n" $iscserv_root_directory/_tmp_count_sunplanets_info.isc.txt
      }
      if { $counter_for_delayed_computing == 101 } {
        set flag_get_moon_info 1
        incr counter_moon_info
        save_text_file "$counter_moon_info\n" $iscserv_root_directory/_tmp_count_moon_info.isc.txt
      }
    } else {
      #
      # remote側
      #
      do_http_get $iscserv_url/_tmp_count_sunplanets_info.isc.txt ret buffer_internal_info_dg
      set cnt [string trim $ret]
      if { $cnt ne "" && $cnt ne $counter_sunplanets_info } {
        set flag_get_sunplanets_info 1
        set counter_sunplanets_info $cnt
      }
      do_http_get $iscserv_url/_tmp_count_moon_info.isc.txt ret buffer_internal_info_dg
      set cnt [string trim $ret]
      if { $cnt ne "" && $cnt ne $counter_moon_info } {
        set flag_get_moon_info 1
        set counter_moon_info $cnt
      }
    }

    # Stellarium のバグ対策
    set count_sunplanets_info 0

    if { $flag_get_sunplanets_info != 0 || $flag_get_moon_info != 0 } {

      set year_valid 0

      if { $flg_use_suncalc == 0 } {

        if { $flag_get_sunplanets_info != 0 } {

          # Stellarium のバグ対策: 値が "" でも"-" でもない個数が返ってくる
          set count_sunplanets_info [get_sunplanets_info]

          set l_sunrise $asarr_sunplanets_info(SunRise)
          set l_sunset $asarr_sunplanets_info(SunSet)
          set l_nightend $asarr_sunplanets_info(AstronomicalDawn)
          set l_night $asarr_sunplanets_info(AstronomicalDusk)
          set l_mercuryrise $asarr_sunplanets_info(MercuryRise)
          set l_mercuryset $asarr_sunplanets_info(MercurySet)
          set l_mercurytransit $asarr_sunplanets_info(MercuryTransit)
          set l_venusrise $asarr_sunplanets_info(VenusRise)
          set l_venusset $asarr_sunplanets_info(VenusSet)
          set l_venustransit $asarr_sunplanets_info(VenusTransit)
          set l_marsrise $asarr_sunplanets_info(MarsRise)
          set l_marsset $asarr_sunplanets_info(MarsSet)
          set l_marstransit $asarr_sunplanets_info(MarsTransit)
          set l_jupiterrise $asarr_sunplanets_info(JupiterRise)
          set l_jupiterset $asarr_sunplanets_info(JupiterSet)
          set l_jupitertransit $asarr_sunplanets_info(JupiterTransit)
          set l_saturnrise $asarr_sunplanets_info(SaturnRise)
          set l_saturnset $asarr_sunplanets_info(SaturnSet)
          set l_saturntransit $asarr_sunplanets_info(SaturnTransit)
          set l_uranusrise $asarr_sunplanets_info(UranusRise)
          set l_uranusset $asarr_sunplanets_info(UranusSet)
          set l_uranustransit $asarr_sunplanets_info(UranusTransit)
          set l_neptunerise $asarr_sunplanets_info(NeptuneRise)
          set l_neptuneset $asarr_sunplanets_info(NeptuneSet)
          set l_neptunetransit $asarr_sunplanets_info(NeptuneTransit)

        }

        get_moon_info
        set l_moonrise $asarr_moon_info(MoonRise)
        set l_moonset $asarr_moon_info(MoonSet)
        set l_moontransit $asarr_moon_info(MoonTransit)
        set l_moonage $asarr_moon_info(MoonAge)

        set year_valid 1

      } else {

        # こちらのコードでは calc_sun_3.js を cscript で実行して天文薄明を取得
        # ただし、計算範囲に制限あり

        set status [regexp -- {^([-+]*)([0-9]+)} $current_ldate match ss_year_pm ss_year_val]
        if { $status == 1 } {
          set ss_year_val [regsub -all {0*(\d+)} $ss_year_val {\1}]
          if { $ss_year_pm ne "-" && 1900 <= $ss_year_val && $ss_year_val <= 2150 } {
            set year_valid 1
          }
        }

        #set buffer_internal_info_dg [format "debug '%s' '%s'" $ss_year_pm $ss_year_val]

        if { $year_valid == 1 } {

          # とりあえず non-0 をセット
          set count_sunplanets_info 1

          set ss_date [regsub -all {[-]} $current_ldate {/}]
          set ss_alt  [regsub -all {m} $buffer_alt_label {}]

          set f [open "| cscript //Nologo calc_sun_3.js $ss_date $buffer_lon_label $buffer_lat_label $ss_alt" r]
          gets $f l_date_ymd
          gets $f l_sunrise
          gets $f l_sunset
          gets $f l_nightend
          gets $f l_night
          gets $f l_moonrise
          gets $f l_moonset
          catch {close $f} std_err
          set l_moonage "-"

          if { $l_date_ymd ne $current_ldate } {
            set year_valid 0
          }

          set buffer_internal_info_dg [format "suncalc3: '%s'" $l_date_ymd]

        }

      }

      if { $year_valid == 1 } {

        # 【注意】Stellariumのバグ(?)
        #
        # ・timeRate != 0 の状態で日付をまたぐと、Stellarium は惑星や月に関する
        #   時刻として "-" を返す。
        # ・この問題は完全に解決できないので、全要素において "-" が返された場合
        #   ($count_sunplanets_info が 0 の場合) は時刻情報はそのまま残している。

        if { $flag_get_sunplanets_info != 0 && 0 < $count_sunplanets_info } {

          set status [regexp -- {([^:]+):([^:]+)} $l_sunrise match h0 m0]
          if { $status == 1 } {
            set buffer_sunrise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_sunrise "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_sunset match h0 m0]
          if { $status == 1 } {
            set buffer_sunset [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_sunset "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_nightend match h0 m0]
          if { $status == 1 } {
            set buffer_nightend [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_nightend "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_night match h0 m0]
          if { $status == 1 } {
            set buffer_night [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_night "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_mercuryrise match h0 m0]
          if { $status == 1 } {
            set buffer_mercuryrise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_mercuryrise "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_mercuryset match h0 m0]
          if { $status == 1 } {
            set buffer_mercuryset [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_mercuryset "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_mercurytransit match h0 m0]
          if { $status == 1 } {
            set buffer_mercurytransit [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_mercurytransit "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_venusrise match h0 m0]
          if { $status == 1 } {
            set buffer_venusrise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_venusrise "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_venusset match h0 m0]
          if { $status == 1 } {
            set buffer_venusset [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_venusset "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_venustransit match h0 m0]
          if { $status == 1 } {
            set buffer_venustransit [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_venustransit "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_marsrise match h0 m0]
          if { $status == 1 } {
            set buffer_marsrise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_marsrise "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_marsset match h0 m0]
          if { $status == 1 } {
            set buffer_marsset [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_marsset "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_marstransit match h0 m0]
          if { $status == 1 } {
            set buffer_marstransit [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_marstransit "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_jupiterrise match h0 m0]
          if { $status == 1 } {
            set buffer_jupiterrise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_jupiterrise "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_jupiterset match h0 m0]
          if { $status == 1 } {
            set buffer_jupiterset [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_jupiterset "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_jupitertransit match h0 m0]
          if { $status == 1 } {
            set buffer_jupitertransit [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_jupitertransit "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_saturnrise match h0 m0]
          if { $status == 1 } {
            set buffer_saturnrise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_saturnrise "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_saturnset match h0 m0]
          if { $status == 1 } {
            set buffer_saturnset [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_saturnset "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_saturntransit match h0 m0]
          if { $status == 1 } {
            set buffer_saturntransit [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_saturntransit "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_uranusrise match h0 m0]
          if { $status == 1 } {
            set buffer_uranusrise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_uranusrise "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_uranusset match h0 m0]
          if { $status == 1 } {
            set buffer_uranusset [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_uranusset "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_uranustransit match h0 m0]
          if { $status == 1 } {
            set buffer_uranustransit [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_uranustransit "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_neptunerise match h0 m0]
          if { $status == 1 } {
            set buffer_neptunerise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_neptunerise "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_neptuneset match h0 m0]
          if { $status == 1 } {
            set buffer_neptuneset [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_neptuneset "--:--"
          }
          set status [regexp -- {([^:]+):([^:]+)} $l_neptunetransit match h0 m0]
          if { $status == 1 } {
            set buffer_neptunetransit [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_neptunetransit "--:--"
          }

        }

        set status0 [regexp -- {([^:]+):([^:]+)} $l_moonrise match h0 m0]
        set status1 [regexp -- {([^:]+):([^:]+)} $l_moonset match h1 m1]
        set status2 [regexp -- {([^:]+):([^:]+)} $l_moontransit match h2 m2]

        # 【注意】これもStellariumのバグ(?)対策
        #  moon{rise,set,transit} とも "-" が返された場合、時刻情報はそのまま残している。
        if { $status0 == 1 || $status1 == 1 || $status2 == 1 } {
          if { $status0 == 1 } {
            set buffer_moonrise [format "%02d:%02d" $h0 $m0]
          } else {
            set buffer_moonrise "--:--"
          }
          if { $status1 == 1 } {
            set buffer_moonset [format "%02d:%02d" $h1 $m1]
          } else {
            set buffer_moonset "--:--"
          }
        }
        set buffer_moonage [format "%.1f" $l_moonage]

      } else {

        # Unsupported/Unreliable YEAR in suncalc...

        set buffer_sunrise "-"
        set buffer_sunset "-"
        set buffer_nightend "-"
        set buffer_night "-"
        set buffer_moonrise "-"
        set buffer_moonset "-"

      }

    }

    if { $flag_host_gui != 0 } {
      if { $flg_date_changed != 0 || $flg_time_changed != 0 || $flg_site_changed != 0 || $flag_needs_update_datetime_info != 0 } {

        # Do scheduling ...
        do_scheduling 0

        # 時刻レートが大きい場合は 1サイクルにつき 2回実行する
        if { 1 < $diff_sec && $diff_sec < 600 } {
          after $half_update_cycle do_scheduling 1
        }

      }
    }

    # reset and count-down
    set flag_needs_update_datetime_info 0
    if { 0 < $counter_for_delayed_computing } {
      set counter_for_delayed_computing [expr $counter_for_delayed_computing - 1]
      if { $counter_for_delayed_computing == 100 } {
        set counter_for_delayed_computing 0
      }
    }

  }

  if { $flag_during_input_ldatetime == 0 } {

    set buffer_ldate_label $current_ldate
    set buffer_ltime_label $current_ltime

  }

  #set buffer_internal_info_dg $diff_sec

  # ウエイト(必要に応じて可変にして良い。ただしマシンパワーの消費とcounter_for_delayed_computingの設定に注意)
  if { 1 < $diff_sec } {
    after $base_update_cycle show_date_time_info
  } else {
    after $base_update_cycle show_date_time_info
  }

}

#
# 惑星情報表示をコンパクト表示にする
#
proc unset_detailed_planets_info {} {
  .pw forget .fr_planetsinfoall
  .pw add .fr_planetsinfo
}

#
# 惑星情報表示を詳細表示にする
#
proc set_detailed_planets_info {} {
  .pw forget .fr_planetsinfo
  .pw add .fr_planetsinfoall
}


####################################################################
##                                                                ##
##                            M A I N                             ##
##                                                                ##
####################################################################

#
# Host用 GUI かどうかのフラグ (リモートマシンの場合は 0)
#
#  ・値が 0 以外ならこの GUI が Host である事を示し、座標系のアップデート、
#    スケジューリングを行う
#
set flag_host_gui 0

#
# 動作モード
#
set flag_large_gui 0

#
# エディタ
#
set flag_external_editor 0

#
# 引数でフラグを設定、サーバURLが与えられる場合は接続先を変更
#
for {set i 0} {$i < $argc} {incr i} {
  set arg [lindex $argv $i]
  if { $arg eq "-host" } {
    set flag_host_gui 1
    set flag_external_editor 1
  } elseif { $arg eq "-large-gui" } {
    set flag_large_gui 1
  } else {
    set status [regexp -- {(http[:][^,]+),(http[:][^,]+)} $arg match url0 url1]
    if { $status == 1 } {
      set stellarium_url "$url0"
      set iscserv_url    "$url1"
    }
  }
}

#
# GUI上の各時刻のアップデート周期
#

set base_update_cycle 500
set half_update_cycle [expr $base_update_cycle / 2]

#
# 開始時刻設定ファイル
#
set config_list_start_time "list_start-time.txt"

#
# GUI デフォルト値
#

# [+][-]ボタンクリック直後に送信
set flag_immediate_input 0

# ユーザ時刻送信時に今日の日付を送信
set flag_send_today_with_users_time 1

#
# スケジューリング デフォルト値
#

# 月明かり/光害を制御
set flag_light_moon_pollution $flag_host_gui

# 朝薄明で太陽系ON
set flag_astronomical_dawn $flag_host_gui

#
# 色・フォントの設定
#

# +-ボタンの色
set date_button_background "#383868"
set time_button_background "#304830"

# PC情報の色
set pcinfo_foreground "#50d040"

if { $flag_large_gui == 0 } {
  # Normal GUI モード
  # デフォルト
  option add *font [list $gui_font_name 12]
  set font_small [list $gui_font_name 10]
  # ラベルフレーム
  set font_labelframe [list $gui_font_name 10]
  # +-ボタン
  set font_pmbtn [list $gui_font_name 12]
  # 惑星
  set font_planets_rs [list $gui_font_name 10]
  # スケジューリング
  set font_scheduling [list $gui_font_name 10]
  # 内部情報
  set font_internal [list $gui_font_name 10]
  #
  ttk::style configure TButton -font [list $gui_font_name 12]
  ttk::style configure TButton -padding [scaled_padding 3 3]
  ttk::style configure Small.TButton -font [list $gui_font_name 10]
  ttk::style configure Small.TButton -padding [scaled_padding 2 1]
  # TCheckbutton の padding は標準チェックボタン(padding値はデフォルト)よりわずかに小さくしてある
  ttk::style configure TCheckbutton -font [list $gui_font_name 12] -padding [scaled_padding 8 8]
  ttk::style configure TCheckbutton -indicatordiameter [scaled_px 10]
  ttk::style configure Scheduling.TCheckbutton -font $font_scheduling -padding [scaled_padding 6 6]
  # +-ボタン
  ttk::style configure Date.TButton -font [list $gui_font_name 12] -padding [scaled_padding 5 6] -background $date_button_background
  ttk::style configure Time.TButton -font [list $gui_font_name 12] -padding [scaled_padding 5 6] -background $time_button_background
} else {
  # Large GUI モード
  # デフォルト
  option add *font [list $gui_font_name 20]
  set font_small [list $gui_font_name 16]
  # ラベルフレーム
  set font_labelframe [list $gui_font_name 12]
  # +-ボタン
  set font_pmbtn [list $gui_font_name 14]
  # 惑星
  set font_planets_rs [list $gui_font_name 15]
  # スケジューリング
  set font_scheduling [list $gui_font_name 16]
  # 内部情報
  set font_internal [list $gui_font_name 10]
  #
  ttk::style configure TButton -font [list $gui_font_name 20]
  ttk::style configure TButton -padding [scaled_padding 12 11]
  ttk::style configure Small.TButton -font [list $gui_font_name 16]
  ttk::style configure Small.TButton -padding [scaled_padding 8 7]
  # TCheckbutton の padding は標準チェックボタン(padding値はデフォルト)よりわずかに小さくしてある
  ttk::style configure TCheckbutton -font [list $gui_font_name 20] -padding [scaled_padding 10 10]
  ttk::style configure TCheckbutton -indicatordiameter [scaled_px 16]
  ttk::style configure Scheduling.TCheckbutton -font $font_scheduling -padding [scaled_padding 8 8]
  # +-ボタン
  ttk::style configure Date.TButton -font [list $gui_font_name 14] -padding [scaled_padding 11 18] -background $date_button_background
  ttk::style configure Time.TButton -font [list $gui_font_name 14] -padding [scaled_padding 11 18] -background $time_button_background
}

#ttk::style configure Scheduling.TCheckbutton -indicatordiameter [lindex $font_scheduling 1]

# マウスオーバー時の背景色
ttk::style map Date.TButton -background [list active $date_button_background]
ttk::style map Time.TButton -background [list active $time_button_background]


#
# Do not change: Set stellarium_script_dir from localConfig.ssc
#
if { [catch {open "localConfig.ssc" r} fid] } {
  wm state . normal
  tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot open localConfig.ssc."
  exit
} else {
  set stellarium_script_dir "interstellar-console"
  while { [gets $fid line] >= 0 } {
    set status [regexp -- {^[ ]*var[ ]+([A-Za-z0-9_]+)[ ]*[=][ ]*["]([^"]+)["]} $line match s0 s1]
    if { $status == 1 && $s0 eq "stellarium_script_dir" } {
      set stellarium_script_dir $s1
    }
  }
  close $fid
}

#
# サーバURL (1)  サーバ側に保存されているスクリプト(staticなコード)
#                を実行する時に利用
# 
set stellarium_run_url "$stellarium_url/api/scripts/run"

#
# サーバURL (2)  クライアントで生成したスクリプト(dynamicなコード)
#                を送信する時に利用
# 
#  【注意】direct を使う場合、実行できるのは１行のみらしい。
#          複数行のコードを送信すると、Stellariumがクラッシュする
set stellarium_direct_url "$stellarium_url/api/scripts/direct"

#
# サーバURL (3)  情報取得用
# 
set stellarium_status_url "$stellarium_url/api/main/status"

# これは host の場合のみセットされる
set iscserv_root_directory ""


####################################################################
##                                                                ##
##                G U I   C O N S T R U C T I O N                 ##
##                                                                ##
####################################################################

# flag,counter (選択表示で使用)
set flag_during_input_ldatetime 0
set flag_needs_update_datetime_info 0
set counter_for_delayed_computing 0

# 初期値
# 現在の時刻を保持
set current_ldate "0000-00-00"
set current_ltime "00:00:00"
# 以下、ラベル用
set buffer_ldate_label "0000-00-00"
set buffer_ltime_label "00:00:00"
set buffer_date_label "0000-00-00"
set buffer_time_label "00:00:00"
set buffer_lon_label "0.0"
set buffer_lat_label "0.0"
set buffer_alt_label "0m"

set buffer_sunrise  "00:00"
set buffer_sunset   "00:00"
set buffer_night    "00:00"
set buffer_nightend "00:00"
set buffer_moonrise "00:00"
set buffer_moonset  "00:00"
set buffer_moonage  "0.0"

set buffer_mercuryrise    "00:00"
set buffer_mercuryset     "00:00"
set buffer_mercurytransit "00:00"
set buffer_venusrise    "00:00"
set buffer_venusset     "00:00"
set buffer_venustransit "00:00"
set buffer_marsrise    "00:00"
set buffer_marsset     "00:00"
set buffer_marstransit "00:00"
set buffer_jupiterrise    "00:00"
set buffer_jupiterset     "00:00"
set buffer_jupitertransit "00:00"
set buffer_saturnrise    "00:00"
set buffer_saturnset     "00:00"
set buffer_saturntransit "00:00"
set buffer_uranusrise    "00:00"
set buffer_uranusset     "00:00"
set buffer_uranustransit "00:00"
set buffer_neptunerise    "00:00"
set buffer_neptuneset     "00:00"
set buffer_neptunetransit "00:00"

set buffer_pcdate_label "0000-00-00"
set buffer_pctime_label "00:00:00"

set buffer_internal_info_dg "-"

# 太陽・月情報用連想配列
#array set asarr_sunmoon_info {}
array set asarr_sunplanets_info {}
array set asarr_moon_info {}

# 太陽・月情報アップデート用カウンタ
# (remote GUI に対し、アップデートされた事を知らせる)
set counter_sunplanets_info 0
set counter_moon_info 0


#
# GUI のオープン
#  top: 新規のtoplevelウィンドウ名。"" の場合は新規に作らず、
#       既存のメインウィンドウを使用する
#
proc open_date_gui { top } {

  global flag_host_gui
  global stellarium_url
  global iscserv_url
  global flag_large_gui
  global config_list_start_time

  global font_small
  global font_labelframe
  global font_pmbtn
  global font_planets_rs
  global font_scheduling
  global font_internal
  global date_button_background
  global time_button_background
  global pcinfo_foreground

  global flag_during_input_ldatetime
  global flag_immediate_input
  global flag_light_moon_pollution
  global flag_astronomical_dawn
  global flag_send_today_with_users_time

  global buffer_ldate_label
  global buffer_ltime_label
  global buffer_date_label
  global buffer_time_label
  global buffer_lon_label
  global buffer_lat_label
  global buffer_alt_label

  global buffer_sunrise
  global buffer_sunset
  global buffer_night
  global buffer_nightend
  global buffer_moonrise
  global buffer_moonset
  global buffer_moonage

  global buffer_mercuryrise
  global buffer_mercuryset
  global buffer_mercurytransit
  global buffer_venusrise
  global buffer_venusset
  global buffer_venustransit
  global buffer_marsrise
  global buffer_marsset
  global buffer_marstransit
  global buffer_jupiterrise
  global buffer_jupiterset
  global buffer_jupitertransit
  global buffer_saturnrise
  global buffer_saturnset
  global buffer_saturntransit
  global buffer_uranusrise
  global buffer_uranusset
  global buffer_uranustransit
  global buffer_neptunerise
  global buffer_neptuneset
  global buffer_neptunetransit

  global buffer_pcdate_label
  global buffer_pctime_label

  global buffer_internal_info_dg

  set status 0
  if { $top ne "" } {
    set status [catch {toplevel $top} ret]
  }

  if { $status == 0 } {

    # ウィンドゥ名称,属性の設定
    set title_str "Interstellar Console :: date time"
    if { $flag_host_gui != 0 } {
      append title_str " (Host)"
    } else {
      append title_str " ($stellarium_url)"
    }

    if { $top eq "" } {
      wm title . $title_str
      wm resizable . 0 0
    } else {
      wm title $top $title_str
      wm resizable $top 0 0
    }

    ########################################################################

    frame $top.fr_rootwin
    frame $top.fr_root_left
    if { $flag_large_gui == 0 } {
      frame $top.fr_root_right
    }

    ########################################################################

    #panedwindow $top.pw -orient vertical
    #pack $top.pw -fill x -padx [scaled_px 3] -pady [scaled_px 2] -in $top.fr_root_left

    ########################################################################

    labelframe $top.lf_dateinfo -text "Date Time Info" -font $font_labelframe

    frame $top.fr_ldatetm
    label $top.l_ldatetm -text "スクリーン日時" -width 16 -padx 0
    entry $top.e_ldate -textvariable buffer_ldate_label -width 14 -state disable -justify center
    entry $top.e_ltime -textvariable buffer_ltime_label -width 12 -state disable -justify center
    pack $top.l_ldatetm $top.e_ldate $top.e_ltime -side left -expand 1 -fill both -in $top.fr_ldatetm

    bind $top.e_ldate <Return> "send_ssc_ldatetime"
    bind $top.e_ltime <Return> "send_ssc_ldatetime"
    bind $top.e_ldate <ButtonPress-1> "setup_ldatetime_field 1"
    bind $top.e_ltime <ButtonPress-1> "setup_ldatetime_field 1"

    frame $top.fr_lonlat
    label $top.l_lonlat -text "場所" -width 16 -padx 0
    label $top.l_lon -textvariable buffer_lon_label
    label $top.l_lat -textvariable buffer_lat_label
    label $top.l_alt -textvariable buffer_alt_label
    pack $top.l_lonlat $top.l_lon $top.l_lat $top.l_alt -side left -expand 1 -fill x -in $top.fr_lonlat

    frame $top.fr_datetm
    label $top.l_datetm -text "スクリーンUTC" -width 16 -padx 0
    label $top.l_date -textvariable buffer_date_label -width 14 -padx 0 -borderwidth 0
    label $top.l_time -textvariable buffer_time_label -width 12 -padx 0 -borderwidth 0
    pack $top.l_datetm $top.l_date $top.l_time -side left -expand 1 -fill x -in $top.fr_datetm

    #

    frame $top.fr_suntime

    label $top.l_suntime0 -text "太陽" -width 16 -padx 0

    frame $top.fr_sunrise
    label $top.l_suntime10 -text "   " -width 3 -padx 0 -borderwidth 0
    label $top.l_suntime11 -text "出 " -width 3 -padx 0 -borderwidth 0
    label $top.l_sunrise -textvariable buffer_sunrise -width 5 -padx 0 -borderwidth 0
    label $top.l_suntime12 -text "   " -width 3 -padx 0 -borderwidth 0
    pack $top.l_suntime10 -expand 1 -side left -in $top.fr_sunrise
    pack $top.l_suntime11 $top.l_sunrise -side left -in $top.fr_sunrise
    pack $top.l_suntime12 -expand 1 -side left -in $top.fr_sunrise

    frame $top.fr_sunset
    label $top.l_suntime20 -text "  " -width 2 -padx 0 -borderwidth 0
    label $top.l_suntime21 -text "没 " -width 3 -padx 0 -borderwidth 0
    label $top.l_sunset  -textvariable buffer_sunset -width 5 -padx 0 -borderwidth 0
    label $top.l_suntime22 -text "  " -width 2 -padx 0 -borderwidth 0
    pack $top.l_suntime20 -expand 1 -side left -in $top.fr_sunset
    pack $top.l_suntime21 $top.l_sunset -side left -in $top.fr_sunset
    pack $top.l_suntime22 -expand 1 -side left -in $top.fr_sunset

    pack $top.l_suntime0 $top.fr_sunrise $top.fr_sunset -side left -expand 1 -fill x -in $top.fr_suntime

    #

    frame $top.fr_nighttm

    label $top.l_nighttm0 -text "天文薄明" -width 16 -padx 0

    frame $top.fr_nightend
    label $top.l_nighttm10 -text "   " -width 3 -padx 0 -borderwidth 0
    label $top.l_nighttm11 -text "始 " -width 3 -padx 0 -borderwidth 0
    label $top.l_nightend  -textvariable buffer_nightend -width 5 -padx 0 -borderwidth 0
    label $top.l_nighttm12 -text "   " -width 3 -padx 0 -borderwidth 0
    pack $top.l_nighttm10 -expand 1 -side left -in $top.fr_nightend
    pack $top.l_nighttm11 $top.l_nightend -side left -in $top.fr_nightend
    pack $top.l_nighttm12 -expand 1 -side left -in $top.fr_nightend

    frame $top.fr_night
    label $top.l_nighttm20 -text "  " -width 2 -padx 0 -borderwidth 0
    label $top.l_nighttm21 -text "終 " -width 3 -padx 0 -borderwidth 0
    label $top.l_night -textvariable buffer_night -width 5 -padx 0 -borderwidth 0
    label $top.l_nighttm22 -text "  " -width 2 -padx 0 -borderwidth 0
    pack $top.l_nighttm20 -expand 1 -side left -in $top.fr_night
    pack $top.l_nighttm21 $top.l_night -side left -in $top.fr_night
    pack $top.l_nighttm22 -expand 1 -side left -in $top.fr_night

    pack $top.l_nighttm0 $top.fr_nightend $top.fr_night -side left -expand 1 -fill x -in $top.fr_nighttm

    #

    frame $top.fr_moontm

    frame $top.fr_moontime
    label $top.l_moontime0 -text " " -width 1 -padx 0 -borderwidth 0
    label $top.l_moontime1 -text "月 " -width 3 -padx 0 -borderwidth 0
    label $top.l_moontime2 -text "(月齢" -width 5 -padx 0 -borderwidth 0
    label $top.l_moontime3 -textvariable buffer_moonage -width 5 -padx 0
    label $top.l_moontime4 -text ")" -width 1 -padx 0 -borderwidth 0
    label $top.l_moontime5 -text " " -width 1 -padx 0 -borderwidth 0
    pack $top.l_moontime0 -side left -expand 1 -in $top.fr_moontime
    pack $top.l_moontime1 $top.l_moontime2 $top.l_moontime3 $top.l_moontime4 -side left -expand 0 -padx 0 -in $top.fr_moontime
    pack $top.l_moontime5 -side left -expand 1 -in $top.fr_moontime

    frame $top.fr_moonrise
    label $top.l_moontime10 -text "   " -width 3 -padx 0 -borderwidth 0
    label $top.l_moontime11 -text "出 " -width 3 -padx 0 -borderwidth 0
    label $top.l_moonrise -textvariable buffer_moonrise -width 5 -padx 0 -borderwidth 0
    label $top.l_moontime12 -text "   " -width 3 -padx 0 -borderwidth 0
    pack $top.l_moontime10 -expand 1 -side left -in $top.fr_moonrise
    pack $top.l_moontime11 $top.l_moonrise -side left -in $top.fr_moonrise
    pack $top.l_moontime12 -expand 1 -side left -in $top.fr_moonrise

    frame $top.fr_moonset
    label $top.l_moontime20 -text "  " -width 2 -padx 0 -borderwidth 0
    label $top.l_moontime21 -text "没 " -width 3 -padx 0 -borderwidth 0
    label $top.l_moonset  -textvariable buffer_moonset -width 5 -padx 0 -borderwidth 0
    label $top.l_moontime22 -text "  " -width 2 -padx 0 -borderwidth 0
    pack $top.l_moontime20 -expand 1 -side left -in $top.fr_moonset
    pack $top.l_moontime21 $top.l_moonset -side left -in $top.fr_moonset
    pack $top.l_moontime22 -expand 1 -side left -in $top.fr_moonset

    pack $top.fr_moontime $top.fr_moonrise $top.fr_moonset -side left -expand 1 -fill x -in $top.fr_moontm

    pack $top.fr_ldatetm $top.fr_lonlat $top.fr_datetm $top.fr_suntime $top.fr_nighttm $top.fr_moontm -expand 1 -fill x -in $top.lf_dateinfo

    pack $top.lf_dateinfo -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
    #.pw add $top.lf_dateinfo


    ################################################################

    labelframe $top.lf_setdate -text "Set Date Time" -font $font_labelframe

    frame $top.fr_setdate

    frame $top.fr_cb1
    ##ttk::checkbutton $top.cb_setdate -text "入力" -variable flag_during_input_ldatetime -command "setup_ldatetime_field"
    ttk::checkbutton $top.cb_immediate -text "即送信" -variable flag_immediate_input
    pack $top.cb_immediate -side left -in $top.fr_cb1

    frame $top.fr_cb2
    #pack $top.cb_setdate -side left -in $top.fr_cb1

    ttk::button $top.b_setdate -text "SEND" -command "send_ssc_ldatetime"
    ttk::button $top.b_cancel -style Small.TButton -text "取消" -command "setup_ldatetime_field 0"
    pack $top.b_setdate $top.b_cancel -side left -fill y -expand 1 -in $top.fr_cb2
    pack $top.fr_cb1 -in $top.fr_setdate
    pack $top.fr_cb2 -side left -fill y -expand 1 -padx [scaled_px 4] -in $top.fr_setdate

    frame $top.fr_bpm_y
    frame $top.fr_bp_y
    ttk::button $top.bp_y3 -style Date.TButton -text "+" -command "set_ldatetime_by_ymds 1000 0 0 0"
    ttk::button $top.bp_y2 -style Date.TButton -text "+" -command "set_ldatetime_by_ymds 100 0 0 0"
    ttk::button $top.bp_y1 -style Date.TButton -text "+" -command "set_ldatetime_by_ymds 10 0 0 0"
    ttk::button $top.bp_y0 -style Date.TButton -text "+" -command "set_ldatetime_by_ymds 1 0 0 0"
    pack $top.bp_y3 $top.bp_y2 $top.bp_y1 $top.bp_y0 -side left -fill both -in $top.fr_bp_y
    frame $top.fr_bm_y
    ttk::button $top.bm_y3 -style Date.TButton -text "-" -command "set_ldatetime_by_ymds -1000 0 0 0"
    ttk::button $top.bm_y2 -style Date.TButton -text "-" -command "set_ldatetime_by_ymds -100 0 0 0"
    ttk::button $top.bm_y1 -style Date.TButton -text "-" -command "set_ldatetime_by_ymds -10 0 0 0"
    ttk::button $top.bm_y0 -style Date.TButton -text "-" -command "set_ldatetime_by_ymds -1 0 0 0"
    pack $top.bm_y3 $top.bm_y2 $top.bm_y1 $top.bm_y0 -side left -fill both -in $top.fr_bm_y
    pack $top.fr_bp_y $top.fr_bm_y -expand 1 -fill both -in $top.fr_bpm_y

    label $top.l_y_mm -text "-" -font $font_pmbtn

    frame $top.fr_bpm_mm
    frame $top.fr_bp_mm
    ttk::button $top.bp_mm1 -style Date.TButton -text "+" -command "set_ldatetime_by_ymds 0 10 0 0"
    ttk::button $top.bp_mm0 -style Date.TButton -text "+" -command "set_ldatetime_by_ymds 0 1 0 0"
    pack $top.bp_mm1 $top.bp_mm0 -side left -fill both -in $top.fr_bp_mm
    frame $top.fr_bm_mm
    ttk::button $top.bm_mm1 -style Date.TButton -text "-" -command "set_ldatetime_by_ymds 0 -10 0 0"
    ttk::button $top.bm_mm0 -style Date.TButton -text "-" -command "set_ldatetime_by_ymds 0 -1 0 0"
    pack $top.bm_mm1 $top.bm_mm0 -side left -fill both -in $top.fr_bm_mm
    pack $top.fr_bp_mm $top.fr_bm_mm -expand 1 -fill both -in $top.fr_bpm_mm

    label $top.l_mm_d -text "-" -font $font_pmbtn

    frame $top.fr_bpm_d
    frame $top.fr_bp_d
    ttk::button $top.bp_d1 -style Date.TButton -text "+" -command "set_ldatetime_by_ymds 0 0 10 0"
    ttk::button $top.bp_d0 -style Date.TButton -text "+" -command "set_ldatetime_by_ymds 0 0 1 0"
    pack $top.bp_d1 $top.bp_d0 -side left -fill both -in $top.fr_bp_d
    frame $top.fr_bm_d
    ttk::button $top.bm_d1 -style Date.TButton -text "-" -command "set_ldatetime_by_ymds 0 0 -10 0"
    ttk::button $top.bm_d0 -style Date.TButton -text "-" -command "set_ldatetime_by_ymds 0 0 -1 0"
    pack $top.bm_d1 $top.bm_d0 -side left -fill both -in $top.fr_bm_d
    pack $top.fr_bp_d $top.fr_bm_d -expand 1 -fill both -in $top.fr_bpm_d

    label $top.l_d_h -text "T" -font $font_pmbtn

    frame $top.fr_bpm_h
    frame $top.fr_bp_h
    ttk::button $top.bp_h1 -style Time.TButton -text "+" -command "set_ldatetime_by_ymds 0 0 0 36000"
    ttk::button $top.bp_h0 -style Time.TButton -text "+" -command "set_ldatetime_by_ymds 0 0 0 3600"
    pack $top.bp_h1 $top.bp_h0 -side left -fill both -in $top.fr_bp_h
    frame $top.fr_bm_h
    ttk::button $top.bm_h1 -style Time.TButton -text "-" -command "set_ldatetime_by_ymds 0 0 0 -36000"
    ttk::button $top.bm_h0 -style Time.TButton -text "-" -command "set_ldatetime_by_ymds 0 0 0 -3600"
    pack $top.bm_h1 $top.bm_h0 -side left -fill both -in $top.fr_bm_h
    pack $top.fr_bp_h $top.fr_bm_h -expand 1 -fill both -in $top.fr_bpm_h

    label $top.l_h_m -text ":" -font $font_pmbtn

    frame $top.fr_bpm_m
    frame $top.fr_bp_m
    ttk::button $top.bp_m1 -style Time.TButton -text "+" -command "set_ldatetime_by_ymds 0 0 0 600"
    ttk::button $top.bp_m0 -style Time.TButton -text "+" -command "set_ldatetime_by_ymds 0 0 0 60"
    pack $top.bp_m1 $top.bp_m0 -side left -fill both -in $top.fr_bp_m
    frame $top.fr_bm_m
    ttk::button $top.bm_m1 -style Time.TButton -text "-" -command "set_ldatetime_by_ymds 0 0 0 -600"
    ttk::button $top.bm_m0 -style Time.TButton -text "-" -command "set_ldatetime_by_ymds 0 0 0 -60"
    pack $top.bm_m1 $top.bm_m0 -side left -fill both -in $top.fr_bm_m
    pack $top.fr_bp_m $top.fr_bm_m -expand 1 -fill both -in $top.fr_bpm_m

    label $top.l_m_s -text ":" -font $font_pmbtn

    frame $top.fr_bpm_s
    frame $top.fr_bp_s
    ttk::button $top.bp_s1 -style Time.TButton -text "+" -command "set_ldatetime_by_ymds 0 0 0 10"
    ttk::button $top.bp_s0 -style Time.TButton -text "+" -command "set_ldatetime_by_ymds 0 0 0 1"
    pack $top.bp_s1 $top.bp_s0 -side left -fill both -in $top.fr_bp_s
    frame $top.fr_bm_s
    ttk::button $top.bm_s1 -style Time.TButton -text "-" -command "set_ldatetime_by_ymds 0 0 0 -10"
    ttk::button $top.bm_s0 -style Time.TButton -text "-" -command "set_ldatetime_by_ymds 0 0 0 -1"
    pack $top.bm_s1 $top.bm_s0 -side left -fill both -in $top.fr_bm_s
    pack $top.fr_bp_s $top.fr_bm_s -expand 1 -fill both -in $top.fr_bpm_s

    pack $top.fr_setdate $top.fr_bpm_y $top.l_y_mm $top.fr_bpm_mm $top.l_mm_d $top.fr_bpm_d $top.l_d_h $top.fr_bpm_h $top.l_h_m $top.fr_bpm_m $top.l_m_s $top.fr_bpm_s -side left -expand 1 -fill both -in $top.lf_setdate

    pack $top.lf_setdate -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
    #.pw add $top.lf_setdate


    ########################################################################

    panedwindow $top.pw -orient vertical
    pack $top.pw -fill x -padx [scaled_px 3] -pady [scaled_px 2] -in $top.fr_root_left

    ########################################################################

    frame $top.fr_planetsinfoall

    ####

    labelframe $top.lf_planetsinfo0 -text "Planets Rise/Transit/Set" -font $font_labelframe

    #frame $top.fr_planetslbl0
    #label $top.l_planetslbl0  -text "    " -width 4
    #label $top.l_planetslbl0r -text " 出 " -width 5
    #label $top.l_planetslbl0s -text " 没 " -width 5
    #pack $top.l_planetslbl0 $top.l_planetslbl0r $top.l_planetslbl0s -side left -expand 1 -fill x -in $top.fr_planetslbl0

    frame $top.fr_mercurytime
    label $top.l_mercurytime0 -text "水星" -width 4 -font $font_small
    #label $top.l_mercurytime1 -text "出" -width 2 -font $font_small
    label $top.l_mercuryrise -textvariable buffer_mercuryrise -width 5 -font $font_small
    label $top.l_mercurytransit -textvariable buffer_mercurytransit -width 5 -font $font_small
    label $top.l_mercuryset  -textvariable buffer_mercuryset -width 5 -font $font_small
    #label $top.l_mercurytime2 -text "没" -width 2 -font $font_small
    pack $top.l_mercurytime0 $top.l_mercuryrise $top.l_mercurytransit $top.l_mercuryset -side left -expand 1 -fill x -in $top.fr_mercurytime

    frame $top.fr_venustime
    label $top.l_venustime0 -text "金星" -width 4 -font $font_small
    #label $top.l_venustime1 -text "出" -width 2 -font $font_small
    label $top.l_venusrise -textvariable buffer_venusrise -width 5 -font $font_small
    label $top.l_venustransit  -textvariable buffer_venustransit  -width 5 -font $font_small
    label $top.l_venusset  -textvariable buffer_venusset  -width 5 -font $font_small
    #label $top.l_venustime2 -text "没" -width 2 -font $font_small
    pack $top.l_venustime0 $top.l_venusrise $top.l_venustransit $top.l_venusset -side left -expand 1 -fill x -in $top.fr_venustime

    frame $top.fr_marstime
    label $top.l_marstime0 -text "火星" -width 4 -font $font_small
    #label $top.l_marstime1 -text "出" -width 2 -font $font_small
    label $top.l_marsrise -textvariable buffer_marsrise -width 5 -font $font_small
    label $top.l_marstransit -textvariable buffer_marstransit -width 5 -font $font_small
    label $top.l_marsset  -textvariable buffer_marsset  -width 5 -font $font_small
    #label $top.l_marstime2 -text "没" -width 2 -font $font_small
    pack $top.l_marstime0 $top.l_marsrise $top.l_marstransit $top.l_marsset -side left -expand 1 -fill x -in $top.fr_marstime

    ttk::button $top.b_plns_compact -style Small.TButton -text "コンパクト表示" -command "unset_detailed_planets_info"

    pack $top.fr_mercurytime $top.fr_venustime $top.fr_marstime $top.b_plns_compact -expand 0 -fill x -in $top.lf_planetsinfo0

    ####

    labelframe $top.lf_planetsinfo1 -text "Planets Rise/Transit/Set" -font $font_labelframe

    #frame $top.fr_planetslbl1
    #label $top.l_planetslbl1  -text "    " -width 4
    #label $top.l_planetslbl1r -text " 出 " -width 5
    #label $top.l_planetslbl1s -text " 没 " -width 5
    #pack $top.l_planetslbl1 $top.l_planetslbl1r $top.l_planetslbl1s -side left -expand 1 -fill x -in $top.fr_planetslbl1

    frame $top.fr_jupitertime
    label $top.l_jupitertime0 -text "木星" -width 6 -font $font_small
    #label $top.l_jupitertime1 -text "出" -width 2 -font $font_small
    label $top.l_jupiterrise -textvariable buffer_jupiterrise -width 5 -font $font_small
    label $top.l_jupitertransit  -textvariable buffer_jupitertransit  -width 5 -font $font_small
    label $top.l_jupiterset  -textvariable buffer_jupiterset  -width 5 -font $font_small
    #label $top.l_jupitertime2 -text "没" -width 2 -font $font_small
    pack $top.l_jupitertime0 $top.l_jupiterrise $top.l_jupitertransit $top.l_jupiterset -side left -expand 1 -fill x -in $top.fr_jupitertime

    frame $top.fr_saturntime
    label $top.l_saturntime0 -text "土星" -width 6 -font $font_small
    #label $top.l_saturntime1 -text "出" -width 2 -font $font_small
    label $top.l_saturnrise -textvariable buffer_saturnrise -width 5 -font $font_small
    label $top.l_saturntransit  -textvariable buffer_saturntransit  -width 5 -font $font_small
    label $top.l_saturnset  -textvariable buffer_saturnset  -width 5 -font $font_small
    #label $top.l_saturntime2 -text "没" -width 2 -font $font_small
    pack $top.l_saturntime0 $top.l_saturnrise $top.l_saturntransit $top.l_saturnset -side left -expand 1 -fill x -in $top.fr_saturntime

    frame $top.fr_uranustime
    label $top.l_uranustime0 -text "天王星" -width 6 -font $font_small
    #label $top.l_uranustime1 -text "出" -width 2 -font $font_small
    label $top.l_uranusrise -textvariable buffer_uranusrise -width 5 -font $font_small
    label $top.l_uranustransit  -textvariable buffer_uranustransit  -width 5 -font $font_small
    label $top.l_uranusset  -textvariable buffer_uranusset  -width 5 -font $font_small
    #label $top.l_uranustime2 -text "没" -width 2 -font $font_small
    pack $top.l_uranustime0 $top.l_uranusrise $top.l_uranustransit $top.l_uranusset -side left -expand 1 -fill x -in $top.fr_uranustime

    frame $top.fr_neptunetime
    label $top.l_neptunetime0 -text "海王星" -width 6 -font $font_small
    #label $top.l_neptunetime1 -text "出" -width 2 -font $font_small
    label $top.l_neptunerise -textvariable buffer_neptunerise -width 5 -font $font_small
    label $top.l_neptunetransit  -textvariable buffer_neptunetransit  -width 5 -font $font_small
    label $top.l_neptuneset  -textvariable buffer_neptuneset  -width 5 -font $font_small
    #label $top.l_neptunetime2 -text "没" -width 2 -font $font_small
    pack $top.l_neptunetime0 $top.l_neptunerise $top.l_neptunetransit $top.l_neptuneset -side left -expand 1 -fill x -in $top.fr_neptunetime

    pack $top.fr_jupitertime $top.fr_saturntime $top.fr_uranustime $top.fr_neptunetime -expand 0 -fill both -in $top.lf_planetsinfo1

    ####

    pack $top.lf_planetsinfo0 $top.lf_planetsinfo1 -side left -expand 1 -fill both -in $top.fr_planetsinfoall

    #pack $top.fr_planetsinfoall -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
    #.pw add $top.fr_planetsinfoall


    ########################################################################

    frame $top.fr_planetsinfo

    ####

    labelframe $top.lf00_planetsinfo -text "Planets Rise/Set" -font $font_labelframe
    #labelframe $top.lf00_planetsinfo -text "Planets Rise/Transit/Set" -font $font_labelframe

    ttk::button $top.b_plns_all -style Small.TButton -text "詳" -command "set_detailed_planets_info"

    ####

    frame $top.fr00_venustime
    label $top.l00_venustime0 -text " 金星" -width 5 -font $font_planets_rs
    label $top.l00_venusrise -textvariable buffer_venusrise -width 5 -font $font_planets_rs
    label $top.l00_venustransit  -textvariable buffer_venustransit  -width 5 -font $font_planets_rs
    label $top.l00_venusset  -textvariable buffer_venusset  -width 5 -font $font_planets_rs
    pack $top.l00_venustime0 $top.l00_venusrise $top.l00_venusset -side left -expand 1 -in $top.fr00_venustime
    #pack $top.l00_venustime0 $top.l00_venusrise $top.l00_venustransit $top.l00_venusset -side left -expand 1 -in $top.fr00_venustime

    frame $top.fr00_jupitertime
    label $top.l00_jupitertime0 -text " 木星" -width 5 -font $font_planets_rs
    label $top.l00_jupiterrise -textvariable buffer_jupiterrise -width 5 -font $font_planets_rs
    label $top.l00_jupitertransit  -textvariable buffer_jupitertransit  -width 5 -font $font_planets_rs
    label $top.l00_jupiterset  -textvariable buffer_jupiterset  -width 5 -font $font_planets_rs
    pack $top.l00_jupitertime0 $top.l00_jupiterrise $top.l00_jupiterset -side left -expand 1 -in $top.fr00_jupitertime
    #pack $top.l00_jupitertime0 $top.l00_jupiterrise $top.l00_jupitertransit $top.l00_jupiterset -side left -expand 1 -in $top.fr00_jupitertime

    frame $top.fr00_saturntime
    label $top.l00_saturntime0 -text " 土星" -width 5 -font $font_planets_rs
    label $top.l00_saturnrise -textvariable buffer_saturnrise -width 5 -font $font_planets_rs
    label $top.l00_saturntransit  -textvariable buffer_saturntransit  -width 5 -font $font_planets_rs
    label $top.l00_saturnset  -textvariable buffer_saturnset  -width 5 -font $font_planets_rs
    pack $top.l00_saturntime0 $top.l00_saturnrise $top.l00_saturnset -side left -expand 1 -in $top.fr00_saturntime
    #pack $top.l00_saturntime0 $top.l00_saturnrise $top.l00_saturntransit $top.l00_saturnset -side left -expand 1 -in $top.fr00_saturntime

    pack $top.b_plns_all -side left -expand 0 -fill x -in $top.lf00_planetsinfo
    pack $top.fr00_venustime $top.fr00_jupitertime $top.fr00_saturntime -side left -expand 1 -fill x -in $top.lf00_planetsinfo
    #pack $top.b_plns_all $top.fr00_jupitertime $top.fr00_saturntime -side left -expand 1 -fill x -in $top.lf00_planetsinfo

    ####

    pack $top.lf00_planetsinfo -side left -expand 1 -fill x -in $top.fr_planetsinfo

    $top.pw add $top.fr_planetsinfo


    ################################################################

    labelframe $top.lf_pcinfo -text "PC Info" -font $font_labelframe -fg $pcinfo_foreground

    frame $top.fr_pc_datetm
    frame $top.fr_pc_datetm_l0
    ttk::button $top.b_pcdatetm -style Small.TButton -width 8 -text "To Form" -command "set_pcdatetime"
    label $top.l_pcdatetm -text "PC日時" -width 8 -fg $pcinfo_foreground
    label $top.l_pcdate -textvariable buffer_pcdate_label -width 14 -fg $pcinfo_foreground
    label $top.l_pctime -textvariable buffer_pctime_label -width 12 -fg $pcinfo_foreground
    if { $flag_large_gui == 0 } {
      pack $top.l_pcdatetm -side left -expand 1 -fill x -in $top.fr_pc_datetm_l0
      pack $top.b_pcdatetm -side left -expand 0 -fill x -in $top.fr_pc_datetm_l0
      pack $top.fr_pc_datetm_l0 $top.l_pcdate $top.l_pctime -expand 0 -fill x -in $top.fr_pc_datetm
    } else {
      pack $top.b_pcdatetm $top.l_pcdatetm -side left -expand 0 -fill x -in $top.fr_pc_datetm_l0
      pack $top.fr_pc_datetm_l0 $top.l_pcdate $top.l_pctime -side left -expand 1 -fill x -in $top.fr_pc_datetm
    }

    pack $top.fr_pc_datetm -expand 1 -fill x -in $top.lf_pcinfo


    ################################################################

    labelframe $top.lf_scheduling -text "Scheduling" -font $font_labelframe

    ttk::checkbutton $top.cb_lightmoon -text "薄明/月明/光害を制御" -variable flag_light_moon_pollution -style "Scheduling.TCheckbutton"
    ttk::checkbutton $top.cb_astrodawn -text "朝薄明で太陽系ON" -variable flag_astronomical_dawn -style "Scheduling.TCheckbutton"

    pack $top.cb_lightmoon $top.cb_astrodawn -side left -fill x -in $top.lf_scheduling


    ########################################################################

    labelframe $top.lf_timelst -text "List of User's Time" -font $font_labelframe

    frame $top.fr_tml_lbl
    label $top.l_timelst -text "ユーザ時刻送信" -font $font_small
    ttk::button $top.b_tml_edit -style Small.TButton -text "編集" -command "open_text_editor .edt_stim $config_list_start_time"

    frame $top.fr_cb_tml_today
    ttk::checkbutton $top.cb_tml_today -text "本日日付も送信" -variable flag_send_today_with_users_time -style "Scheduling.TCheckbutton"
    pack $top.cb_tml_today -in $top.fr_cb_tml_today

    if { $flag_large_gui == 0 } {
      pack $top.l_timelst -expand 1 -side left -fill x -in $top.fr_tml_lbl
      pack $top.b_tml_edit -expand 0 -side left -fill x -in $top.fr_tml_lbl
      pack $top.fr_tml_lbl -expand 0 -fill x -in $top.lf_timelst
      pack $top.fr_cb_tml_today -expand 0 -fill x -in $top.lf_timelst
    } else {
      pack $top.l_timelst -expand 1 -side left -fill x -in $top.fr_tml_lbl
      pack $top.fr_cb_tml_today -expand 1 -side left -fill x -in $top.fr_tml_lbl
      pack $top.b_tml_edit -expand 0 -side left -fill x -in $top.fr_tml_lbl
      pack $top.fr_tml_lbl -expand 0 -fill x -in $top.lf_timelst
    }

    frame $top.fr_tml_margin -height [scaled_px 2]
    pack $top.fr_tml_margin -expand 0 -fill x -in $top.lf_timelst

    #

    array set timelist {}

    if { [catch {open $config_list_start_time r} fid] } {
      # do nothing
    } else {
      set i 0
      while { [gets $fid line] >= 0 } {
        set s [string trim $line]
        if { [string length $s] != 0 && [string index $s 0] ne "#" } {
          set status [regexp -- {^([0-9]+)[ ]*[:][ ]*([0-9]+)} $s match h0 m0]
          if { $status == 1 } {
            set h1 [expr $h0 % 24]
            set m1 [expr $m0 % 60]
            set timelist($i) [format "%2d:%02d" $h1 $m1]
            incr i
          } else {
            set status [regexp -- {^([A-Za-z]+)[ ]*} $s match s0]
            if { $status == 1 } {
              if { [string tolower $s0] eq "now" } {
                set timelist($i) "-"
                incr i
              }
            }
          }
        }
      }
      close $fid
    }

    if { [array size timelist] == 0 } {
      set timelist(0) "-"
      set timelist(1) "20:00"
    }

    #

    if { $flag_large_gui == 0 } {
      set len_col 3
    } else {
      set len_col 8
    }
    set len_timelist [array size timelist]
    set fr_id 0
    set m [expr $len_timelist % $len_col]

    if { $m == 0 } {
      set i_len $len_timelist
    } else {
      set i_len [expr $len_timelist + $len_col - $m]
    }

    for {set i 0} {$i < $i_len} {incr i} {

      set m [expr $i % $len_col]
      if { $m == 0 } {
        set fr_id [expr $i / $len_col]
        frame $top.fr_tml_$fr_id
      }

      if { $i < $len_timelist } {
        if { $timelist($i) eq "-" } {
          ttk::button $top.b_tml_$i -style Small.TButton -text " NOW " -command "set_start_time $timelist($i)"
        } else {
          ttk::button $top.b_tml_$i -style Small.TButton -text $timelist($i) -command "set_start_time $timelist($i)"
        }
      } else {
        ttk::button $top.b_tml_$i -style Small.TButton -text "--:--"
      }
      pack $top.b_tml_$i -expand 1 -side left -fill x -in $top.fr_tml_$fr_id

      incr m
      if { $m == $len_col } {
        pack $top.fr_tml_$fr_id -expand 0 -fill x -in $top.lf_timelst
      }

    }

    ################################################################

    labelframe $top.lf_internal -text "Internal Info" -font $font_labelframe

    entry $top.e_debug -textvariable buffer_internal_info_dg -font $font_internal
    pack $top.e_debug -expand 1 -fill x -in $top.lf_internal


    ########################################################################

    # PC日時、時刻設定ボタンの配置
    # - Normal GUI モード : 右側に縦長で表示
    # - Large GUI モード  : 下側に横長で表示

    if { $flag_large_gui == 0 } {
      if { $flag_host_gui != 0 } {
        pack $top.lf_scheduling -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
      }
      pack $top.lf_internal -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
      # 下記行の -fill y が無いと、惑星表示の「詳」「コンパクト表示」切り替え時に GUI 表示が乱れる
      pack $top.fr_root_left -expand 0 -fill y -side left -padx 0 -pady 0 -in $top.fr_rootwin
      pack $top.lf_pcinfo -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_right
      pack $top.lf_timelst -expand 1 -fill both -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_right
      pack $top.fr_root_right -expand 0 -fill y -side left -padx 0 -pady 0 -in $top.fr_rootwin
    } else {
      pack $top.lf_pcinfo -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
      if { $flag_host_gui != 0 } {
        pack $top.lf_scheduling -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
      }
      pack $top.lf_timelst -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
      pack $top.lf_internal -expand 0 -fill x -padx [scaled_px 4] -pady [scaled_px 3] -in $top.fr_root_left
      pack $top.fr_root_left -expand 0 -padx 0 -pady 0 -in $top.fr_rootwin
    }

    ########################################################################

    pack $top.fr_rootwin

  }

  return

}


########################################################################

if { $flag_host_gui != 0 } {

  set iscserv_root_directory ""

  # ISC Server の root ディレクトリを取得する
  do_http_get $iscserv_url ret buffer_internal_info_dg
  set lst [split [string trim $ret] ","]

  if { 1 < [llength $lst] } {
    set iscserv_root_directory [string trim [lindex $lst 1]]
    set buffer_internal_info_dg "iscserv_root_directory=$iscserv_root_directory"
  }

  # リモートGUIから参照されるカウンタをリセットする
  if { $iscserv_root_directory ne "" } {
    save_text_file "$counter_sunplanets_info\n" $iscserv_root_directory/_tmp_count_sunplanets_info.isc.txt
    save_text_file "$counter_moon_info\n" $iscserv_root_directory/_tmp_count_moon_info.isc.txt
  } else {
    wm state . normal
    tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot get iscserv_root_directory."
    exit
  }

}

########################################################################


# hide default wish window
wm state . withdrawn

open_date_gui ""

# display main window
wm state . normal


# 無限ループ
show_date_time_info

