#########################################################################
##                                                                     ##
##  I N T E R S T E L L A R   C O N S O L E  Main GUI                  ##
##            Copyright (C) 2025- Misato Observatory, Wakayama, Japan  ##
##                                                                     ##
##  This software is available under the GPL (General Public License)  ##
##                                                                     ##
##       Tcl/Tk References: http://www.nct9.ne.jp/m_hiroi/tcl_tk_doc/  ##
##                                                                     ##
#########################################################################

#
#                             Block Diagram
#
# +-----------------+       ______________      +------------------+
# |                 |      /              |     |                  |
# | ISC Info Server | <-- | Config/Status | <-- |                  |
# |  (iscserv.tcl)  |     | of Stellarium |     |    Stellarium    |
# |                 |     |_______________|     |                  |
# +-----------------+    %HOMEDRIVE%%HOMEPATH%/ |                  |
#          |             AppData/Roaming/       +------------------+
#          |             Stellarium               |  A      |
#          |                                Status|  |      +------> Projector
#          |Results of Scripts         (port:8090)|  |
#          |(port:8091)                           |  |Script(port:8090)
#          |                                      V  |
#          |         +--------------------------------------+
#          +-------> |                                      ||
#     _________      |              ISC GUIs                ||
#    /         |     | (misato-gui.tcl, date-gui.tcl, etc.) || -----> Display
#   |  Config  | --> |                                      ||
#   |  of ISC  |     |______________________________________||
#   |__________|       --------------------------------------+
#   %PROGRAMFILES%/
#   Stellarium/scripts/
#   interstellar-console
#


####################################################################

#
# サーバURL, OS依存の定数, GUIの基本設定(フォント,色)をセット
#
source local-config.tcl

#
# 基本的procをインポートする
#
source base-proc.tcl


####################################################################
##                                                                ##
##              G E N E R I C   P R O C E D U R E S               ##
##                                                                ##
####################################################################

# 天体情報取得の時だけ利用（http::geturlはiscserv.tclに対して使用不可）
#package require http

#
# ISC用 基本的procをインポートする
#
source isc-proc.tcl

#
# ボタンウィジェット作成用 wrapper proc
# MacOS のボタンがダメダメな件の対策
#

# ttkのbuttonを使う場合は1 (これはもう変更ないだろう)
set flag_using_ttk_button 1

#
# ボタンの作成 (ノーマルサイズ)
#
proc create_button { arg_id arg_text arg_command } {

  global flag_using_ttk_button

  if { $flag_using_ttk_button == 0 } {
    # 通常のボタンを使用する (MacOSだと色が壊れる)
    button $arg_id -text $arg_text -command $arg_command
  } else {
    # ttkのボタンを使用する (MacOSでもOK)
    ttk::button $arg_id -text $arg_text -command $arg_command
  }

}

#
# ボタンの作成 (smallサイズ)
#
proc create_button_small { arg_id arg_text arg_command } {

  global flag_using_ttk_button
  global font_small

  if { $flag_using_ttk_button == 0 } {
    # 通常のボタンを使用する (MacOSだと色が壊れる)
    button $arg_id -font $font_small -text $arg_text -command $arg_command
  } else {
    # ttkのボタンを使用する (MacOSでもOK)
    ttk::button $arg_id -style "Small.TButton" -text $arg_text -command $arg_command
  }

}

#
# ボタンの作成 (code用)
#
proc create_button_code { arg_id arg_text arg_command } {

  global flag_using_ttk_button
  global font_codefield

  if { $flag_using_ttk_button == 0 } {
    # 通常のボタンを使用する (MacOSだと色が壊れる)
    button $arg_id -font $font_codefield -text $arg_text -command $arg_command
  } else {
    # ttkのボタンを使用する (MacOSでもOK)
    ttk::button $arg_id -style "Code.TButton" -text $arg_text -command $arg_command
  }

}

#
# タブとして使うボタンの状態を変化させる
#
proc configure_tab_button { arg_id flag_selected } {

  global flag_using_ttk_button
  global tab_background
  global tab_foreground
  global tab_selected_background
  global tab_selected_foreground

  if { $flag_using_ttk_button == 0 } {
    if { $flag_selected == 0 } {
      $arg_id configure -bg $tab_background -fg $tab_foreground -relief raised
    } else {
      $arg_id configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
    }
  } else {
    if { $flag_selected == 0 } {
      $arg_id configure -style TButton
    } else {
      $arg_id configure -style Selected.TButton
    }
  }

}

#
# 汎用: stellariumサーバ上の指定されたスクリプト(foo.ssc)を実行 (可変引数)
#
proc exec_ssc_on_server { args } {

  global stellarium_run_url stellarium_script_dir
  global buffer_codefield

  foreach script_file $args {

    set buffer_codefield "$script_file"

    do_http_post $stellarium_run_url "id=$stellarium_script_dir/$script_file" ret buffer_codefield
    set status [string trim $ret]

    if { $status ne "ok" } {
      set buffer_codefield "$script_file / ERROR: $status"
    }

  }

}

#
# 汎用: stellariumスクリプトn行 を送信 (可変引数)
#
#  【注意】
#    ・args[n] に空白を入れたい場合は [~] を、["] を入れたい場合は ['] とする。
#      例: "Ursa Major" は 'Ursa~Major' とする
#    ・送信できるのは、関数の羅列だけである。
#      Stellariumがクラッシュするため、for()文などの制御構文を送ってはならない。
#
proc send_ssc_lines { args } {

  global stellarium_direct_url
  global buffer_codefield

  foreach code1line $args {

    # replace [~] with [ ], and ['] with ["]
    set code_p0 [regsub -all {~} $code1line { }]
    set code_p1 [regsub -all {'} $code_p0 {"}]
    set str_code [string trim $code_p1]

    set buffer_codefield $str_code

    # send a code using POST
    do_http_post $stellarium_direct_url "code=$str_code" ret buffer_codefield

  }

}

#
# 超汎用: stellariumスクリプトn行 を送信
#         または
#         サーバ上の指定されたスクリプト(foo.ssc)を実行 (可変引数)
#
proc send_or_exec_ssc { args } {

  foreach code $args {

    set code_case [string tolower $code]
    if { [string match "*.ssc" $code_case] } {
      exec_ssc_on_server $code
    } else {
      send_ssc_lines $code
    }

  }

}

#
# 汎用: filename_text を読み込み、1行=1要素の配列を array_ret に返す
#
proc get_line_array_from_text_file { filename_text array_ret } {

  upvar $array_ret array_ref

  array set array_ref {}

  if { [catch {open $filename_text r} fid] } {
    # do nothing
  } else {
    set i 0
    while { [gets $fid line] >= 0 } {
      set s [string trim $line]
      # skip comments
      if { [string length $s] != 0 && [string index $s 0] ne "#" } {
        set array_ref($i) $s
        incr i
      }
    }
    close $fid
  }

}

#
# 汎用: ISC Server からファイル filename にある値をGETし、
#       連想配列を作って asarr_ret に返す
#       返り値はステータス  0: 正常  non-zero: エラー
#
proc get_data_from_iscserv { filename asarr_ret } {

  global iscserv_url
  global buffer_codefield
  upvar $asarr_ret asarr_ref

  set return_val -1

  do_http_get $iscserv_url/$filename ret_string buffer_codefield

  if { $ret_string eq "" } {
    # ERROR
    # wm state . normal
    # tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET $iscserv_url/$filename"
  } else {
    set lines [split $ret_string "\n"]
    set len_lines [llength $lines]
    set flg_get 0
    for {set i 0} {$i < $len_lines} {incr i} {
      set txt_line [lindex $lines $i]
      set lst [split [string trim $txt_line] ","]
      if { [llength $lst] == 2 } {
        set k [string trim [lindex $lst 0]]
        set v [string trim [lindex $lst 1]]
        set asarr_ref($k) $v
        set flg_get 1
      }
    }
    if { $flg_get != 0 } {
      set return_val 0
    } else {
      # ERROR
      # wm state . normal
      # tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET $iscserv_url/$filename"
    }
  }

  return $return_val

}


####################################################################
##                                                                ##
##                      P R O C E D U R E S                       ##
##                                                                ##
####################################################################

#
# １つの星座のセレクト/アンセレクト
#
#  【注意】name に空白を入れたい場合は ~ とすること
#          例: "Ursa Major" は "Ursa~Major" とする
#
proc update_constellation { name flg } {

  global stellarium_direct_url
  global buffer_codefield

  # replace '~' with ' '
  set arg_name [regsub -all {~} $name { }]

  if { [string tolower $flg] eq "true" } {
    set str_code "ConstellationMgr.selectConstellation(\"$arg_name\")"
  } else {
    set str_code "ConstellationMgr.deselectConstellation(\"$arg_name\")"
  }

  set buffer_codefield $str_code

  # send a code using POST
  do_http_post $stellarium_direct_url "code=$str_code" ret buffer_codefield

}

#
# 1つまたは複数の星ラベルのON
#
proc set_star_label { args } {

  global asarr_star_label

  if { [array size asarr_star_label] == 0 } {
    initialize_star_labels
  }

  foreach name $args {
    send_ssc_lines "LabelMgr.setLabelShow($asarr_star_label($name),true)"
  }

}

#
# 1つまたは複数の星ラベルのOff
#
proc unset_star_label { args } {

  global asarr_star_label

  if { [array size asarr_star_label] == 0 } {
    initialize_star_labels
  }

  foreach name $args {
    send_ssc_lines "LabelMgr.setLabelShow($asarr_star_label($name),false)"
  }

}

#                                        #
# ここからのprocは特定のウィジェット専用 #
#                                        #

#
# filename_text を読み込み、Objects n フレームでのボタンを生成
#
#                                                        "spring" "p1"
proc build_dso_buttons { filename_text proc_name len_col sufx0    lf_sufx } {

  global font_small
  array set lines {}

  get_line_array_from_text_file $filename_text lines

  set len_lines [array size lines]

  #set len_col 3
  set i 0
  set j 0
  for {set i 0} {$i < $len_lines} {incr i} {
    if { [expr $i % $len_col] == 0 } {
      frame .fr_dso_l$j$sufx0
    }
    set lst [split $lines($i) ","]
    if { 1 < [llength $lst] } {
      set s0 [string trim [lindex $lst 0]]
      set s1 [string trim [lindex $lst 1]]
      set s1 [regsub -all { } $s1 {~}]
      create_button_small .b_dso_$i$sufx0 "$s0" "$proc_name $s1"
      pack .b_dso_$i$sufx0 -side left -expand 1 -fill x -in .fr_dso_l$j$sufx0
    } else {
      if { $lines($i) ne "-" } {
        set s0 [string trim $lines($i)]
        set s1 [regsub -all { } $s0 {~}]
        create_button_small .b_dso_$i$sufx0 "$s0" "$proc_name $s1"
        pack .b_dso_$i$sufx0 -side left -expand 1 -fill x -in .fr_dso_l$j$sufx0
      }
    }
    if { [expr ($i + 1) % $len_col] == 0 } {
      pack .fr_dso_l$j$sufx0 -expand 0 -fill x -in .lf_dso_$lf_sufx
      incr j
    }
  }
  if { [expr $i % $len_col] != 0 } {
    pack .fr_dso_l$j$sufx0 -expand 0 -fill x -in .lf_dso_$lf_sufx
  }

}

#
# ISC Server から 星、天の川等の現在の値をGETし、
# 連想配列 asarr_initial_settings に格納する
#
#proc get_current_settings {} {
#
#  global asarr_initial_settings
#
#  # asarr_initial_settings is updated
#  get_data_from_iscserv $txt_current_settings_isc asarr_initial_settings
#
#}

#
# ISC Server から 星、天の川等の現在の値をGETし、
# 連想配列 asarr_ret に返す
#
proc get_current_settings_to { asarr_ret } {

  global txt_current_settings_isc
  upvar $asarr_ret asarr_ref

  set s [get_data_from_iscserv $txt_current_settings_isc asarr_ref]

  if { $s != 0 } {
    wm state . normal
    tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET $txt_current_settings_isc"
  }

}

#
# ISC Server から MinFps の現在の値をGETし、
# 連想配列 asarr_ret に返す
#
proc get_current_min_fps_to { asarr_ret } {

  upvar $asarr_ret asarr_ref

  set s [get_data_from_iscserv "_tmp_currentMinFps.isc.txt" asarr_ref]

  if { $s != 0 } {
    wm state . normal
    tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET _tmp_currentMinFps.isc.txt"
  }

}

#
# ISC Server から 星、天の川等の初期値をGETし、
# 連想配列 asarr_ret に返す
#
proc get_initial_settings_to { asarr_ret } {

  upvar $asarr_ret asarr_ref

  set s [get_data_from_iscserv "_tmp_initialSettings.isc.txt" asarr_ref]

  if { $s != 0 } {
    wm state . normal
    tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET _tmp_initialSettings.isc.txt"
  }

}

#
# Stellarium から現在の動作モード(Dome or Navi)を取得
# 0: Navigationモード  1: Domeモード
#
proc get_flag_dome_mode_on_server {} {

  global ssc_get_current_settings

  # FlagDomeMode についてはサーバから最新情報を取得
  exec_ssc_on_server $ssc_get_current_settings
  after 500
  array set asarr_cnt_settings {}
  get_current_settings_to asarr_cnt_settings

  set v 0
  if { [string tolower $asarr_cnt_settings(FlagDomeMode)] eq "true" } {
    set v 1
  }

  return $v

}

#
# ISC Server 上で初期値を保存
#
proc save_current_to_initial_settings_on_iscserv {} {

  global txt_current_settings_isc
  global flag_host_gui
  global buffer_codefield
  global iscserv_url

  if { $flag_host_gui != 0 } {

    # ISC Server の root ディレクトリを取得する
    do_http_get $iscserv_url ret buffer_codefield
    set lst [split [string trim $ret] ","]

    if { 1 < [llength $lst] } {
      set iscserv_root [string trim [lindex $lst 1]]
      set buffer_codefield "iscserv_root=$iscserv_root"

      # 設定ファイルをコピーする
      set buffer_codefield "copy: $txt_current_settings_isc -> _tmp_initialSettings.isc.txt"
      file copy -force $iscserv_root/$txt_current_settings_isc $iscserv_root/_tmp_initialSettings.isc.txt
    }

  }

}


#
# ISC Server からラベルIDをGETし、連想配列 asarr_star_label を作る
#
proc get_star_label_id {} {

  global asarr_star_label

  # asarr_star_label is updated
  set s [get_data_from_iscserv "_tmp_starLabels.isc.txt" asarr_star_label]

  return $s

}

#
# 星ラベル(stellarium・GUI)の初期化 (更新には両サーバの再起動が必須)
#
# _tmp_starLabels.isc.txt は host 起動時に消去され、
# １度だけ生成される
#
proc initialize_star_labels {} {

  set s [get_star_label_id]
  if { $s != 0 } {
    send_ssc_lines "LabelMgr.deleteAllLabels()"
    exec_ssc_on_server createStarLabels.ssc
  }
  after 500
  set s [get_star_label_id]
  if { $s != 0 } {
    wm state . normal
    tk_messageBox -type ok -icon info -title "ERROR" -message "ERROR: cannot GET _tmp_starLabels.isc.txt"
  }

}

#
# GUI 初期化 (Configuration/Maintenance)
#
proc initialize_config_gui_params {} {

  send_ssc_of_projection_mode /
  send_ssc_of_display_gamma /
  send_ssc_of_view_fov /
  send_ssc_of_view_altitude /
  send_ssc_of_view_azimuth /
  send_ssc_of_viewport_x /
  send_ssc_of_viewport_y /
  send_ssc_of_disk_viewport /
  send_ssc_of_dso_fov_circular_marker /

}

#
# GUI 初期化 (すべて)
#
proc initialize_all_gui_params {} {

  send_ssc_of_timerate /
  send_ssc_of_milkyway /
  send_ssc_of_relative_star_scale /
  send_ssc_of_absolute_star_scale /
  send_ssc_of_landscape_transparency /
  send_ssc_of_meteor_zhr /
  send_ssc_of_min_fps /
  send_ssc_of_location_latitude /
  send_ssc_of_stellarlabels /
  send_ssc_of_solsyslabels /
  send_ssc_of_dsolabels /
  send_ssc_of_dsohints /

  initialize_config_gui_params

}

#
# 現在のstellarium上の設定値を取得し、GUIに反映
#
proc reload_all_gui_params {} {

  global ssc_get_current_settings
  global asarr_initial_settings

  # backup
  set tmp_array_list [array get asarr_initial_settings]

  exec_ssc_on_server $ssc_get_current_settings
  after 500
  get_current_settings_to asarr_initial_settings
  initialize_all_gui_params

  # restore
  array set asarr_initial_settings $tmp_array_list

}

#
# VIEW の値をテストする。NaN が見つかれば non-0 を返す
#
proc test_view_prms {} {

  global ssc_get_current_settings
  #global buffer_codefield

  array set asarr_current_settings {}

  # 設定値を取得
  after 500
  exec_ssc_on_server $ssc_get_current_settings
  after 500
  get_current_settings_to asarr_current_settings

  #set buffer_codefield ""

  set flg_found 0
  foreach {i} [array name asarr_current_settings] {
    set v $asarr_current_settings($i)
    # debug...
    #append buffer_codefield "$v,"
    #if { $i eq "TestKey" } {
    #  tk_dialog .dlg_test "TEST" "i,v = ($i),($v)" "" 0 "Yes"
    #}
    if { [string tolower $v] eq "nan" } {
      set flg_found 1
      break
    }
  }

  return $flg_found

}

#
# ビュー(stellarium・GUI)初期化
#  arg_flg_init_config_gui_prms: 1 をセットすると、最後に Configurationフレーム内パラメータを初期化
#  arg_displaying_dlg: ダイアログを表示中にこのprocを呼ぶ場合にその window ID をセット
#
proc send_ssc_to_initialize_view { arg_flg_init_config_gui_prms arg_displaying_dlg } {

  #global ssc_get_current_settings
  #global buffer_codefield

  # Recovering from view initialization failures

  set count_retry 0
  while { 1 } {

    exec_ssc_on_server localInitView.ssc

    # もし値に NaN が見つかった場合、2秒おきに 5回初期化をリトライする
    if { [test_view_prms] == 0 } {
      break
    } elseif { $count_retry < 5 } {
      after 2000
      incr count_retry
    } else {
      # このダイアログが出ることは、まず無いであろう
      if { $arg_displaying_dlg ne "-" } {
        set junk [catch {destroy $arg_displaying_dlg} wid]
      }
      set ans_retly [isc_dialog .dlg_retly_init "Question" "Viewの初期化に失敗したようです。再試行しますか？" 0 ""  0 "Yes" "No" "" ""]
      if { $ans_retly != 0 } {
        break
      }
      set count_retry 0
    }

  }

  # initialize GUI
  if { $arg_flg_init_config_gui_prms != 0 } {
    initialize_config_gui_params
  }

}

#
# 全(stellarium・GUI)初期化
#  flag_dialog_and_fast_init: 確認ダイアログ＆速い初期化の有無 (non-0 なら有)
#
# VIEWを安定して初期化するのが難しいため、最初と最後で実施している。
#
proc send_ssc_to_initialize_all { flag_dialog_and_fast_init } {

  global flag_host_gui
  global flag_dome_mode
  global flag_light_pollution_after_initialization
  global asarr_initial_settings

  if { $flag_dialog_and_fast_init != 0 } {
    set ans_do [isc_dialog .dlg_initialize_all "Question" "全初期化を実行しますか？" 0 ""  1 "全初期化する" "操作へ戻る" "" ""]
    if { $ans_do == 1 } {
      return
    }
    set ans [isc_dialog .dlg_initializing "Info" "    全初期化中 ...    " 0 ""  0 "" "" "" ""]
  }

  # To stop Scheduling in date-gui.tcl
  send_ssc_lines core.setTimeRate(0.0) core.setMinFps(4.0)

  # 以前のモードの方位表示を消去しておく
  exec_ssc_on_server unsetCardinal.ssc

  # 動作モードをセット。最初にこれが必須
  if { $flag_host_gui != 0 } {
    # set mode
    if { $flag_dome_mode == 0 } {
      exec_ssc_on_server unsetFlagDomeMode.ssc
    } else {
      exec_ssc_on_server setFlagDomeMode.ssc
    }
  }

  # Ver.1.3
  # ここは本番のVIEW初期化ではない
  after 500
  exec_ssc_on_server localInitView.ssc
  after 500
  if { $flag_dialog_and_fast_init == 0 } {
    after 1000
  }

  # Mask stars ...
  send_ssc_lines MilkyWay.setIntensity(0.1) StelSkyDrawer.setFlagStarMagnitudeLimit(true) StelSkyDrawer.setCustomStarMagnitudeLimit(1.0) StelSkyDrawer.setAbsoluteStarScale(0.1)

  # initialize and create lables
  initialize_star_labels

  # initialize location
  after 100
  exec_ssc_on_server localInitLocation.ssc

  # Mask stars ...
  send_ssc_lines StelSkyDrawer.setLightPollutionLuminance(0.01)

  # Ver.1.2 まではここでも初期化していたが、↑へ移動
  #after 100
  # ここは本番のVIEW初期化ではない
  #exec_ssc_on_server localInitView.ssc

  after 200
  exec_ssc_on_server localInitSettings.ssc

  after 200
  if { $flag_light_pollution_after_initialization != 0 } {
    exec_ssc_on_server setLightPollution.ssc
  } else {
    exec_ssc_on_server unsetLightPollution.ssc
  }

  # 本番のVIEW初期化
  # (Recovering from view initialization failures)
  # ここでは getCurrentSettings.ssc も実行される
  after 500
  if { $flag_dialog_and_fast_init == 0 } {
    after 1000
    send_ssc_to_initialize_view 0 -
  } else {
    send_ssc_to_initialize_view 0 .dlg_initializing
  }

  get_current_settings_to asarr_initial_settings

  initialize_all_gui_params

  if { $flag_host_gui != 0 } {
    save_current_to_initial_settings_on_iscserv
  }

  # 「全初期化中...」ウィンドウのクローズ
  if { $flag_dialog_and_fast_init != 0 } {
    set junk [catch {destroy .dlg_initializing} wid]
  }

  return

}

#
# stellariumスクリプトを送信または実行
# (SSC送信コード入力フィールド専用)
#
proc send_ssc_from_codefield {} {

  global buffer_codefield

  set buffer_codefield [string trim $buffer_codefield]

  if { ![string match "http*:*" $buffer_codefield] } {
    send_or_exec_ssc "$buffer_codefield"
  }

}

#
# 時刻レートを stellariumサーバへ送信
#
proc set_timerate { prm } {

  global buffer_timerate

  set buffer_timerate [format "%.1f" $prm]

  send_ssc_lines "core.setTimeRate($buffer_timerate)"

}

#
# send_ssc_of_xxxx 共通の仕様:
#
#  ・"/" が与えられた場合、初期値を GUI にセットし、stellarium に送信しない
#  ・"-" が与えられた場合、0もしくは初期値(ボタンによる)をセットし、stellarium に送信する
#

#
# 時刻レートを stellariumサーバへ送信
#
proc send_ssc_of_timerate { args } {

  global buffer_timerate
  global asarr_initial_settings

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "/" } {
      set v $asarr_initial_settings(TimeRate)
    } elseif { $prm eq "-" } {
      set v 0.0
    } else {
      set v [expr $buffer_timerate + $prm]
    }
    set buffer_timerate $v
  }

  set v [string trim $buffer_timerate]
  set buffer_timerate [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "core.setTimeRate($buffer_timerate)"
  }

}

#
# 時刻を stellariumサーバへ送信
#
proc send_ssc_from_timearg { args } {

  global buffer_datearg

  set tm_strreal [clock format [clock seconds] -format {%Y-%m-%dT%H:%M:%S}]
  set tm_str0 [clock format [clock seconds] -format {%Y-%m-%dT}]

  set buffer_datearg $tm_strreal

  foreach prm $args {
    set buffer_datearg $tm_str0
    append buffer_datearg $prm
  }

  send_ssc_lines "core.setDate('$buffer_datearg','local')"

}

#
# 時刻設定sscスクリプトを buffer_codefield にセット
#
proc set_buffer_codefield_from_buffer_datearg {} {

  global buffer_codefield
  global buffer_datearg

  set buffer_codefield "core.setDate(\"$buffer_datearg\",\"local\")"

}

#
# 散在流星ZHRを stellariumサーバへ送信
#
proc send_ssc_of_meteor_zhr { args } {

  global buffer_meteor_zhr
  global asarr_initial_settings

  set flg_show "true"
  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "/" } {
      set v $asarr_initial_settings(SporadicMeteorMgr.ZHR)
    } elseif { $prm eq "-" } {
      set v 0
      set flg_show "false"
    } else {
      set v [expr $buffer_meteor_zhr + $prm]
    }
    set buffer_meteor_zhr $v
  }

  set v [string trim $buffer_meteor_zhr]
  if { $v < 0.0 } {
    set v 0
  }
  set buffer_meteor_zhr [format "%.0f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "SporadicMeteorMgr.setFlagShow($flg_show)"
    send_ssc_lines "SporadicMeteorMgr.setZHR($buffer_meteor_zhr)"
  }

}

#
# 星ラベルのフラグ・量を stellariumサーバへ送信
#
proc send_ssc_of_stellarlabels { args } {

  global buffer_stellarlabels
  global asarr_initial_settings

  set flg_labels "true"
  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "/" } {
      set v $asarr_initial_settings(StarMgr.labelsAmount)
    } elseif { $prm eq "-" } {
      set v 0.0
      set flg_labels "false"
    } else {
      set v [expr $buffer_stellarlabels + $prm]
    }
    set buffer_stellarlabels $v
  }

  set v [string trim $buffer_stellarlabels]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_stellarlabels [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StarMgr.setFlagLabels($flg_labels)"
    send_ssc_lines "StarMgr.setLabelsAmount($buffer_stellarlabels)"
  }

}

#
# 太陽系を非表示にする
#
proc unset_solarsystem { args } {

  unset_star_label Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto

  set flg 0
  foreach prm $args {
    exec_ssc_on_server unsetSolarSystemLightPollution.ssc
    set flg 1
  }

  if { $flg == 0 } {
    exec_ssc_on_server unsetSolarSystem.ssc updateLightMoonPollution.ssc
  }

}

#
# 太陽系ラベルのフラグ・量を stellariumサーバへ送信
#
proc send_ssc_of_solsyslabels { args } {

  global buffer_solsyslabels
  global asarr_initial_settings

  set flg_labels "true"
  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "/" } {
      set v $asarr_initial_settings(SolarSystem.labelsAmount)
    } elseif { $prm eq "-" } {
      set v 0.0
      set flg_labels "false"
    } else {
      set v [expr $buffer_solsyslabels + $prm]
    }
    set buffer_solsyslabels $v
  }

  set v [string trim $buffer_solsyslabels]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_solsyslabels [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "SolarSystem.setFlagLabels($flg_labels)"
    send_ssc_lines "SolarSystem.setLabelsAmount($buffer_solsyslabels)"
  }

}

#
# 天の川の光量を stellariumサーバへ送信
#
proc send_ssc_of_milkyway { args } {

  global asarr_initial_settings
  global buffer_milkyway

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(MilkyWay.intensity)
    } else {
      set v [expr $buffer_milkyway + $prm]
    }
    set buffer_milkyway $v
  }

  set v [string trim $buffer_milkyway]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_milkyway [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "MilkyWay.setIntensity($buffer_milkyway)"
  }

}

#
# 星表現 相対パラメータを stellariumサーバへ送信
#
proc send_ssc_of_relative_star_scale { args } {

  global asarr_initial_settings
  global buffer_relative_star_scale

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(StelSkyDrawer.relativeStarScale)
    } else {
      set v [expr $buffer_relative_star_scale + $prm]
    }
    set buffer_relative_star_scale $v
  }

  set v [string trim $buffer_relative_star_scale]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_relative_star_scale [format "%.2f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelSkyDrawer.setRelativeStarScale($buffer_relative_star_scale)"
  }

}

#
# 星表現 絶対パラメータを stellariumサーバへ送信
#
proc send_ssc_of_absolute_star_scale { args } {

  global asarr_initial_settings
  global buffer_absolute_star_scale

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(StelSkyDrawer.absoluteStarScale)
    } else {
      set v [expr $buffer_absolute_star_scale + $prm]
    }
    set buffer_absolute_star_scale $v
  }

  set v [string trim $buffer_absolute_star_scale]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_absolute_star_scale [format "%.2f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelSkyDrawer.setAbsoluteStarScale($buffer_absolute_star_scale)"
  }

}

#
# 地上風景透過パラメータを stellariumサーバへ送信
#
proc send_ssc_of_landscape_transparency { args } {

  global asarr_initial_settings
  global buffer_landscape_transparency

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(LandscapeMgr.landscapeTransparency)
    } else {
      set v [expr $buffer_landscape_transparency + $prm]
    }
    set buffer_landscape_transparency $v
  }

  set v [string trim $buffer_landscape_transparency]
  if { $v < 0.0 } {
    set v 0.0
  } elseif { 1.0 < $v } {
    set v 1.0
  }
  set buffer_landscape_transparency [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "LandscapeMgr.setLandscapeTransparency($buffer_landscape_transparency)"
  }

}

#
# 方角切替スピードのパラメータを stellariumサーバへ送信
#
proc update_azchange_duration { args } {

  global buffer_azchange_duration

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 5
    } else {
      set v [expr $buffer_azchange_duration + $prm]
    }
    set buffer_azchange_duration $v
  }

  set v [string trim $buffer_azchange_duration]
  if { 99 < $v } {
    set v 99
  } elseif { $v < 0 } {
    set v 0
  }
  set buffer_azchange_duration [format "%d" $v]

}

#
# 観測地の緯度を stellariumサーバへ送信
#
proc send_ssc_of_location_latitude { args } {

  global asarr_initial_settings
  global buffer_location_latitude

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(Latitude)
    } else {
      set v [expr $buffer_location_latitude + $prm]
    }
    set buffer_location_latitude $v
  }

  set v [string trim $buffer_location_latitude]
  if { 90.0 < $v } {
    set v 90.0
  } elseif { $v < -90.0 } {
    set v -90.0
  }
  set buffer_location_latitude [format "%.8f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "core.setObserverLocation($asarr_initial_settings(Longitude),$buffer_location_latitude,$asarr_initial_settings(Altitude),0.0,'Our Observatory','Earth')"
  }

}

#
# 最小FPS値を stellariumサーバへ送信
#
proc send_ssc_of_min_fps { args } {

  global asarr_initial_settings
  global buffer_min_fps

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(MinFps)
    } else {
      if { 0 < $prm } {
        if { $buffer_min_fps < 2 } {
          set v [expr $buffer_min_fps + $prm]
        } else {
          set v [expr $buffer_min_fps + $prm + $prm]
        }
      } else {
        if { $buffer_min_fps < 3 } {
          set v [expr $buffer_min_fps + $prm]
        } else {
          set v [expr $buffer_min_fps + $prm + $prm]
        }
      }
    }
    set buffer_min_fps $v
  }

  set v [string trim $buffer_min_fps]
  if { $prm eq "-" || $prm eq "/" } {
    # do nothing
  } else {
    if { $v < 1.0 } {
      set v 1.0
    }
  }
  set buffer_min_fps [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "core.setMinFps($buffer_min_fps)"
  }

  if { $buffer_min_fps < 1.0 } {
    .l_minfps configure -bg "#3000c0"
  } else {
    .l_minfps configure -bg "#000000"
  }

}

#
# 星雲・星団ラベルの量を stellariumサーバへ送信
#
proc send_ssc_of_dsolabels { args } {

  global buffer_dsolabels
  global asarr_initial_settings

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "/" } {
      set v $asarr_initial_settings(NebulaMgr.labelsAmount)
    } elseif { $prm eq "-" } {
      set v 0.0
    } else {
      set v [expr $buffer_dsolabels + $prm]
    }
    set buffer_dsolabels $v
  }

  set v [string trim $buffer_dsolabels]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_dsolabels [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "NebulaMgr.setLabelsAmount($buffer_dsolabels)"
  }

}

#
# 星雲・星団ヒントの量を stellariumサーバへ送信
#
proc send_ssc_of_dsohints { args } {

  global buffer_dsohints
  global asarr_initial_settings

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "/" } {
      set v $asarr_initial_settings(NebulaMgr.hintsAmount)
    } elseif { $prm eq "-" } {
      set v 0.0
    } else {
      set v [expr $buffer_dsohints + $prm]
    }
    set buffer_dsohints $v
  }

  set v [string trim $buffer_dsohints]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_dsohints [format "%.1f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "NebulaMgr.setHintsAmount($buffer_dsohints)"
  }

}

#
# 1つの天体の選択を stellariumサーバへ送信し、その天体についての
# 情報(プレインテキスト)を Object Info フレームにセット
#
proc select_a_dso { args } {

  global buffer_selected_dsoname

  foreach name $args {
    set buffer_selected_dsoname [regsub -all {~} $name { }]
  }

  set buffer_selected_dsoname [string trim $buffer_selected_dsoname]
  send_ssc_lines core.selectObjectByName('$buffer_selected_dsoname',true)

  after 500 get_object_data_from_stellarium

}

#
# 視野円サイズを stellariumサーバへ送信
#
proc send_ssc_of_dso_fov_circular_marker { args } {

  global buffer_dso_fov_circular_size

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v 1.0
    } else {
      set v $prm
    }
    set buffer_dso_fov_circular_size $v
  }

  set v [string trim $buffer_dso_fov_circular_size]
  if { $v < 0.0 } {
    set v 0.0001
  }
  set buffer_dso_fov_circular_size [format "%.4f" $v]

  if { $prm ne "/" } {
    send_ssc_lines SpecialMarkersMgr.setFOVCircularMarkerSize($buffer_dso_fov_circular_size) SpecialMarkersMgr.setFlagFOVCircularMarker(true)
  }

}

#
# stellariumサーバ上で選択されている天体について、
# 情報(プレインテキスト)を Object Info フレームにセット
#
proc get_object_data_from_stellarium {} {

  global stellarium_status_url
  global buffer_codefield
  set flag_using_http_geturl 0

  # nkf の出力側のオプション(未使用)
  set v [info tclversion]
  if { 9.0 <= $v } {
    # output UTF-8
    set nkf_o "-w"
  } else {
    # output Shift-JIS
    set nkf_o "-s"
  }

  if { $flag_using_http_geturl != 0 } {
#    # httpパッケージを使用
#    set r [::http::geturl $stellarium_status_url -binary 1]
#    # httpステータス ("200"なら正常)
#    set ret_code [::http::ncode $r]
#    # 内容
#    set ret_string [::http::data $r]
#    ::http::cleanup $r
  } else {
    # ISCのprocを使用
    do_http_get $stellarium_status_url ret_string buffer_codefield
    if { $ret_string eq "" } {
      set ret_code ""
    } else {
      set ret_code "200"
    }
  }

  if { $ret_code ne "200" } {

    set buffer_codefield "ERROR in get_object_data_from_stellarium: Cannot connect to Stellarium."

  } else {

    # 現在のTcl文字コードへ変換
    set line [encoding convertfrom utf-8 $ret_string]

    # Replace: \" -> \a
    set line [regsub -all {[\\]"} $line "\a"]
    set status [regexp -- {.*["]selectioninfo["]:["]([^"]+)["]} $line match info_str]
    if { $status == 1 } {
      # Restore: \a -> \"
      set info_str [string map {"\a" "\""} $info_str]
      # 横線は GUI の幅マイナス1 に設定 (text .t_objinfo のオプションを参照)
      set hr_str "-------------------------------------------------------------------"
      set info_str [regsub -all {<br[ ]*[/]*>} $info_str "\n"]
      set info_str [regsub -all {<h[0-9]>} $info_str "$hr_str\n"]
      set info_str [regsub -all {</h[0-9]>} $info_str "\n$hr_str\n"]
      set info_str [regsub -all {</sup>|</sub>|<[/]*b>|<[/]*strong>} $info_str ""]
      set info_str [string map {"<sup>" "^" "<sub>" "_" "&alpha;" "α" "&delta;" "δ" "&phi;" "φ"} $info_str]
      # delete all in text area
      .t_objinfo configure -state normal
      .t_objinfo delete 1.0 end
      .t_objinfo insert end "$info_str"
      .t_objinfo configure -state disabled
    }

  }

}

#
# 天体情報(プレインテキスト)をコピーバッファにセット
#
proc set_object_text_to_clipboard {} {

  set info_str [.t_objinfo get 1.0 end]

  clipboard clear
  clipboard append $info_str

}


#
# 日付時刻 GUI を起動
#
proc exec_date_gui {} {

  global flag_host_gui
  global flag_dome_mode
  global flag_large_gui
  global stellarium_status_url
  global buffer_codefield
  global stellarium_url
  global iscserv_url

  # check status of stellarium
  do_http_get $stellarium_status_url status buffer_codefield

  if { $status eq "" } {
    set buffer_codefield "ERROR in exec_date_gui: Cannot connect to Stellarium."
  } else {
    set arg_host ""
    if { $flag_host_gui != 0 } {
      set arg_host "-host"
    } else {
      set arg_host "$stellarium_url,$iscserv_url"
    }
    if { $flag_large_gui == 0 } {
      set ret [exec -ignorestderr wish date-gui.tcl $arg_host &]
    } else {
      set ret [exec -ignorestderr wish date-gui.tcl $arg_host -large-gui &]
    }
  }

}

#
# Media GUI を起動
#
proc exec_media_gui { args } {

  global flag_host_gui
  global flag_dome_mode
  global flag_large_gui
  global stellarium_status_url
  global buffer_codefield
  global stellarium_url
  global iscserv_url

  set exec_arg ""
  foreach v $args {
    # use 1st arg only
    set exec_arg $v
    break
  }

  # check status of stellarium
  do_http_get $stellarium_status_url status buffer_codefield

  if { $status eq "" } {
    set buffer_codefield "ERROR in exec_date_gui: Cannot connect to Stellarium."
  } else {
    set arg_host ""
    if { $flag_host_gui != 0 } {
      set arg_host "-host"
    } else {
      set arg_host "$stellarium_url,$iscserv_url"
    }
    if { $flag_large_gui == 0 } {
      set ret [exec -ignorestderr wish media-gui.tcl $arg_host $exec_arg &]
    } else {
      set ret [exec -ignorestderr wish media-gui.tcl $arg_host -large-gui $exec_arg &]
    }
  }

}

#
# オンスクリーン時計を起動
#
proc exec_date_monitor {} {

  global stellarium_status_url
  global buffer_codefield

  # check status of stellarium
  do_http_get $stellarium_status_url status buffer_codefield

  if { $status eq "" } {
    set buffer_codefield "ERROR in exec_date_monitor: Cannot connect to Stellarium."
  } else {
    set ret [exec -ignorestderr exec_date-monitor.bat screen &]
  }

}

#
# オンスクリーン時計を閉じる
#
proc kill_date_monitor {} {

  set ret [exec -ignorestderr exec_date-monitor.bat kill &]

}

#
# オンスクリーン時計の調整用表示ON/Off
#
proc change_date_monitor {} {

  set ret [exec -ignorestderr exec_date-monitor.bat change &]

}

#
# スタンバイモードへ移行し、ダイアログで 終了 / モード切替 / GUI再起動 / 復帰
# を選択させる
# リモートマシンで MinFps 値が変更された場合、ダイアログを閉じる
#

# private proc (コールバックproc から呼ばれる)
proc proc_for_callback_exit_or_reboot { top } {

  global isc_dialog_destroy_flag

  array set asarr_current_min_fps {}

  get_current_min_fps_to asarr_current_min_fps

  set v $asarr_current_min_fps(MinFps)

  if { 1.0 <= $v } {
    set junk [catch {destroy $top} wid]
    # 大域変数で呼び出し元に知らせる
    set isc_dialog_destroy_flag 1
  } else {
    after 1000 callback_exit_or_reboot $top
  }

}

# private proc (コールバック):
#  ダイアログが閉じられたか、MinFps値が1.0以上になったら知らせる
proc callback_exit_or_reboot { top } {

  global isc_dialog_destroy_flag

  #array set asarr_current_settings {}

  # ダイアログが存在するかを判定
  if { [catch {toplevel $top} wid] } {

    exec_ssc_on_server getCurrentMinFps.ssc
    after 500 proc_for_callback_exit_or_reboot $top

    # 以下のコードでも動くが「after 500」が GUI の邪魔をするので↑のコードにした

    #exec_ssc_on_server $ssc_get_current_settings
    #after 500
    #get_current_settings_to asarr_current_settings

    #set v $asarr_current_settings(MinFps)

    #if { 1.0 <= $v } {
    #  set junk [catch {destroy $top} wid]
    #  # 大域変数で呼び出し元に知らせる
    #  set isc_dialog_destroy_flag 1
    #} else {
    #  after 1000 callback_exit_or_reboot $top
    #}

  } else {

    set junk [catch {destroy $top} wid]
    # 大域変数で呼び出し元に知らせる
    set isc_dialog_destroy_flag 1

  }

}

# 本体 1
proc exit_or_reboot {} {

  global flag_dome_mode
  global buffer_timerate
  global buffer_min_fps
  global asarr_initial_settings

  array set asarr_current_min_fps {}

  set min_fps_standby "0.2"

  send_ssc_lines "core.setTimeRate(0.0)"

  # Stellariumのバグ対策:
  #  もし値が壊れているとここでダイアログが出てしまうのでコメントアウト
  #reload_all_gui_params

  set bak_min_fps $buffer_min_fps

  send_ssc_lines "core.setMinFps($min_fps_standby)"

  set bak_init_min_fps $asarr_initial_settings(MinFps)
  set asarr_initial_settings(MinFps) "$min_fps_standby"
  send_ssc_of_min_fps /
  set asarr_initial_settings(MinFps) $bak_init_min_fps

  set ans [isc_dialog .dlg_stdby_mode "Standby mode: Exit?" "現在、省電力待機中です\n\n以下より選択してください: " 0 "callback_exit_or_reboot"  3 "全終了" "Navi/Dome切替" "GUI再起動" "操作へ戻る"]

  if { $ans == 0 } {
    send_ssc_lines "core.setMinFps($bak_min_fps)"
    exec isc_startup.bat -kill &
  } elseif { $ans == 1 } {
    send_ssc_lines "core.setMinFps($bak_min_fps)"
    exec isc_startup.bat -gui-only &
  } elseif { $ans == 2 } {
    send_ssc_lines "core.setMinFps($bak_min_fps)"
    if { $flag_dome_mode == 0 } {
      exec isc_startup.bat -gui-only -navi &
    } else {
      exec isc_startup.bat -gui-only -dome &
    }
  } else {
    exec_ssc_on_server getCurrentMinFps.ssc
    after 500
    get_current_min_fps_to asarr_current_min_fps
    if { $asarr_current_min_fps(MinFps) < 1.0 } {
      set buffer_min_fps $bak_min_fps
    } else {
      set buffer_min_fps $asarr_current_min_fps(MinFps)
    }
    send_ssc_of_min_fps
    #
    #reload_all_gui_params
    #if { $buffer_min_fps < 1.0 } {
    #  set buffer_min_fps $bak_min_fps
    #  send_ssc_of_min_fps
    #}
  }

}

# 本体 2
proc urgent_exit_or_reboot {} {

  set ans [isc_dialog .dlg_urgent_exit "Urgent Exit" "非常用 終了メニュー\n\n以下より選択してください: " 0 ""  3 "全終了" "Navi/Dome切替" "GUI再起動" "操作へ戻る"]

  if { $ans == 0 } {
    exec isc_startup.bat -kill &
  } elseif { $ans == 1 } {
    exec isc_startup.bat -gui-only &
  } elseif { $ans == 2 } {
    if { $flag_dome_mode == 0 } {
      exec isc_startup.bat -gui-only -navi &
    } else {
      exec isc_startup.bat -gui-only -dome &
    }
  }

}


#                               #
# Configurationフレーム用のproc #
#                               #

proc send_ssc_of_projection_mode { args } {

  global asarr_initial_settings
  global buffer_projection_mode

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ProjectionMode)
    } else {
      set v $prm
    }
    set buffer_projection_mode $v
  }

  set v [string trim $buffer_projection_mode]
  set buffer_projection_mode $v

  if { $prm ne "/" } {
    send_ssc_lines "core.setProjectionMode('$buffer_projection_mode')"
  }

}

proc send_ssc_of_display_gamma { args } {

  global asarr_initial_settings
  global buffer_display_gamma

  set prm ""
  foreach p $args {
    set prm $p
    set prm_idx0 [string index $prm 0]
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(DisplayGamma)
    } elseif { $prm_idx0 eq "-" || $prm_idx0 eq "+" } {
      set v [expr $buffer_display_gamma + $prm]
    } else {
      set v $prm
    }
    set buffer_display_gamma $v
  }

  set v [string trim $buffer_display_gamma]
  if { $v < 0.0 } {
    set v 0.0
  }
  set buffer_display_gamma [format "%.2f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "core.setDisplayGamma($buffer_display_gamma)"
  }

}

proc send_ssc_of_view_fov { args } {

  global asarr_initial_settings
  global buffer_view_fov

  set prm ""
  set duration_tm 0.0
  foreach p $args {
    set prm $p
    set prm_idx0 [string index $prm 0]
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewFov)
    } elseif { $prm_idx0 eq "-" || $prm_idx0 eq "+" } {
      set v [expr $buffer_view_fov + $prm]
      set duration_tm 0.0
    } else {
      # not used ...
      set v $prm
      set duration_tm 0.0
    }
    set buffer_view_fov $v
  }

  set v [string trim $buffer_view_fov]
  if { $v < 5.0 } {
    set v 5.0
  }
  set buffer_view_fov [format "%.2f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelMovementMgr.zoomTo($buffer_view_fov,$duration_tm)"
  }

}

proc send_ssc_of_view_altitude { args } {

  global asarr_initial_settings
  global buffer_view_altitude
  global buffer_view_azimuth

  set prm ""
  set duration_tm 0.0
  foreach p $args {
    set prm $p
    set prm_idx0 [string index $prm 0]
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewAltitude)
      # Stellarium のバグ対策:
      #  理由はわからないが、Dome モードで NaN になる事があるので、その対策
      if { [string tolower $v] eq "nan" } {
        set ans [isc_dialog .dlg_err_alti "Info" "Altitudeの値がNaNですので再設定を試行します。全初期化をお勧めします。" 0 ""  0 "OK" "" "" ""]
        # ISC Serverに保存された初期値を取得
        array set asarr_settings {}
        get_initial_settings_to asarr_settings
        set v $asarr_settings(ViewAltitude)
        if { [string tolower $v] eq "nan" } {
          set v 90.0
        }
        set prm "-"
      }
    } elseif { $prm_idx0 eq "-" || $prm_idx0 eq "+" } {
      set v [expr $buffer_view_altitude + $prm]
      set duration_tm 0.0
    } else {
      # not used ...
      set v $prm
      set duration_tm 0.0
    }
    set buffer_view_altitude $v
  }

  set v [string trim $buffer_view_altitude]
  if { $v < 0.0 } {
    set v 0.0
  }
  while { 90.0 < $v } {
    set v 90.0
  }
  set buffer_view_altitude [format "%.3f" $v]

  if { $prm ne "/" } {

    set count_retry 0
    while { 1 } {

      if { 89.99999 < $buffer_view_altitude } {
        send_ssc_lines "core.moveToAltAzi('89.99999','$buffer_view_azimuth',$duration_tm)"
      } else {
        send_ssc_lines "core.moveToAltAzi('$buffer_view_altitude','$buffer_view_azimuth',$duration_tm)"
      }

      # 引数が "-" の場合のみ NaN かどうかチェックする
      if { $prm ne "-" } {
        break
      }

      # もし値に NaN が見つかった場合、2秒おきに 5回初期化をリトライする
      if { [test_view_prms] == 0 } {
        break
      } elseif { $count_retry < 5 } {
        after 2000
        incr count_retry
      } else {
        # このダイアログが出ることは、まず無いであろう
        set ans_retly [isc_dialog .dlg_retly_alti "Question" "Altitudeの設定に失敗したようです。再試行しますか？" 0 ""  0 "Yes" "No" "" ""]
        if { $ans_retly != 0 } {
          break
        }
        set count_retry 0
      }

    }

  }

}

proc send_ssc_of_view_azimuth { args } {

  global asarr_initial_settings
  global buffer_view_altitude
  global buffer_view_azimuth

  global buffer_azchange_duration

  set prm ""
  set duration_tm 0.0
  foreach p $args {
    set prm $p
    set prm_idx0 [string index $prm 0]
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewAzimuth)
      # Stellarium のバグ対策:
      #  理由はわからないが、Dome モードで NaN になる事があるので、その対策
      if { [string tolower $v] eq "nan" } {
        set ans [isc_dialog .dlg_err_azim "Info" "Azimuthの値がNaNですので再設定を試行します。全初期化をお勧めします" 0 ""  0 "OK" "" "" ""]
        # ISC Serverに保存された初期値を取得
        array set asarr_settings {}
        get_initial_settings_to asarr_settings
        set v $asarr_settings(ViewAzimuth)
        if { [string tolower $v] eq "nan" } {
          set v 180.0
        }
        set prm "-"
      }
    } elseif { [string range $prm 0 1] eq "++" } {
      # Domeモードでゆっくり方角を変更する場合のみ使用
      set v [expr $asarr_initial_settings(ViewAzimuth) + [string range $prm 2 end]]
      set duration_tm $buffer_azchange_duration
    } elseif { $prm_idx0 eq "-" || $prm_idx0 eq "+" } {
      set v [expr $buffer_view_azimuth + $prm]
      set duration_tm 0.0
    } else {
      set v $prm
      set duration_tm 0.0
    }
    set buffer_view_azimuth $v
  }

  set v [format "%.2f" [string trim $buffer_view_azimuth]]
  while { $v < 0.0 } {
    set v [format "%.2f" [expr $v + 360.0]]
  }
  while { 360.0 <= $v } {
    set v [format "%.2f" [expr $v - 360.0]]
  }
  set buffer_view_azimuth $v

  if { $prm ne "/" } {

    set count_retry 0
    while { 1 } {

      if { 89.99999 < $buffer_view_altitude } {
        send_ssc_lines "core.moveToAltAzi('89.99999','$buffer_view_azimuth',$duration_tm)"
      } else {
        send_ssc_lines "core.moveToAltAzi('$buffer_view_altitude','$buffer_view_azimuth',$duration_tm)"
      }

      # 引数が "-" の場合のみ NaN かどうかチェックする
      if { $prm ne "-" } {
        break
      }

      # もし値に NaN が見つかった場合、2秒おきに 5回初期化をリトライする
      if { [test_view_prms] == 0 } {
        break
      } elseif { $count_retry < 5 } {
        after 2000
        incr count_retry
      } else {
        # このダイアログが出ることは、まず無いであろう
        set ans_retly [isc_dialog .dlg_retly_azim "Question" "Azimuthの設定に失敗したようです。再試行しますか？" 0 ""  0 "Yes" "No" "" ""]
        if { $ans_retly != 0 } {
          break
        }
        set count_retry 0
      }

    }

  }

}

proc send_ssc_of_viewport_x { args } {

  global asarr_initial_settings
  global buffer_viewport_x
  global buffer_viewport_y

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewportX)
    } else {
      set v [expr $buffer_viewport_x + $prm]
    }
    set buffer_viewport_x $v
  }

  set v [string trim $buffer_viewport_x]
  set buffer_viewport_x [format "%.3f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelMovementMgr.moveViewport($buffer_viewport_x,$buffer_viewport_y,0.0)"
  }

}

proc send_ssc_of_viewport_y { args } {

  global asarr_initial_settings
  global buffer_viewport_x
  global buffer_viewport_y

  set prm ""
  foreach p $args {
    set prm $p
    if { $prm eq "-" || $prm eq "/" } {
      set v $asarr_initial_settings(ViewportY)
    } else {
      set v [expr $buffer_viewport_y + $prm]
    }
    set buffer_viewport_y $v
  }

  set v [string trim $buffer_viewport_y]
  set buffer_viewport_y [format "%.3f" $v]

  if { $prm ne "/" } {
    send_ssc_lines "StelMovementMgr.moveViewport($buffer_viewport_x,$buffer_viewport_y,0.0)"
  }

}

proc send_ssc_of_disk_viewport { prm } {

  global asarr_initial_settings

  set v "false"
  if { $prm eq "-" || $prm eq "/" } {
    set v $asarr_initial_settings(DiskViewport)
  } else {
    set v $prm
  }

  if { $prm ne "/" } {
    send_ssc_lines "core.setDiskViewport($v)"
  }

}

proc send_ssc_of_init_view_xyfov {} {
  send_ssc_of_view_fov -
  send_ssc_of_view_azimuth -
  send_ssc_of_view_altitude -
}


#                    #
# タブの切替用のproc #
#                    #

proc deselect_all {} {
  global tab_background
  global tab_foreground
  .pw forget .lf_general
  .pw forget .lf_constel
  .pw forget .lf_spring
  .pw forget .lf_summer
  .pw forget .lf_autumn
  .pw forget .lf_winter
  .pw forget .lf_spring_asterism_star
  .pw forget .lf_summer_asterism_star
  .pw forget .lf_autumn_asterism_star
  .pw forget .lf_winter_asterism_star
  .pw forget .lf_summer_star
  .pw forget .lf_autumn_star
  .pw forget .lf_winter_star
  .pw forget .lf_solarsystem
  .pw forget .lf_dso_p0
  .pw forget .lf_dso_p1
  .pw forget .lf_dso_p2
  .pw forget .lf_gridlines
  .pw forget .lf_objinfo
  .pw forget .lf_config
  for {set i 0} {$i < 8} {incr i} {
    #.b_tab$i configure -bg $tab_background -fg $tab_foreground -relief raised
    configure_tab_button .b_tab$i 0
  }
}

proc select_general {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_solarsystem
  .pw add    .lf_gridlines
  #.b_tab0 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
  configure_tab_button .b_tab0 1
}

proc select_spring {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_spring
  .pw add    .lf_spring_asterism_star
  #.b_tab1 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
  configure_tab_button .b_tab1 1
}

proc select_summer {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_summer
  .pw add    .lf_summer_asterism_star
  #.b_tab2 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
  configure_tab_button .b_tab2 1
}

proc select_autumn {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_autumn
  .pw add    .lf_autumn_asterism_star
  #.b_tab3 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
  configure_tab_button .b_tab3 1
}

proc select_winter {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_constel
  .pw add    .lf_winter
  .pw add    .lf_winter_asterism_star
  #.b_tab4 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
  configure_tab_button .b_tab4 1
}

proc select_dso {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  #.pw add    .lf_solarsystem
  .pw add    .lf_dso_p0
  #.pw add    .lf_gridlines
  .pw add    .lf_dso_p1
  .pw add    .lf_dso_p2
  #.b_tab5 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
  configure_tab_button .b_tab5 1
}

proc select_objinfo {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_dso_p0
  .pw add    .lf_objinfo
  #.b_tab6 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
  configure_tab_button .b_tab6 1
}

proc select_config {} {
  global tab_selected_background
  global tab_selected_foreground
  deselect_all
  .pw add    .lf_general
  .pw add    .lf_solarsystem
  .pw add    .lf_gridlines
  .pw add    .lf_config
  #.b_tab7 configure -bg $tab_selected_background -fg $tab_selected_foreground -relief sunken
  configure_tab_button .b_tab7 1
}


####################################################################
##                                                                ##
##                          M A I N  (1)                          ##
##                                                                ##
####################################################################

# hide default wish window
wm state . withdrawn

#
# Host用 GUI かどうかのフラグ (リモートマシンの場合は 0)
#
set flag_host_gui 0

#
# 初期化, 動作モード などのフラグ
#
set flag_initialize 0
set flag_dome_mode "-"
set flag_large_gui 0
set flag_rescue 0

#
# エディタのフラグ
#
set flag_external_editor 0

#
# 引数から動作モードなどを決定
#
for {set i 0} {$i < $argc} {incr i} {
  set arg [lindex $argv $i]
  if { $arg eq "-host" } {
    set flag_host_gui 1
    set flag_external_editor 1
  } elseif { $arg eq "-dome" } {
    set flag_dome_mode 1
  } elseif { [string range $arg 0 4] eq "-navi" } {
    set flag_dome_mode 0
  } elseif { [string range $arg 0 4] eq "-init" } {
    set flag_initialize 1
  } elseif { $arg eq "-rescue" } {
    set flag_rescue 1
  }
}

#
# Do not change: Set stellarium_script_dir from localConfig.ssc
#
if { [catch {open "localConfig.ssc" r} fid] } {
  wm state . normal
  tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot open localConfig.ssc."
  exit
} else {
  set stellarium_script_dir "interstellar-console"
  while { [gets $fid line] >= 0 } {
    set status [regexp -- {^[ ]*var[ ]+([A-Za-z0-9_]+)[ ]*[=][ ]*["]([^"]+)["]} $line match s0 s1]
    if { $status == 1 && $s0 eq "stellarium_script_dir" } {
      set stellarium_script_dir $s1
    }
  }
  close $fid
}

# private proc (ダイアログ内部専用):
#  この下の接続先選択用ダイアログだけで使用
proc set_server_url { id_win url0 url1 } {

  global stellarium_url
  global iscserv_url
  global buffer_remote_hostname

  if { $url0 eq "-" } {
    exit
  } elseif { $url0 eq "/" } {
    set v [string trim $buffer_remote_hostname]
    set stellarium_url "http://$v:8090"
    set iscserv_url    "http://$v:8091"
  } else {
    set stellarium_url "$url0"
    set iscserv_url    "$url1"
  }

  destroy $id_win

}

#
# list_host.txt を読み取り、接続先選択用ダイアログを出す
#
if { $flag_host_gui == 0 } {

  set stellarium_url ""
  set iscserv_url    ""

  # リモート用GUIはデフォルトでは Large GUI とする
  set flag_large_gui 1

  array set lines {}
  #array set host0_url {}
  #array set host1_url {}
  #array set description {}

  get_line_array_from_text_file "list_host.txt" lines

  set len_lines [array size lines]

  # Toplevel Window ID
  set top ".dlg_remote"

  toplevel $top
  wm title $top "Interstellar Console --- remote control"
  wm resizable $top 0 0

  frame $top.fr_select_host_relief -relief raised -bd [scaled_thick_border 2]

  frame $top.fr_select_host_splash

  image create photo splashim -file splash.gif
  label $top.l_splash -image splashim -borderwidth 0
  pack $top.l_splash -in $top.fr_select_host_splash

  frame $top.fr_select_host_body

  frame $top.fr_select_host_0

  label $top.l_select_host -text "接続先を選んでください"
  pack $top.l_select_host -side left -expand 1 -fill x -in $top.fr_select_host_0

  create_button $top.b_select_host_ed "編集" "open_text_editor .edt_host list_host.txt"
  pack $top.b_select_host_ed -expand 0 -fill y -in $top.fr_select_host_0

  pack $top.fr_select_host_0 -expand 1 -fill x -padx [scaled_px 2] -pady [scaled_px 2] -in $top.fr_select_host_body

  set j 0
  for {set i 0} {$i < $len_lines} {incr i} {
    set lst [split $lines($i) ","]
    set len_lst [llength $lst]
    if { 0 < $len_lst } {
      set hostname [string trim [lindex $lst 0]]
      set port0 "8090"
      set port1 "8091"
      set desc ""
      if { 1 < $len_lst } {
        set s [string trim [lindex $lst 1]]
        if { $s ne "" } {
          set port0 $s
        }
      }
      if { 2 < $len_lst } {
        set s [string trim [lindex $lst 2]]
        if { $s ne "" } {
          set port1 $s
        }
      }
      if { 3 < $len_lst } {
        set s [string trim [lindex $lst 3]]
        set desc $s
      }
      set host0_url "http://$hostname:$port0"
      set host1_url "http://$hostname:$port1"
      #
      create_button $top.b_select_host_$j "$hostname --- $desc" "set_server_url $top $host0_url $host1_url"
      pack $top.b_select_host_$j -fill x -padx [scaled_px 2] -pady [scaled_px 2] -in $top.fr_select_host_body
      incr j
    }
  }
  set n_hosts $j

  set buffer_remote_hostname "192.168."

  frame $top.fr_input_host
  label $top.l_input_host_0 -text "http://"
  entry $top.e_input_host -textvariable buffer_remote_hostname -width 15
  bind $top.e_input_host <Return> "set_server_url $top / /"
  label $top.l_input_host_1 -text ":8090,8091 へ "
  create_button $top.l_input_host "接続" "set_server_url $top / /"
  pack $top.l_input_host_0 -side left -in $top.fr_input_host
  pack $top.e_input_host -side left -fill x -expand 1 -in $top.fr_input_host
  pack $top.l_input_host_1 $top.l_input_host -side left -in $top.fr_input_host

  pack $top.fr_input_host -fill x -expand 1 -padx [scaled_px 2] -pady [scaled_px 2] -in $top.fr_select_host_body

  frame $top.fr_cb_large_gui
  ttk::checkbutton $top.cb_large_gui -text "Large GUI を使用" -variable flag_large_gui
  pack $top.cb_large_gui -in $top.fr_cb_large_gui

  pack $top.fr_cb_large_gui -side left -fill x -expand 1 -padx [scaled_px 2] -pady [scaled_px 2] -in $top.fr_select_host_body

  create_button $top.b_select_host_$j "キャンセル" "set_server_url $top - -"
  pack $top.b_select_host_$j -padx [scaled_px 2] -pady [scaled_px 2] -in $top.fr_select_host_body

  if { $n_hosts < 5 } {
    pack $top.fr_select_host_splash -padx [scaled_px 5] -pady [scaled_px 5] -in $top.fr_select_host_relief
  } else {
    pack $top.fr_select_host_splash -side left -padx [scaled_px 5] -pady [scaled_px 5] -in $top.fr_select_host_relief
  }

  pack $top.fr_select_host_body -expand 0 -padx [scaled_px 8] -pady [scaled_px 8] -in $top.fr_select_host_relief

  pack $top.fr_select_host_relief

  wm withdraw $top
  update

  set pos_x [ expr {( [winfo screenwidth .] - [winfo width $top] ) / 2 } ]
  set pos_y [ expr {( [winfo screenheight .] - [winfo height $top] ) / 2 } ]

  wm geometry $top +$pos_x+$pos_y
  wm transient $top .
  wm deiconify $top
  wm state $top normal

  raise $top
  #wm attributes $top -topmost yes

  $top.e_input_host icursor end
  focus $top.e_input_host

  # 他のウィンドゥの操作を禁止
  #grab set $top
  # $top が destroy されるまで待つ
  tkwait window $top

}

if { $stellarium_url eq "" } {
  exit
}


#
# サーバURL (1)  サーバ側に保存されているスクリプト(staticなコード)
#                を実行する時に利用
# 
set stellarium_run_url "$stellarium_url/api/scripts/run"

#
# サーバURL (2)  クライアントで生成したスクリプト(dynamicなコード)
#                を送信する時に利用
# 
#  【注意】direct を使う場合、実行できるのは１行のみらしい。
#          複数行のコードを送信すると、Stellariumがクラッシュする
set stellarium_direct_url "$stellarium_url/api/scripts/direct"

#
# サーバURL (3)  情報取得用
# 
set stellarium_status_url "$stellarium_url/api/main/status"

#
# Stellarium との通信を確認
#
do_http_get $stellarium_status_url status buffer_codefield
if { $status eq "" } {
  isc_dialog .dlg_no_connection "ERROR" "Stellariumと通信できません" 0 ""  0 " OK " "" "" ""
  exit
}

#
# 動作モードの決定
#

proc set_selection { id_win v0 v1 } {
  global flag_dome_mode
  global flag_large_gui
  if { $v0 ne "-" } {
    set flag_dome_mode $v0
  }
  if { $v1 ne "-" } {
    set flag_large_gui $v1
  }
  destroy $id_win
}

if { $flag_host_gui != 0 } {

  #
  # Host用 GUI (動作モード、GUIサイズを選択)
  #

  #
  # ステータス取得用 sscファイル, txtファイル
  #
  set ssc_get_current_settings "getCurrentSettings_0.ssc"
  set txt_current_settings_isc "_tmp_currentSettings_0.isc.txt"

  #
  # 動作モード/GUIサイズ 選択 ダイアログ
  #

  # Toplevel Window ID
  set top ".dlg_selmode"

  while { 1 } {

    toplevel $top
    wm title $top "Interstellar Console --- Selection"
    wm resizable $top 0 0
    wm attributes $top -topmost yes

    frame $top.fr_select_gui_dlg_relief -relief raised -bd [scaled_thick_border 2]

    frame $top.fr_select_gui_dlg

    if { $flag_dome_mode eq "-" } {

      if { [get_flag_dome_mode_on_server] != 0 } {
        set flag_large_gui 1
      } else {
        set flag_large_gui 0
      }

      label $top.l_select_gui_dlg -text "動作モードを選択してください"

      frame $top.fr_select_gui_dlg_0

      create_button $top.b_select_gui_dlg_0 "Navigation" "set_selection $top 0 -"
      create_button $top.b_select_gui_dlg_1 "   Dome   " "set_selection $top 1 -"
      pack $top.b_select_gui_dlg_0 $top.b_select_gui_dlg_1 -side left -expand 1 -padx [scaled_px 2] -in $top.fr_select_gui_dlg_0
      if { $flag_rescue != 0 } {
        create_button $top.b_select_gui_dlg_2 "全終了" "exec isc_startup.bat -kill &"
        pack $top.b_select_gui_dlg_2 -side left -expand 1 -padx [scaled_px 2] -in $top.fr_select_gui_dlg_0
      }

      frame $top.fr_cb_large_gui
      ttk::checkbutton $top.cb_large_gui -text "Large GUI を使用" -variable flag_large_gui
      pack $top.cb_large_gui -in $top.fr_cb_large_gui

      pack $top.l_select_gui_dlg -fill x -expand 1 -padx [scaled_px 2] -pady [scaled_px 10] -in $top.fr_select_gui_dlg
      pack $top.fr_select_gui_dlg_0 $top.fr_cb_large_gui -fill x -expand 1 -padx [scaled_px 2] -pady [scaled_px 4] -in $top.fr_select_gui_dlg

      bind $top.b_select_gui_dlg_0 <Return> {%W invoke}
      bind $top.b_select_gui_dlg_1 <Return> {%W invoke}

      pack $top.fr_select_gui_dlg -padx [scaled_px 8] -pady [scaled_px 4] -in $top.fr_select_gui_dlg_relief

      pack $top.fr_select_gui_dlg_relief

      wm withdraw $top
      update

      set pos_x [ expr {( [winfo screenwidth .] - [winfo width $top] ) / 2 } ]
      set pos_y [ expr {( [winfo screenheight .] - [winfo height $top] ) / 2 } ]

      wm geometry $top +$pos_x+$pos_y
      wm transient $top .
      wm deiconify $top
      wm state $top normal

      raise $top

      if { $flag_large_gui == 0 } {
        focus $top.b_select_gui_dlg_0
      } else {
        focus $top.b_select_gui_dlg_1
      }

    } else {

      label $top.l_select_gui_dlg -text "GUIサイズを選択してください"
      pack $top.l_select_gui_dlg -expand 0 -padx [scaled_px 2] -pady [scaled_px 8] -in $top.fr_select_gui_dlg

      create_button $top.b_select_gui_dlg_0 "Normal" "set_selection $top - 0"
      create_button $top.b_select_gui_dlg_1 "Large"  "set_selection $top - 1"
      pack $top.b_select_gui_dlg_0 $top.b_select_gui_dlg_1 -side left -expand 1 -padx [scaled_px 2] -pady [scaled_px 8] -in $top.fr_select_gui_dlg

      bind $top.b_select_gui_dlg_0 <Return> {%W invoke}
      bind $top.b_select_gui_dlg_1 <Return> {%W invoke}

      pack $top.fr_select_gui_dlg -padx [scaled_px 8] -pady [scaled_px 4] -in $top.fr_select_gui_dlg_relief

      pack $top.fr_select_gui_dlg_relief

      wm withdraw $top
      update

      set pos_x [ expr {( [winfo screenwidth .] - [winfo width $top] ) / 2 } ]
      set pos_y [ expr {( [winfo screenheight .] - [winfo height $top] ) / 2 } ]

      wm geometry $top +$pos_x+$pos_y
      wm transient $top .
      wm deiconify $top
      wm state $top normal

      raise $top

      if { $flag_dome_mode == 0 } {
        focus $top.b_select_gui_dlg_0
      } else {
        focus $top.b_select_gui_dlg_1
      }

    }

    # 他のウィンドゥの操作を禁止
    grab set $top
    # $top が destroy されるまで待つ
    tkwait window $top

    if { $flag_dome_mode ne "-" } {
      break
    }

  }

} else {

  #
  # Remote用 設定
  #

  # ISC Server から remote ID (1から9) を取得する
  do_http_get $iscserv_url/new_remote_id ret buffer_codefield
  set remote_id [string trim $ret]
  if { $remote_id eq "" } {
    isc_dialog .dlg_exceed_n_remote "ERROR" "リモート接続の台数上限を超えました" 0 ""  0 " OK " "" "" ""
    exit
  }

  #
  # ステータス取得用 sscファイル, txtファイル
  #
  set ssc_get_current_settings "getCurrentSettings_$remote_id.ssc"
  set txt_current_settings_isc "_tmp_currentSettings_$remote_id.isc.txt"

  # ホストの動作モードと同じにする

  if { [get_flag_dome_mode_on_server] != 0 } {
    set flag_dome_mode 1
  } else {
    set flag_dome_mode 0
  }

}


####################################################################
##                                                                ##
##                          S P L A S H                           ##
##                                                                ##
####################################################################

#
# Splash はホストで初期化する時のみ表示。
# (別プロセスでないとうまく動かないので exec wish できない OS では表示しない)
#
if { $flag_available_exec != 0 && $flag_host_gui != 0 && $flag_initialize != 0 } {
  # Stellarium の初期化に最低でも4秒はかかる
  set ret [exec -ignorestderr wish splash.tcl 4000 &]
}


####################################################################
##                                                                ##
##                    G U I   S E T T I N G S                     ##
##                                                                ##
####################################################################

#
# 色・フォントの設定
#

set label_inverted_background "#d05050"
set label_inverted_foreground "#000000"

set tab_background "#383838"
set tab_foreground "#ff6060"
set tab_selected_background "#d08000"
set tab_selected_foreground "#000000"

set reload_button_background "#203060"
set reset_button_foreground "#ff6060"
set reset_button_background "#383838"
set exit_button_foreground "#ff9090"
set exit_button_background "#701010"

if { $flag_large_gui == 0 } {
  # Normal GUI モード
  # デフォルト
  option add *font [list $gui_font_name 12]
  set font_small [list $gui_font_name 10]
  set font_filename [list $gui_font_name 10]
  # ラベルフレーム
  set font_labelframe [list $gui_font_name 10]
  # コードフィールド
  set font_codefield [list $gui_font_name 10]
  # ノーマルなbuttonと全く同じサイズにする
  ttk::style configure TButton -font [list $gui_font_name 12]
  ttk::style configure TButton -padding [scaled_padding 3 3]
  ttk::style configure Exit.TButton -background $exit_button_background -foreground $exit_button_foreground
  ttk::style configure Allreset.TButton -background $reset_button_background -foreground $reset_button_foreground
  ttk::style configure Small.TButton -font [list $gui_font_name 10]
  ttk::style configure Small.TButton -padding [scaled_padding 2 1]
  ttk::style configure Reload.TButton -font [list $gui_font_name 10]
  ttk::style configure Reload.TButton -padding [scaled_padding 2 1] -background $reload_button_background
  ttk::style configure Code.TButton -font [list $gui_font_name 10]
  ttk::style configure Code.TButton -padding [scaled_padding 2 1]
  #
  ttk::style configure TCheckbutton -font [list $gui_font_name 12] -padding [scaled_padding 8 8]
  ttk::style configure TCheckbutton -indicatordiameter [scaled_px 10]
  ttk::style configure Small.TCheckbutton -font [list $gui_font_name 10]
  ttk::style configure TCombobox -arrowsize [scaled_px 10]
} else {
  # Large GUI モード
  # デフォルト
  option add *font [list $gui_font_name 20]
  set font_small [list $gui_font_name 16]
  set font_filename [list $gui_font_name 14]
  # ラベルフレーム
  set font_labelframe [list $gui_font_name 12]
  # コードフィールド
  set font_codefield [list $gui_font_name 14]
  # ノーマルなbuttonと全く同じサイズにする
  ttk::style configure TButton -font [list $gui_font_name 20]
  ttk::style configure TButton -padding [scaled_padding 12 11]
  ttk::style configure Exit.TButton -background $exit_button_background -foreground $exit_button_foreground
  ttk::style configure Allreset.TButton -background $reset_button_background -foreground $reset_button_foreground
  ttk::style configure Small.TButton -font [list $gui_font_name 16]
  ttk::style configure Small.TButton -padding [scaled_padding 8 7]
  ttk::style configure Reload.TButton -font [list $gui_font_name 16]
  ttk::style configure Reload.TButton -padding [scaled_padding 8 7] -background $reload_button_background
  ttk::style configure Code.TButton -font [list $gui_font_name 14]
  ttk::style configure Code.TButton -padding [scaled_padding 6 5]
  #
  ttk::style configure TCheckbutton -font [list $gui_font_name 20] -padding [scaled_padding 10 10]
  ttk::style configure TCheckbutton -indicatordiameter [scaled_px 16]
  ttk::style configure Small.TCheckbutton -font [list $gui_font_name 16]
  ttk::style configure TCombobox -arrowsize [scaled_px 16]
}

ttk::style configure Selected.TButton -foreground $tab_selected_foreground -background $tab_selected_background -focuscolor "#9a3030" -relief sunken
ttk::style map Selected.TButton -background [list active $tab_selected_background]

option add *TCombobox*Listbox.font $font_small

# マウスオーバー時の背景色
ttk::style map Exit.TButton -background [list active $exit_button_background]
ttk::style map Allreset.TButton -background [list active $reset_button_background]
ttk::style map Reload.TButton -background [list active $reload_button_background]


# ウィンドゥ名称,属性の設定
set title_str "Interstellar Console :: main"
if { $flag_dome_mode == 0 } {
  append title_str " (Navigation Mode, $os_name, $windows_scaling_pct%)"
} else {
  append title_str " (Dome Mode, $os_name, $windows_scaling_pct%)"
}
if { $flag_host_gui != 0 } {
  append title_str " (Host)"
} else {
  append title_str " ($stellarium_url)"
}
wm title . $title_str
wm resizable . 0 0
if { $flag_host_gui != 0 } {
  # ホストの場合 Xボタンは強制終了ボタンとする
  wm protocol . WM_DELETE_WINDOW [list urgent_exit_or_reboot]
}

# 初期設定での光害ON/OFF
set flag_light_pollution_after_initialization 1

# 方角変更にかかる秒数の初期値
set buffer_azchange_duration 5

# DSO名の初期値
set buffer_selected_dsoname "M42"

# 最後にGUIで設定された日付時刻。入力フィールドの[DATE]ボタンで呼び出せる
set buffer_datearg [clock format [clock seconds] -format {%Y-%m-%dT%H:%M:%S}]

# 星ラベル用連想配列
array set asarr_star_label {}

# 星、天の川の明るさ等 初期値保存用 連想配列
array set asarr_initial_settings {
  StelSkyDrawer.relativeStarScale 1.0
  StelSkyDrawer.absoluteStarScale 1.0
  MilkyWay.intensity 1.0
  ZodiacalLight.intensity 1.0
  LandscapeMgr.landscapeTransparency 0.0
  Altitude 0
  Longitude 135.18989563
  Latitude 34.19189835
  ProjectionMode ProjectionStereographic
  DisplayGamma 2.2
  ViewFov 190.0
  ViewAltitude 90.0
  ViewAzimuth 180.0
  ViewportX 0.0
  ViewportY 0.0
  DiskViewport false
  TimeRate 0
  MinFps 18.0
  StarMgr.labelsAmount 0
  SolarSystem.labelsAmount 0
  NebulaMgr.hintsAmount 0
  NebulaMgr.labelsAmount 0
  SporadicMeteorMgr.ZHR 0
}


####################################################################
##                                                                ##
##                G U I   C O N S T R U C T I O N                 ##
##                                                                ##
####################################################################

########################################################################

frame .fr_rootwin

########################################################################

frame .fr_guitab

labelframe .lf_gui -text "GUI" -font $font_labelframe

# 終了ボタンは Host用 GUI のみ
if { $flag_host_gui != 0 } {
  #button .b_exitall -text "終" -background $exit_button_background -command "exit_or_reboot"
  #create_button .b_exitall "終" "exit_or_reboot"
  ttk::button .b_exitall -style Exit.TButton -text "終" -command "exit_or_reboot"
  pack .b_exitall -side left -expand 0 -fill y -in .lf_gui
}

if { $flag_host_gui != 0 } {
  #button .b_dategui -text "日時" -font $font_small -command "exec_date_gui"
  create_button_small .b_dategui "日時" "exec_date_gui"
} else {
  #button .b_dategui -text "日時" -command "exec_date_gui"
  create_button .b_dategui "日時" "exec_date_gui"
}

#button .b_mediagui -text "Media" -font $font_small -command "exec_media_gui"
#button .b_reload -text "Reload" -background $reload_button_background -font $font_small -command "reload_all_gui_params"
create_button_small .b_mediagui "Media"  "exec_media_gui"
#create_button_small .b_reload   "Reload" "reload_all_gui_params"
ttk::button .b_reload -style Reload.TButton -text "Reload" -command "reload_all_gui_params"
pack .b_dategui .b_mediagui .b_reload -side left -expand 0 -fill y -in .lf_gui

labelframe .lf_selection -text "Selection" -font $font_labelframe

#button .b_tab0 -text "一般" -command "select_general"
#button .b_tab1 -text "星座 春" -command "select_spring"
#button .b_tab2 -text "星座 夏" -command "select_summer"
#button .b_tab3 -text "星座 秋" -command "select_autumn"
#button .b_tab4 -text "星座 冬" -command "select_winter"
#button .b_tab5 -text "天体" -command "select_dso"
#button .b_tab6 -text "情報" -font $font_small -command "select_objinfo"
#button .b_tab7 -text "設定" -font $font_small -command "select_config"
create_button .b_tab0 "一般" "select_general"
create_button .b_tab1 "星座 春" "select_spring"
create_button .b_tab2 "星座 夏" "select_summer"
create_button .b_tab3 "星座 秋" "select_autumn"
create_button .b_tab4 "星座 冬" "select_winter"
create_button .b_tab5 "天体" "select_dso"
create_button .b_tab6 "情報" "select_objinfo"
create_button .b_tab7 "設定" "select_config"

pack .b_tab0 .b_tab1 .b_tab2 .b_tab3 .b_tab4 .b_tab5 -side left -expand 1 -fill x -in .lf_selection
pack .b_tab6 .b_tab7 -side left -expand 0 -fill y -in .lf_selection

pack .lf_gui -side left -expand 0 -padx [scaled_px 3] -pady [scaled_px 2] -in .fr_guitab
pack .lf_selection -side left -expand 1 -fill x  -padx [scaled_px 3] -pady [scaled_px 2] -in .fr_guitab

pack .fr_guitab -expand 0 -fill x -padx [scaled_px 1] -pady [scaled_px 1] -in .fr_rootwin


################################################################

panedwindow .pw
pack .pw -fill x -padx [scaled_px 3] -pady [scaled_px 2] -in .fr_rootwin

################################################################

labelframe .lf_general -text "General" -font $font_labelframe

frame .fr_init
label .l_init -text "初期設定"
create_button .b_init3 "20時" "send_ssc_from_timearg 20:00:00"
create_button .b_init2 "場所" "send_ssc_of_location_latitude -"
create_button .b_init1 "VIEW" "send_ssc_to_initialize_view 1 -"
# Note: Light Pollution settings are disabled by localInitLocation.ssc.
#create_button .b_init0 "全" "send_ssc_to_initialize_all 1"
ttk::button .b_init0 -style Allreset.TButton -text "全" -command "send_ssc_to_initialize_all 1"
pack .l_init -in .fr_init
pack .b_init0 .b_init1 .b_init2 .b_init3 -side left -expand 1 -fill x -in .fr_init

frame .fr_tmrate
frame .fr_tmrate_l
label .l_tmrate -text "時刻レート" -padx 0
label .l_tmrate_b -text "(日周運動)" -font $font_small -padx 0
pack .l_tmrate .l_tmrate_b -side left -padx 0 -in .fr_tmrate_l
create_button .b_tmrate0 "０" "set_timerate 0.0"
create_button .b_tmrate1 "実" "set_timerate 1.0"
entry .e_tmrate -width 5 -textvariable buffer_timerate
set buffer_timerate "0.0"
create_button .b_tmrate2 "＜" "send_ssc_of_timerate -20.0"
create_button .b_tmrate3 "＞" "send_ssc_of_timerate +20.0"
bind .e_tmrate <Return> "send_ssc_of_timerate"
pack .fr_tmrate_l -in .fr_tmrate
pack .b_tmrate0 .b_tmrate1 -side left -fill both -in .fr_tmrate
pack .e_tmrate -expand 1 -side left -fill both -in .fr_tmrate
pack .b_tmrate2 .b_tmrate3 -side left -fill both -in .fr_tmrate

frame .fr_illumi
label .l_illumi -text "太陽系・光害"
create_button .b_illumi2 "ON" "exec_ssc_on_server setLightPollution.ssc"
create_button .b_illumi1 "Off" "exec_ssc_on_server unsetLightPollution.ssc"
create_button .b_illumi0 "太陽系Off" "unset_solarsystem -"
pack .l_illumi -in .fr_illumi
pack .b_illumi0 .b_illumi1 .b_illumi2 -side left -expand 1 -fill x -in .fr_illumi

frame .fr_atmos
label .l_atmos -text "大気" -font $font_small
create_button_small .b_atmos2 "FOG" "send_ssc_lines LandscapeMgr.setFlagFog(true) LandscapeMgr.setFlagAtmosphere(true)"
create_button_small .b_atmos1 "ON" "send_ssc_lines LandscapeMgr.setFlagFog(false) LandscapeMgr.setFlagAtmosphere(true)"
create_button_small .b_atmos0 "Off" "send_ssc_lines LandscapeMgr.setFlagFog(false) LandscapeMgr.setFlagAtmosphere(false)"
pack .l_atmos -in .fr_atmos
pack .b_atmos0 .b_atmos1 .b_atmos2 -side left -expand 1 -fill x -in .fr_atmos

set buffer_milkyway $asarr_initial_settings(MilkyWay.intensity)
frame .fr_milky
label .l_milky -text "天の川 光量" -font $font_small
create_button_small .b_milky0 "初期値" "send_ssc_of_milkyway -"
entry .e_milky -width 5 -textvariable buffer_milkyway -font $font_small
create_button_small .b_milky1 "＜" "send_ssc_of_milkyway -0.1"
create_button_small .b_milky2 "＞" "send_ssc_of_milkyway +0.1"
bind .e_milky <Return> "send_ssc_of_milkyway"
pack .l_milky -in .fr_milky
pack .b_milky0 -side left -fill both -in .fr_milky
pack .e_milky -expand 1 -side left -fill both -in .fr_milky
pack .b_milky1 .b_milky2 -side left -fill both -in .fr_milky

#
# "星表現:絶対" "星表現:相対"
# Domeモードでは横並び、Naviモードでは縦並びにする
#

frame .fr_gen_lbl01

# モードでボタンのラベルを変える
if { $flag_dome_mode != 0 } {
  set txt_init "初"
  set txt_arw_l "<"
  set txt_arw_r ">"
} else {
  set txt_init "初期値"
  set txt_arw_l "＜"
  set txt_arw_r "＞"
}

set buffer_absolute_star_scale $asarr_initial_settings(StelSkyDrawer.absoluteStarScale)
frame .fr_gen_label001
label .l_gen_label001 -text "星表現:絶対" -font $font_small
create_button_small .b_absstar0 $txt_init "send_ssc_of_absolute_star_scale -"
entry .e_absstar -width 4 -font $font_small -textvariable buffer_absolute_star_scale
create_button_small .b_absstar1 $txt_arw_l "send_ssc_of_absolute_star_scale -0.05"
create_button_small .b_absstar2 $txt_arw_r "send_ssc_of_absolute_star_scale +0.05"
bind .e_absstar <Return> "send_ssc_of_absolute_star_scale"
pack .l_gen_label001 -in .fr_gen_label001
pack .b_absstar0 -side left -fill both -in .fr_gen_label001
pack .e_absstar -side left -expand 1 -fill both -in .fr_gen_label001
pack .b_absstar1 .b_absstar2 -side left -fill both -in .fr_gen_label001

set buffer_relative_star_scale $asarr_initial_settings(StelSkyDrawer.relativeStarScale)
frame .fr_gen_label002
label .l_gen_label002 -text "星表現:相対" -font $font_small
create_button_small .b_relstar0 $txt_init "send_ssc_of_relative_star_scale -"
entry .e_relstar -width 4 -font $font_small -textvariable buffer_relative_star_scale
create_button_small .b_relstar1 $txt_arw_l "send_ssc_of_relative_star_scale -0.05"
create_button_small .b_relstar2 $txt_arw_r "send_ssc_of_relative_star_scale +0.05"
bind .e_relstar <Return> "send_ssc_of_relative_star_scale"
pack .l_gen_label002 -in .fr_gen_label002
pack .b_relstar0 -side left -fill both -in .fr_gen_label002
pack .e_relstar -side left -expand 1 -fill both -in .fr_gen_label002
pack .b_relstar1 .b_relstar2 -side left -fill both -in .fr_gen_label002

if { $flag_dome_mode != 0 } {
  pack .fr_gen_label001 .fr_gen_label002 -side left -expand 1 -fill both -in .fr_gen_lbl01
} else {
  pack .fr_gen_label001 .fr_gen_label002 -expand 1 -fill x -in .fr_gen_lbl01
}

#

frame .fr_gen_lbl02

frame .fr_gen_label003
label .l_gen_label003 -text "流星群名" -font $font_small
create_button_small .b_metlbl0 "Off" "send_ssc_lines MeteorShowers.setEnableMarker(false)"
create_button_small .b_metlbl1 "ON" "send_ssc_lines MeteorShowers.setEnableMarker(true)"
pack .l_gen_label003 -in .fr_gen_label003
pack .b_metlbl0 .b_metlbl1 -side left -expand 1 -fill both -in .fr_gen_label003

set buffer_meteor_zhr "0"
frame .fr_gen_label004
label .l_gen_label004 -text "流星ZHR" -font $font_small
create_button_small .b_metzhr0 "０" "send_ssc_of_meteor_zhr -"
entry .e_metzhr -width 5 -font $font_small -textvariable buffer_meteor_zhr
create_button_small .b_metzhr1 "＜" "send_ssc_of_meteor_zhr -1000"
create_button_small .b_metzhr2 "＞" "send_ssc_of_meteor_zhr +1000"
bind .e_metzhr <Return> "send_ssc_of_meteor_zhr"
pack .l_gen_label004 -in .fr_gen_label004
pack .b_metzhr0 -side left -expand 0 -fill both -in .fr_gen_label004
pack .e_metzhr  -side left -expand 1 -fill both -in .fr_gen_label004
pack .b_metzhr1 .b_metzhr2 -side left -expand 0 -fill both -in .fr_gen_label004

pack .fr_gen_label003 .fr_gen_label004 -side left -expand 1 -fill both -in .fr_gen_lbl02

#

frame .fr_gen_lbl03

frame .fr_gen_label005
label .l_gen_label005 -text "方角" -font $font_small
create_button_small .b_gen_label005_1 "ON" "exec_ssc_on_server setCardinal.ssc"
create_button_small .b_gen_label005_0 "Off" "exec_ssc_on_server unsetCardinal.ssc"
pack .l_gen_label005 -expand 1 -fill y -in .fr_gen_label005
pack .b_gen_label005_0 .b_gen_label005_1 -side left -expand 1 -fill x -in .fr_gen_label005

set buffer_landscape_transparency $asarr_initial_settings(LandscapeMgr.landscapeTransparency)
frame .fr_gen_label006
frame .fr_gen_label006u
label .l_gen_label006 -text "風景" -font $font_small
label .l_gen_label006_0 -text "透" -font $font_small
create_button_small .b_gen_label006_1 "初" "send_ssc_of_landscape_transparency -"
entry .e_land_trans -width 3 -font $font_small -textvariable buffer_landscape_transparency
create_button_small .b_gen_label006_3 "ON" "send_ssc_lines LandscapeMgr.setFlagLandscape(true)"
create_button_small .b_gen_label006_2 "Off" "send_ssc_lines LandscapeMgr.setFlagLandscape(false)"
bind .e_land_trans <Return> "send_ssc_of_landscape_transparency"
pack .l_gen_label006 -side left -expand 1 -fill both -in .fr_gen_label006u
pack .l_gen_label006_0 .b_gen_label006_1 .e_land_trans -side left -expand 0 -fill both -in .fr_gen_label006u
pack .fr_gen_label006u -expand 1 -fill x -in .fr_gen_label006
pack .b_gen_label006_2 .b_gen_label006_3 -side left -expand 1 -fill x -in .fr_gen_label006

pack .fr_gen_label005 .fr_gen_label006 -side left -expand 1 -fill both -in .fr_gen_lbl03

#

frame .fr_gen_lbl04

if { $flag_dome_mode != 0 } {
  label .l_gen_label007 -text "方角切替" -font $font_small
  entry .e_azch_dt -width 2 -font $font_small -textvariable buffer_azchange_duration
  label .l_gen_label008 -text "秒" -font $font_small
  create_button_small .b_gen_label0080 "＜" "update_azchange_duration -1"
  create_button_small .b_gen_label0081 "＞" "update_azchange_duration +1"
  bind .e_azch_dt <Return> "update_azchange_duration"
  pack .l_gen_label007 -side left -expand 1 -fill both -in .fr_gen_lbl04
  pack .e_azch_dt .l_gen_label008 .b_gen_label0080 .b_gen_label0081 -side left -expand 0 -fill both -in .fr_gen_lbl04
} else {
  label .l_gen_label007 -text "VIEW" -font $font_small
  create_button_small .b_gen_label0080 "北" "send_ssc_of_view_azimuth 0"
  create_button_small .b_gen_label0081 "東" "send_ssc_of_view_azimuth 90"
  create_button_small .b_gen_label0082 "南" "send_ssc_of_view_azimuth 180"
  create_button_small .b_gen_label0083 "西" "send_ssc_of_view_azimuth 270"
  create_button_small .b_gen_label0084 "↑" "send_ssc_of_view_altitude +5.0"
  pack .l_gen_label007 -side left -expand 1 -fill both -in .fr_gen_lbl04
  pack .b_gen_label0080 .b_gen_label0081 .b_gen_label0082 .b_gen_label0083 .b_gen_label0084 -side left -expand 0 -fill both -in .fr_gen_lbl04
}

frame .fr_gen_lbl05

if { $flag_dome_mode != 0 } {
  create_button_small .b_gen_label0090 "  0" "send_ssc_of_view_azimuth ++0"
  create_button_small .b_gen_label0091 " 90" "send_ssc_of_view_azimuth ++90"
  create_button_small .b_gen_label0092 "180" "send_ssc_of_view_azimuth ++180"
  create_button_small .b_gen_label0093 "270" "send_ssc_of_view_azimuth ++270"
  pack .b_gen_label0090 .b_gen_label0091 .b_gen_label0092 .b_gen_label0093 -side left -expand 1 -fill x -in .fr_gen_lbl05
} else {
  create_button_small .b_gen_label0090 "初期値" "send_ssc_of_init_view_xyfov"
  create_button_small .b_gen_label0091 "−"     "send_ssc_of_view_fov +5.0"
  create_button_small .b_gen_label0092 "＋"     "send_ssc_of_view_fov -5.0"
  create_button_small .b_gen_label0093 "←"     "send_ssc_of_view_azimuth -10.0"
  create_button_small .b_gen_label0094 "→"     "send_ssc_of_view_azimuth +10.0"
  create_button_small .b_gen_label0095 "↓"     "send_ssc_of_view_altitude -5.0"
  pack .b_gen_label0090 -side left -expand 1 -fill x -in .fr_gen_lbl05
  pack .b_gen_label0091 .b_gen_label0092 .b_gen_label0093 .b_gen_label0094 .b_gen_label0095 -side left -expand 0 -fill x -in .fr_gen_lbl05
}

#

#frame .fr_land
#label .l_land -text "風景"
#create_button .b_land1 "ON" "send_ssc_lines LandscapeMgr.setFlagLandscape(true)"
#create_button .b_land0 "Off" "send_ssc_lines LandscapeMgr.setFlagLandscape(false)"
#pack .l_land -in .fr_land
#pack .b_land0 .b_land1 -side left -expand 1 -fill x -in .fr_land

#frame .fr_ordinal
#label .l_ordinal -text "方角"
#create_button .b_ordinal1 "ON" "exec_ssc_on_server setCardinal.ssc"
#create_button .b_ordinal0 "Off" "exec_ssc_on_server unsetCardinal.ssc"
#pack .l_ordinal -in .fr_ordinal
#pack .b_ordinal0 .b_ordinal1 -side left -expand 1 -fill x -in .fr_ordinal

#

set buffer_location_latitude $asarr_initial_settings(Latitude)
frame .fr_loclat
label .l_loclat -text "観測地 緯度" -font $font_small
create_button_small .b_loclat0 " 初 " "send_ssc_of_location_latitude -"
entry .e_loclat -width 10 -textvariable buffer_location_latitude -font $font_small
create_button_small .b_loclat1 "＜" "send_ssc_of_location_latitude -5.0"
create_button_small .b_loclat2 "＞" "send_ssc_of_location_latitude +5.0"
bind .e_loclat <Return> "send_ssc_of_location_latitude"
pack .l_loclat -in .fr_loclat
pack .b_loclat0 -side left -fill both -in .fr_loclat
pack .e_loclat -expand 1 -side left -fill both -in .fr_loclat
pack .b_loclat1 .b_loclat2 -side left -fill both -in .fr_loclat

#

set buffer_min_fps $asarr_initial_settings(MinFps)
frame .fr_minfps
label .l_minfps -text "最小FPS" -font $font_small
create_button_small .b_minfps0 "初期値" "send_ssc_of_min_fps -"
entry .e_minfps -width 5 -textvariable buffer_min_fps -font $font_small
create_button_small .b_minfps1 "＜" "send_ssc_of_min_fps -1.0"
create_button_small .b_minfps2 "＞" "send_ssc_of_min_fps +1.0"
bind .e_minfps <Return> "send_ssc_of_min_fps"
pack .l_minfps -fill both -in .fr_minfps
pack .b_minfps0 -side left -fill both -in .fr_minfps
pack .e_minfps -expand 1 -side left -fill both -in .fr_minfps
pack .b_minfps1 .b_minfps2 -side left -fill both -in .fr_minfps

pack .fr_init .fr_tmrate .fr_illumi .fr_atmos .fr_milky .fr_gen_lbl01 .fr_gen_lbl02 .fr_gen_lbl03 -expand 0 -fill x -in .lf_general
if { 1 } {
  pack .fr_gen_lbl04 .fr_gen_lbl05 -expand 0 -fill x -in .lf_general
}
pack .fr_loclat .fr_minfps -expand 0 -fill x -in .lf_general

####

if { $flag_host_gui != 0 && $flag_dome_mode != 0 } {

  frame .fr_datetm
  label .l_datetm -text "時計 on Screen" -font $font_small
  create_button_small .b_datetm0 "Off" "kill_date_monitor"
  create_button_small .b_datetm1 "ON" "exec_date_monitor"
  create_button_small .b_datetm2 "調整用" "change_date_monitor"
  pack .l_datetm -in .fr_datetm
  pack .b_datetm0 .b_datetm1 .b_datetm2 -side left -expand 1 -fill x -in .fr_datetm

  pack .fr_datetm -expand 0 -fill x -in .lf_general

}

.pw add .lf_general


################################################################

labelframe .lf_constel -text "Constellation" -font $font_labelframe

frame .fr_name
label .l_name -text "星座名称"
create_button .b_name1 "ON" "send_ssc_lines ConstellationMgr.setFlagLabels(true)"
create_button .b_name0 "Off" "send_ssc_lines ConstellationMgr.setFlagLabels(false)"
pack .l_name -in .fr_name
pack .b_name0 .b_name1 -side left -expand 1 -fill x -in .fr_name

frame .fr_line
label .l_line -text "星座線"
create_button .b_line1 "ON" "send_ssc_lines ConstellationMgr.setFlagLines(true)"
create_button .b_line0 "Off" "send_ssc_lines ConstellationMgr.setFlagLines(false)"
pack .l_line -in .fr_line
pack .b_line0 .b_line1 -side left -expand 1 -fill x -in .fr_line

frame .fr_art
label .l_art -text "星座絵"
create_button .b_art0 "Off" "send_ssc_lines ConstellationMgr.setFlagArt(false)"
create_button .b_art1 "1" "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.1)"
create_button .b_art2 "2" "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.2)"
create_button .b_art3 "3" "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.3)"
create_button .b_art4 "4" "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.4)"
create_button .b_art5 "5" "send_ssc_lines ConstellationMgr.setFlagArt(true) ConstellationMgr.setArtIntensity(0.5)"
pack .l_art -in .fr_art
pack .b_art0 .b_art1 .b_art2 .b_art3 .b_art4 .b_art5 -side left -expand 1 -fill x -in .fr_art

frame .fr_boun
label .l_boun -text "星座境界線"
create_button .b_boun1 "ON" "send_ssc_lines ConstellationMgr.setFlagBoundaries(true)"
create_button .b_boun0 "Off" "send_ssc_lines ConstellationMgr.setFlagBoundaries(false)"
pack .l_boun -in .fr_boun
pack .b_boun0 .b_boun1 -side left -expand 1 -fill x -in .fr_boun

frame .fr_astername
label .l_astername -text "アステリズム名称"
create_button .b_astername0 " Off " "send_ssc_lines AsterismMgr.setFlagLabels(false)"
create_button .b_astername1 " ON "  "send_ssc_lines AsterismMgr.setFlagLabels(true)"
pack .l_astername -in .fr_astername
pack .b_astername0 .b_astername1 -side left -expand 1 -fill x -in .fr_astername

frame .fr_aster
label .l_aster -text "アステリズム"
create_button .b_aster0 "Off" "send_ssc_lines AsterismMgr.deselectAsterisms() AsterismMgr.deselectAsterisms()"
pack .l_aster -in .fr_aster
pack .b_aster0 -side left -expand 1 -fill x -in .fr_aster

frame .fr_all
label .l_all -text "星座 選択"
create_button .b_all2 "全88" "exec_ssc_on_server setConstellationAll.ssc"
create_button .b_all1 "黄12" "exec_ssc_on_server setEclipticConstellationAll.ssc"
create_button .b_all0 "Unset" "exec_ssc_on_server unsetConstellationAll.ssc"
pack .l_all -in .fr_all
pack .b_all0 .b_all1 .b_all2 -side left -expand 1 -fill x -in .fr_all

frame .fr_strlabel
label .l_strlabel -text "恒星 名称"
create_button .b_strlabel0 "Off" "send_ssc_of_stellarlabels -"
entry .e_strlabel -width 4 -textvariable buffer_stellarlabels
set buffer_stellarlabels "0.0"
create_button .b_strlabel1 "＜" "send_ssc_of_stellarlabels -0.1"
create_button .b_strlabel2 "＞" "send_ssc_of_stellarlabels +0.1"
bind .e_strlabel <Return> "send_ssc_of_stellarlabels"
pack .l_strlabel -in .fr_strlabel
pack .b_strlabel0 -side left -fill both -in .fr_strlabel
pack .e_strlabel -expand 1 -side left -fill both -in .fr_strlabel
pack .b_strlabel1 .b_strlabel2 -side left -fill both -in .fr_strlabel

pack .fr_name .fr_line .fr_art .fr_boun .fr_astername .fr_aster .fr_all .fr_strlabel -expand 0 -fill x -in .lf_constel

.pw add .lf_constel


################################################################
# 星座 春

labelframe .lf_spring -text "Spring" -font $font_labelframe

frame .fr_Leo
label .l_Leo -text "    しし座    "
create_button .b_Leo1 " SET " "update_constellation Leo true"
create_button .b_Leo0 "Unset" "update_constellation Leo false"
pack .l_Leo -in .fr_Leo
pack .b_Leo0 .b_Leo1 -side left -expand 1 -fill x -in .fr_Leo

frame .fr_Vir
label .l_Vir -text "おとめ座"
create_button .b_Vir1 " SET " "update_constellation Virgo true"
create_button .b_Vir0 "Unset" "update_constellation Virgo false"
pack .l_Vir -in .fr_Vir
pack .b_Vir0 .b_Vir1 -side left -expand 1 -fill x -in .fr_Vir

frame .fr_Boo
label .l_Boo -text "うしかい座"
create_button .b_Boo1 " SET " "update_constellation Bootes true"
create_button .b_Boo0 "Unset" "update_constellation Bootes false"
pack .l_Boo -in .fr_Boo
pack .b_Boo0 .b_Boo1 -side left -expand 1 -fill x -in .fr_Boo

frame .fr_UMa
label .l_UMa -text "おおぐま座"
create_button .b_UMa1 " SET " "update_constellation Ursa~Major true"
create_button .b_UMa0 "Unset" "update_constellation Ursa~Major false"
pack .l_UMa -in .fr_UMa
pack .b_UMa0 .b_UMa1 -side left -expand 1 -fill x -in .fr_UMa

frame .fr_UMi
label .l_UMi -text "こぐま座"
create_button .b_UMi1 " SET " "update_constellation Ursa~Minor true"
create_button .b_UMi0 "Unset" "update_constellation Ursa~Minor false"
pack .l_UMi -in .fr_UMi
pack .b_UMi0 .b_UMi1 -side left -expand 1 -fill x -in .fr_UMi

frame .fr_Crv
label .l_Crv -text "からす座"
create_button .b_Crv1 " SET " "update_constellation Corvus true"
create_button .b_Crv0 "Unset" "update_constellation Corvus false"
pack .l_Crv -in .fr_Crv
pack .b_Crv0 .b_Crv1 -side left -expand 1 -fill x -in .fr_Crv

frame .fr_Cnc
label .l_Cnc -text "かに座"
create_button .b_Cnc1 " SET " "update_constellation Cancer true"
create_button .b_Cnc0 "Unset" "update_constellation Cancer false"
pack .l_Cnc -in .fr_Cnc
pack .b_Cnc0 .b_Cnc1 -side left -expand 1 -fill x -in .fr_Cnc

frame .fr_Com
label .l_Com -text "かみのけ座" -font $font_small
create_button_small .b_Com1 " SET " "update_constellation Coma~Berenices true"
create_button_small .b_Com0 "Unset" "update_constellation Coma~Berenices false"
pack .l_Com -in .fr_Com
pack .b_Com0 .b_Com1 -side left -expand 1 -fill x -in .fr_Com

frame .fr_Hya
label .l_Hya -text "うみへび座" -font $font_small
create_button_small .b_Hya1 " SET " "update_constellation Hydra true"
create_button_small .b_Hya0 "Unset" "update_constellation Hydra false"
pack .l_Hya -in .fr_Hya
pack .b_Hya0 .b_Hya1 -side left -expand 1 -fill x -in .fr_Hya

frame .fr_CVn
label .l_CVn -text "りょうけん座" -font $font_small
create_button_small .b_CVn1 " SET " "update_constellation Canes~Venatici true"
create_button_small .b_CVn0 "Unset" "update_constellation Canes~Venatici false"
pack .l_CVn -in .fr_CVn
pack .b_CVn0 .b_CVn1 -side left -expand 1 -fill x -in .fr_CVn

frame .fr_Cra
label .l_Cra -text "コップ座" -font $font_small
create_button_small .b_Cra1 " SET " "update_constellation Crater true"
create_button_small .b_Cra0 "Unset" "update_constellation Crater false"
pack .l_Cra -in .fr_Cra
pack .b_Cra0 .b_Cra1 -side left -expand 1 -fill x -in .fr_Cra

pack .fr_Leo .fr_Vir .fr_Boo .fr_UMa .fr_UMi .fr_Crv .fr_Cnc .fr_Com .fr_Hya .fr_CVn .fr_Cra -expand 0 -fill x -in .lf_spring

.pw add .lf_spring

################################################################
# 星座 夏

labelframe .lf_summer -text "Summer" -font $font_labelframe

frame .fr_lyr
label .l_lyr -text "こと座"
create_button .b_lyr1 " SET " "update_constellation Lyra true"
create_button .b_lyr0 "Unset" "update_constellation Lyra false"
pack .l_lyr -in .fr_lyr
pack .b_lyr0 .b_lyr1 -side left -expand 1 -fill x -in .fr_lyr

frame .fr_cyg
label .l_cyg -text "はくちょう座"
create_button .b_cyg1 " SET " "update_constellation Cygnus true"
create_button .b_cyg0 "Unset" "update_constellation Cygnus false"
pack .l_cyg -in .fr_cyg
pack .b_cyg0 .b_cyg1 -side left -expand 1 -fill x -in .fr_cyg

frame .fr_aqu
label .l_aqu -text "わし座"
create_button .b_aqu1 " SET " "update_constellation Aquila true"
create_button .b_aqu0 "Unset" "update_constellation Aquila false"
pack .l_aqu -in .fr_aqu
pack .b_aqu0 .b_aqu1 -side left -expand 1 -fill x -in .fr_aqu

frame .fr_Sco
label .l_Sco -text "さそり座"
create_button .b_Sco1 " SET " "update_constellation Scorpius true"
create_button .b_Sco0 "Unset" "update_constellation Scorpius false"
pack .l_Sco -in .fr_Sco
pack .b_Sco0 .b_Sco1 -side left -expand 1 -fill x -in .fr_Sco

frame .fr_Sgr
label .l_Sgr -text "いて座"
create_button .b_Sgr1 " SET " "update_constellation Sagittarius true"
create_button .b_Sgr0 "Unset" "update_constellation Sagittarius false"
pack .l_Sgr -in .fr_Sgr
pack .b_Sgr0 .b_Sgr1 -side left -expand 1 -fill x -in .fr_Sgr

frame .fr_Lib
label .l_Lib -text "てんびん座"
create_button .b_Lib1 " SET " "update_constellation Libra true"
create_button .b_Lib0 "Unset" "update_constellation Libra false"
pack .l_Lib -in .fr_Lib
pack .b_Lib0 .b_Lib1 -side left -expand 1 -fill x -in .fr_Lib

frame .fr_Her
label .l_Her -text "ヘルクレス座"
create_button .b_Her1 " SET " "update_constellation Hercules true"
create_button .b_Her0 "Unset" "update_constellation Hercules false"
pack .l_Her -in .fr_Her
pack .b_Her0 .b_Her1 -side left -expand 1 -fill x -in .fr_Her

frame .fr_CrB
label .l_CrB -text "かんむり座" -font $font_small
create_button_small .b_CrB1 " SET " "update_constellation Corona~Borealis true"
create_button_small .b_CrB0 "Unset" "update_constellation Corona~Borealis false"
pack .l_CrB -in .fr_CrB
pack .b_CrB0 .b_CrB1 -side left -expand 1 -fill x -in .fr_CrB

frame .fr_Oph
label .l_Oph -text "へびつかい座" -font $font_small
create_button_small .b_Oph1 " SET " "update_constellation Ophiuchus true"
create_button_small .b_Oph0 "Unset" "update_constellation Ophiuchus false"
pack .l_Oph -in .fr_Oph
pack .b_Oph0 .b_Oph1 -side left -expand 1 -fill x -in .fr_Oph

frame .fr_Ser
label .l_Ser -text "へび座" -font $font_small
create_button_small .b_Ser1 " SET " "update_constellation Serpens true"
create_button_small .b_Ser0 "Unset" "update_constellation Serpens false"
pack .l_Ser -in .fr_Ser
pack .b_Ser0 .b_Ser1 -side left -expand 1 -fill x -in .fr_Ser

#frame .fr_Del
#label .l_Del -text "いるか座" -font $font_small
#create_button_small .b_Del1 " SET " "update_constellation Delphinus true"
#create_button_small .b_Del0 "Unset" "update_constellation Delphinus false"
#pack .l_Del -in .fr_Del
#pack .b_Del0 .b_Del1 -side left -expand 1 -fill x -in .fr_Del

#frame .fr_Dra
#label .l_Dra -text "りゅう座" -font $font_small
#create_button_small .b_Dra1 " SET " "update_constellation Draco true"
#create_button_small .b_Dra0 "Unset" "update_constellation Draco false"
#pack .l_Dra -in .fr_Dra
#pack .b_Dra0 .b_Dra1 -side left -expand 1 -fill x -in .fr_Dra

#

frame .fr_cnst_lbl11

frame .fr_c_101
label .l_c_101 -text "りゅう座" -font $font_small
create_button_small .b_c_1011 "SET" "update_constellation Draco true"
create_button_small .b_c_1010 "Unset" "update_constellation Draco false"
pack .l_c_101 -in .fr_c_101
pack .b_c_1010 .b_c_1011 -side left -expand 1 -fill x -in .fr_c_101

frame .fr_c_102
label .l_c_102 -text "いるか座" -font $font_small
create_button_small .b_c_1021 "SET" "update_constellation Delphinus true"
create_button_small .b_c_1020 "Unset" "update_constellation Delphinus false"
pack .l_c_102 -in .fr_c_102
pack .b_c_1020 .b_c_1021 -side left -expand 1 -fill x -in .fr_c_102

pack .fr_c_101 .fr_c_102 -side left -expand 1 -fill x -in .fr_cnst_lbl11

#

pack .fr_lyr .fr_cyg .fr_aqu .fr_Sco .fr_Sgr .fr_Lib .fr_Her .fr_CrB .fr_Oph .fr_Ser .fr_cnst_lbl11 -expand 0 -fill x -in .lf_summer

.pw add .lf_summer

################################################################
# 星座 秋

labelframe .lf_autumn -text "Autumn" -font $font_labelframe

frame .fr_Cas
label .l_Cas -text "カシオペヤ座"
create_button .b_Cas1 " SET " "update_constellation Cassiopeia true"
create_button .b_Cas0 "Unset" "update_constellation Cassiopeia false"
pack .l_Cas -in .fr_Cas
pack .b_Cas0 .b_Cas1 -side left -expand 1 -fill x -in .fr_Cas

frame .fr_Per
label .l_Per -text "ペルセウス座"
create_button .b_Per1 " SET " "update_constellation Perseus true"
create_button .b_Per0 "Unset" "update_constellation Perseus false"
pack .l_Per -in .fr_Per
pack .b_Per0 .b_Per1 -side left -expand 1 -fill x -in .fr_Per

frame .fr_And
label .l_And -text "アンドロメダ座"
create_button .b_And1 " SET " "update_constellation Andromeda true"
create_button .b_And0 "Unset" "update_constellation Andromeda false"
pack .l_And -in .fr_And
pack .b_And0 .b_And1 -side left -expand 1 -fill x -in .fr_And

frame .fr_Peg
label .l_Peg -text "ペガスス座"
create_button .b_Peg1 " SET " "update_constellation Pegasus true"
create_button .b_Peg0 "Unset" "update_constellation Pegasus false"
pack .l_Peg -in .fr_Peg
pack .b_Peg0 .b_Peg1 -side left -expand 1 -fill x -in .fr_Peg

frame .fr_Psc
label .l_Psc -text "うお座"
create_button .b_Psc1 " SET " "update_constellation Pisces true"
create_button .b_Psc0 "Unset" "update_constellation Pisces false"
pack .l_Psc -in .fr_Psc
pack .b_Psc0 .b_Psc1 -side left -expand 1 -fill x -in .fr_Psc

frame .fr_Cet
label .l_Cet -text "くじら座"
create_button .b_Cet1 " SET " "update_constellation Cetus true"
create_button .b_Cet0 "Unset" "update_constellation Cetus false"
pack .l_Cet -in .fr_Cet
pack .b_Cet0 .b_Cet1 -side left -expand 1 -fill x -in .fr_Cet

frame .fr_Cep
label .l_Cep -text "ケフェウス座"
create_button .b_Cep1 " SET " "update_constellation Cepheus true"
create_button .b_Cep0 "Unset" "update_constellation Cepheus false"
pack .l_Cep -in .fr_Cep
pack .b_Cep0 .b_Cep1 -side left -expand 1 -fill x -in .fr_Cep

frame .fr_PsA
label .l_PsA -text "みなみのうお座" -font $font_small
create_button_small .b_PsA1 " SET " "update_constellation Piscis~Austrinus true"
create_button_small .b_PsA0 "Unset" "update_constellation Piscis~Austrinus false"
pack .l_PsA -in .fr_PsA
pack .b_PsA0 .b_PsA1 -side left -expand 1 -fill x -in .fr_PsA

frame .fr_Aqr
label .l_Aqr -text "みずがめ座" -font $font_small
create_button_small .b_Aqr1 " SET " "update_constellation Aquarius true"
create_button_small .b_Aqr0 "Unset" "update_constellation Aquarius false"
pack .l_Aqr -in .fr_Aqr
pack .b_Aqr0 .b_Aqr1 -side left -expand 1 -fill x -in .fr_Aqr

#frame .fr_Cap
#label .l_Cap -text "やぎ座" -font $font_small
#create_button_small .b_Cap1 " SET " "update_constellation Capricornus true"
#create_button_small .b_Cap0 "Unset" "update_constellation Capricornus false"
#pack .l_Cap -in .fr_Cap
#pack .b_Cap0 .b_Cap1 -side left -expand 1 -fill x -in .fr_Cap

#frame .fr_Ari
#label .l_Ari -text "おひつじ座" -font $font_small
#create_button_small .b_Ari1 " SET " "update_constellation Aries true"
#create_button_small .b_Ari0 "Unset" "update_constellation Aries false"
#pack .l_Ari -in .fr_Ari
#pack .b_Ari0 .b_Ari1 -side left -expand 1 -fill x -in .fr_Ari

#

frame .fr_cnst_lbl21

frame .fr_c_201
label .l_c_201 -text "おひつじ座" -font $font_small
create_button_small .b_c_2011 "SET" "update_constellation Aries true"
create_button_small .b_c_2010 "Unset" "update_constellation Aries false"
pack .l_c_201 -in .fr_c_201
pack .b_c_2010 .b_c_2011 -side left -expand 1 -fill x -in .fr_c_201

frame .fr_c_202
label .l_c_202 -text "やぎ座" -font $font_small
create_button_small .b_c_2021 "SET" "update_constellation Capricornus true"
create_button_small .b_c_2020 "Unset" "update_constellation Capricornus false"
pack .l_c_202 -in .fr_c_202
pack .b_c_2020 .b_c_2021 -side left -expand 1 -fill x -in .fr_c_202

pack .fr_c_201 .fr_c_202 -side left -expand 1 -fill x -in .fr_cnst_lbl21

#

frame .fr_Scu
label .l_Scu -text "ちょうこくしつ座" -font $font_small
create_button_small .b_Scu1 " SET " "update_constellation Sculptor true"
create_button_small .b_Scu0 "Unset" "update_constellation Sculptor false"
pack .l_Scu -in .fr_Scu
pack .b_Scu0 .b_Scu1 -side left -expand 1 -fill x -in .fr_Scu

pack .fr_Cas .fr_Per .fr_And .fr_Peg .fr_Psc .fr_Cet .fr_Cep .fr_PsA .fr_Aqr .fr_cnst_lbl21 .fr_Scu -expand 0 -fill x -in .lf_autumn

.pw add .lf_autumn

################################################################
# 星座 冬
labelframe .lf_winter -text "Winter" -font $font_labelframe

frame .fr_Ori
label .l_Ori -text "  オリオン座  "
create_button .b_Ori1 " SET " "update_constellation Orion true"
create_button .b_Ori0 "Unset" "update_constellation Orion false"
pack .l_Ori -in .fr_Ori
pack .b_Ori0 .b_Ori1 -side left -expand 1 -fill x -in .fr_Ori

frame .fr_CMa
label .l_CMa -text "おおいぬ座"
create_button .b_CMa1 " SET " "update_constellation Canis~Major true"
create_button .b_CMa0 "Unset" "update_constellation Canis~Major false"
pack .l_CMa -in .fr_CMa
pack .b_CMa0 .b_CMa1 -side left -expand 1 -fill x -in .fr_CMa

frame .fr_CMi
label .l_CMi -text "こいぬ座"
create_button .b_CMi1 " SET " "update_constellation Canis~Minor true"
create_button .b_CMi0 "Unset" "update_constellation Canis~Minor false"
pack .l_CMi -in .fr_CMi
pack .b_CMi0 .b_CMi1 -side left -expand 1 -fill x -in .fr_CMi

frame .fr_Gem
label .l_Gem -text "ふたご座"
create_button .b_Gem1 " SET " "update_constellation Gemini true"
create_button .b_Gem0 "Unset" "update_constellation Gemini false"
pack .l_Gem -in .fr_Gem
pack .b_Gem0 .b_Gem1 -side left -expand 1 -fill x -in .fr_Gem

frame .fr_Tau
label .l_Tau -text "おうし座"
create_button .b_Tau1 " SET " "update_constellation Taurus true"
create_button .b_Tau0 "Unset" "update_constellation Taurus false"
pack .l_Tau -in .fr_Tau
pack .b_Tau0 .b_Tau1 -side left -expand 1 -fill x -in .fr_Tau

frame .fr_Aur
label .l_Aur -text "ぎょしゃ座"
create_button .b_Aur1 " SET " "update_constellation Auriga true"
create_button .b_Aur0 "Unset" "update_constellation Auriga false"
pack .l_Aur -in .fr_Aur
pack .b_Aur0 .b_Aur1 -side left -expand 1 -fill x -in .fr_Aur

#frame .fr_Ari_w
#label .l_Ari_w -text "おひつじ座" -font $font_small
#create_button_small .b_Ari_w1 " SET " "update_constellation Aries true"
#create_button_small .b_Ari_w0 "Unset" "update_constellation Aries false"
#pack .l_Ari_w -in .fr_Ari_w
#pack .b_Ari_w0 .b_Ari_w1 -side left -expand 1 -fill x -in .fr_Ari_w

frame .fr_Lep
label .l_Lep -text "うさぎ座" -font $font_small
create_button_small .b_Lep1 " SET " "update_constellation Lepus true"
create_button_small .b_Lep0 "Unset" "update_constellation Lepus false"
pack .l_Lep -in .fr_Lep
pack .b_Lep0 .b_Lep1 -side left -expand 1 -fill x -in .fr_Lep

frame .fr_Col
label .l_Col -text "はと座" -font $font_small
create_button_small .b_Col1 " SET " "update_constellation Columba true"
create_button_small .b_Col0 "Unset" "update_constellation Columba false"
pack .l_Col -in .fr_Col
pack .b_Col0 .b_Col1 -side left -expand 1 -fill x -in .fr_Col

frame .fr_Mon
label .l_Mon -text "いっかくじゅう座" -font $font_small
create_button_small .b_Mon1 " SET " "update_constellation Monoceros true"
create_button_small .b_Mon0 "Unset" "update_constellation Monoceros false"
pack .l_Mon -in .fr_Mon
pack .b_Mon0 .b_Mon1 -side left -expand 1 -fill x -in .fr_Mon

frame .fr_Lyn
label .l_Lyn -text "やまねこ座" -font $font_small
create_button_small .b_Lyn1 " SET " "update_constellation Lynx true"
create_button_small .b_Lyn0 "Unset" "update_constellation Lynx false"
pack .l_Lyn -in .fr_Lyn
pack .b_Lyn0 .b_Lyn1 -side left -expand 1 -fill x -in .fr_Lyn

frame .fr_Cam
label .l_Cam -text "きりん座" -font $font_small
create_button_small .b_Cam1 " SET " "update_constellation Camelopardalis true"
create_button_small .b_Cam0 "Unset" "update_constellation Camelopardalis false"
pack .l_Cam -in .fr_Cam
pack .b_Cam0 .b_Cam1 -side left -expand 1 -fill x -in .fr_Cam

pack .fr_Ori .fr_CMa .fr_CMi .fr_Gem .fr_Tau .fr_Aur .fr_Lep .fr_Col .fr_Mon .fr_Lyn .fr_Cam -expand 0 -fill x -in .lf_winter
.pw add .lf_winter

################################################################

#
# See https://github.com/Stellarium/stellarium/blob/master/skycultures/modern/asterism_names.eng.fab
#

labelframe .lf_spring_asterism_star -text "Spring Asterism/Stars" -font $font_labelframe

frame .fr_001
label .l_001 -text "北斗七星"
create_button .b_0011 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~(Plough)')"
create_button .b_0010 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~(Plough)')"
pack .l_001 -in .fr_001
pack .b_0010 .b_0011 -side left -expand 1 -fill x -in .fr_001

frame .fr_002
label .l_002 -text "北斗七星5倍線"
create_button .b_0021 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~5times')"
create_button .b_0020 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~5times')"
pack .l_002 -in .fr_002
pack .b_0020 .b_0021 -side left -expand 1 -fill x -in .fr_002

frame .fr_003
label .l_003 -text "春の大三角"
create_button .b_0031 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Spring~Triangle')"
create_button .b_0030 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Spring~Triangle')"
pack .l_003 -in .fr_003
pack .b_0030 .b_0031 -side left -expand 1 -fill x -in .fr_003

frame .fr_004
label .l_004 -text "春のダイヤモンド"
create_button .b_0041 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Great~Diamond')"
create_button .b_0040 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Great~Diamond')"
pack .l_004 -in .fr_004
pack .b_0040 .b_0041 -side left -expand 1 -fill x -in .fr_004

frame .fr_005
label .l_005 -text "春の大曲線"
create_button .b_0051 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Spring~Curve')"
create_button .b_0050 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Spring~Curve')"
pack .l_005 -in .fr_005
pack .b_0050 .b_0051 -side left -expand 1 -fill x -in .fr_005

#frame .fr_006
#label .l_006 -text "北極星"
#create_button .b_0061 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Polaris')"
#create_button .b_0060 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Polaris')"
#pack .l_006 -in .fr_006
#pack .b_0060 .b_0061 -side left -expand 1 -fill x -in .fr_006

frame .fr_007
label .l_007 -text "カシオペヤ座5倍線" -font $font_small
create_button_small .b_0071 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Cassio~5times')"
create_button_small .b_0070 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Cassio~5times')"
pack .l_007 -in .fr_007
pack .b_0070 .b_0071 -side left -expand 1 -fill x -in .fr_007

#

frame .fr_astar_lbl00 -height [scaled_px 12]

#

frame .fr_astar_lbl01

label .l_astar_eachlbl02 -text "個別表示" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
create_button_small .b_astar_off02 "全Off" "unset_star_label Polaris Spica Arcturus Regulus Denebola HIP63125A"
pack .l_astar_eachlbl02 .b_astar_off02 -side left -expand 1 -fill both -in .fr_astar_lbl01

#

frame .fr_astar_lbl02

frame .fr_021
label .l_021 -text "北極星" -font $font_small
create_button_small .b_0211 "ON" "set_star_label Polaris"
create_button_small .b_0210 "Off" "unset_star_label Polaris"
pack .l_021 -in .fr_021
pack .b_0210 .b_0211 -side left -expand 1 -fill x -in .fr_021

pack .fr_021 -side left -expand 1 -fill x -in .fr_astar_lbl02

#

frame .fr_astar_lbl03

frame .fr_022
label .l_022 -text "アークトゥルス" -font $font_small
create_button_small .b_0221 "ON" "set_star_label Arcturus"
create_button_small .b_0220 "Off" "unset_star_label Arcturus"
pack .l_022 -in .fr_022
pack .b_0220 .b_0221 -side left -expand 1 -fill x -in .fr_022

pack .fr_022 -side left -expand 1 -fill x -in .fr_astar_lbl03

#

frame .fr_astar_lbl04

frame .fr_023
label .l_023 -text "スピカ" -font $font_small
create_button_small .b_0231 "ON" "set_star_label Spica"
create_button_small .b_0230 "Off" "unset_star_label Spica"
pack .l_023 -in .fr_023
pack .b_0230 .b_0231 -side left -expand 1 -fill x -in .fr_023

frame .fr_024
label .l_024 -text "レグルス" -font $font_small
create_button_small .b_0241 "ON" "set_star_label Regulus"
create_button_small .b_0240 "Off" "unset_star_label Regulus"
pack .l_024 -in .fr_024
pack .b_0240 .b_0241 -side left -expand 1 -fill x -in .fr_024

pack .fr_023 .fr_024 -side left -expand 1 -fill x -in .fr_astar_lbl04

#

frame .fr_astar_lbl05

frame .fr_025
label .l_025 -text "デネボラ" -font $font_small
create_button_small .b_0251 "ON" "set_star_label Denebola"
create_button_small .b_0250 "Off" "unset_star_label Denebola"
pack .l_025 -in .fr_025
pack .b_0250 .b_0251 -side left -expand 1 -fill x -in .fr_025

frame .fr_026
label .l_026 -text "コル・カロリ" -font $font_small
create_button_small .b_0261 "ON" "set_star_label HIP63125A"
create_button_small .b_0260 "Off" "unset_star_label HIP63125A"
pack .l_026 -in .fr_026
pack .b_0260 .b_0261 -side left -expand 1 -fill x -in .fr_026

pack .fr_025 .fr_026 -side left -expand 1 -fill x -in .fr_astar_lbl05

#

pack .fr_001 .fr_002 .fr_003 .fr_004 .fr_005 .fr_007 .fr_astar_lbl00 .fr_astar_lbl01 .fr_astar_lbl02 .fr_astar_lbl03 .fr_astar_lbl04 .fr_astar_lbl05 -expand 0 -fill x -in .lf_spring_asterism_star

.pw add .lf_spring_asterism_star

################################################################

#
# See https://github.com/Stellarium/stellarium/blob/master/skycultures/modern/asterism_names.eng.fab
#

labelframe .lf_summer_asterism_star -text "Summer Asterism/Stars" -font $font_labelframe


frame .fr_101
label .l_101 -text "北斗七星"
create_button .b_1011 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~(Plough)')"
create_button .b_1010 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~(Plough)')"
pack .l_101 -in .fr_101
pack .b_1010 .b_1011 -side left -expand 1 -fill x -in .fr_101

frame .fr_102
label .l_102 -text "北斗七星5倍線"
create_button .b_1021 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~5times')"
create_button .b_1020 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~5times')"
pack .l_102 -in .fr_102
pack .b_1020 .b_1021 -side left -expand 1 -fill x -in .fr_102

frame .fr_103
label .l_103 -text "夏の大三角"
create_button .b_1031 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Summer~Triangle')"
create_button .b_1030 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Summer~Triangle')"
pack .l_103 -in .fr_103
pack .b_1030 .b_1031 -side left -expand 1 -fill x -in .fr_103

frame .fr_104
label .l_104 -text "ティーポット"
create_button .b_1041 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Teapot')"
create_button .b_1040 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Teapot')"
pack .l_104 -in .fr_104
pack .b_1040 .b_1041 -side left -expand 1 -fill x -in .fr_104

frame .fr_105
label .l_105 -text "南斗六星"
create_button .b_1051 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Milk~Dipper')"
create_button .b_1050 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Milk~Dipper')"
pack .l_105 -in .fr_105
pack .b_1050 .b_1051 -side left -expand 1 -fill x -in .fr_105

#frame .fr_106
#label .l_106 -text "北極星"
#create_button .b_1061 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Polaris')"
#create_button .b_1060 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Polaris')"
#pack .l_106 -in .fr_106
#pack .b_1060 .b_1061 -side left -expand 1 -fill x -in .fr_106

frame .fr_107
label .l_107 -text "カシオペヤ座5倍線"
create_button .b_1071 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Cassio~5times')"
create_button .b_1070 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Cassio~5times')"
pack .l_107 -in .fr_107
pack .b_1070 .b_1071 -side left -expand 1 -fill x -in .fr_107

#

frame .fr_astar_lbl10 -height [scaled_px 12]

#

frame .fr_astar_lbl11

label .l_astar_eachlbl12 -text "個別表示" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
create_button_small .b_astar_off12 "全Off" "unset_star_label Polaris Vega Altair Deneb Albireo Antares"
pack .l_astar_eachlbl12 .b_astar_off12 -side left -expand 1 -fill both -in .fr_astar_lbl11

#

frame .fr_astar_lbl12

frame .fr_121
label .l_121 -text "北極星" -font $font_small
create_button_small .b_1211 "ON" "set_star_label Polaris"
create_button_small .b_1210 "Off" "unset_star_label Polaris"
pack .l_121 -in .fr_121
pack .b_1210 .b_1211 -side left -expand 1 -fill x -in .fr_121

pack .fr_121 -side left -expand 1 -fill x -in .fr_astar_lbl12

#

frame .fr_astar_lbl13

frame .fr_122
label .l_122 -text "ベガ" -font $font_small
create_button_small .b_1221 "ON" "set_star_label Vega"
create_button_small .b_1220 "Off" "unset_star_label Vega"
pack .l_122 -in .fr_122
pack .b_1220 .b_1221 -side left -expand 1 -fill x -in .fr_122

frame .fr_123
label .l_123 -text "アルタイル" -font $font_small
create_button_small .b_1231 "ON" "set_star_label Altair"
create_button_small .b_1230 "Off" "unset_star_label Altair"
pack .l_123 -in .fr_123
pack .b_1230 .b_1231 -side left -expand 1 -fill x -in .fr_123

pack .fr_122 .fr_123 -side left -expand 1 -fill x -in .fr_astar_lbl13

#

frame .fr_astar_lbl14

frame .fr_124
label .l_124 -text "デネブ" -font $font_small
create_button_small .b_1241 "ON" "set_star_label Deneb"
create_button_small .b_1240 "Off" "unset_star_label Deneb"
pack .l_124 -in .fr_124
pack .b_1240 .b_1241 -side left -expand 1 -fill x -in .fr_124

frame .fr_125
label .l_125 -text "アルビレオ" -font $font_small
create_button_small .b_1251 "ON" "set_star_label Albireo"
create_button_small .b_1250 "Off" "unset_star_label Albireo"
pack .l_125 -in .fr_125
pack .b_1250 .b_1251 -side left -expand 1 -fill x -in .fr_125

pack .fr_124 .fr_125 -side left -expand 1 -fill x -in .fr_astar_lbl14

#

frame .fr_astar_lbl15

frame .fr_126
label .l_126 -text "アンタレス" -font $font_small
create_button_small .b_1261 "ON" "set_star_label Antares"
create_button_small .b_1260 "Off" "unset_star_label Antares"
pack .l_126 -in .fr_126
pack .b_1260 .b_1261 -side left -expand 1 -fill x -in .fr_126

pack .fr_126 -side left -expand 1 -fill x -in .fr_astar_lbl15

#

pack .fr_101 .fr_102 .fr_103 .fr_104 .fr_105 .fr_107 .fr_astar_lbl10 .fr_astar_lbl11 .fr_astar_lbl12 .fr_astar_lbl13 .fr_astar_lbl14 .fr_astar_lbl15 -expand 0 -fill x -in .lf_summer_asterism_star

.pw add .lf_summer_asterism_star

################################################################

#
# See https://github.com/Stellarium/stellarium/blob/master/skycultures/modern/asterism_names.eng.fab
#

labelframe .lf_autumn_asterism_star -text "Autumn Asterism/Stars" -font $font_labelframe

frame .fr_201
label .l_201 -text "北斗七星"
create_button .b_2011 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~(Plough)')"
create_button .b_2010 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~(Plough)')"
pack .l_201 -in .fr_201
pack .b_2010 .b_2011 -side left -expand 1 -fill x -in .fr_201

frame .fr_202
label .l_202 -text "北斗七星5倍線"
create_button .b_2021 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~5times')"
create_button .b_2020 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~5times')"
pack .l_202 -in .fr_202
pack .b_2020 .b_2021 -side left -expand 1 -fill x -in .fr_202

frame .fr_203
label .l_203 -text "秋の四辺形"
create_button .b_2031 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Great~Square~of~Pegasus')"
create_button .b_2030 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Great~Square~of~Pegasus')"
pack .l_203 -in .fr_203
pack .b_2030 .b_2031 -side left -expand 1 -fill x -in .fr_203

#frame .fr_204
#label .l_204 -text "北極星"
#create_button .b_2041 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Polaris')"
#create_button .b_2040 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Polaris')"
#pack .l_204 -in .fr_204
#pack .b_2040 .b_2041 -side left -expand 1 -fill x -in .fr_204

frame .fr_205
label .l_205 -text "カシオペヤ座5倍線"
create_button .b_2051 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Cassio~5times')"
create_button .b_2050 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Cassio~5times')"
pack .l_205 -in .fr_205
pack .b_2050 .b_2051 -side left -expand 1 -fill x -in .fr_205

frame .fr_206
label .l_206 -text "四辺形から北極星"
create_button .b_2061 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Square~Lines')"
create_button .b_2060 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Square~Lines')"
pack .l_206 -in .fr_206
pack .b_2060 .b_2061 -side left -expand 1 -fill x -in .fr_206

#

frame .fr_astar_lbl20 -height [scaled_px 12]

#

frame .fr_astar_lbl21

label .l_astar_eachlbl22 -text "個別表示" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
create_button_small .b_astar_off22 "全Off" "unset_star_label Polaris Fomalhaut HIP3419 HIP677 HIP10826"
pack .l_astar_eachlbl22 .b_astar_off22 -side left -expand 1 -fill both -in .fr_astar_lbl21

#

frame .fr_astar_lbl22

frame .fr_221
label .l_221 -text "北極星" -font $font_small
create_button_small .b_2211 "ON" "set_star_label Polaris"
create_button_small .b_2210 "Off" "unset_star_label Polaris"
pack .l_221 -in .fr_221
pack .b_2210 .b_2211 -side left -expand 1 -fill x -in .fr_221

frame .fr_222
label .l_222 -text "ミラ" -font $font_small
create_button_small .b_2221 "ON" "set_star_label HIP10826"
create_button_small .b_2220 "Off" "unset_star_label HIP10826"
pack .l_222 -in .fr_222
pack .b_2220 .b_2221 -side left -expand 1 -fill x -in .fr_222

pack .fr_221 .fr_222 -side left -expand 1 -fill x -in .fr_astar_lbl22

#

frame .fr_astar_lbl23
label .l_223 -text "フォーマルハウト" -font $font_small
create_button_small .b_2231 "ON" "set_star_label Fomalhaut"
create_button_small .b_2230 "Off" "unset_star_label Fomalhaut"
pack .l_223 -in .fr_astar_lbl23
pack .b_2230 .b_2231 -side left -expand 1 -fill x -in .fr_astar_lbl23

frame .fr_astar_lbl24
label .l_224 -text "デネブ・カイトス" -font $font_small
create_button_small .b_2241 "ON" "set_star_label HIP3419"
create_button_small .b_2240 "Off" "unset_star_label HIP3419"
pack .l_224 -in .fr_astar_lbl24
pack .b_2240 .b_2241 -side left -expand 1 -fill x -in .fr_astar_lbl24

frame .fr_astar_lbl25
label .l_225 -text "アルフェラッツ" -font $font_small
create_button_small .b_2251 "ON" "set_star_label HIP677"
create_button_small .b_2250 "Off" "unset_star_label HIP677"
pack .l_225 -in .fr_astar_lbl25
pack .b_2250 .b_2251 -side left -expand 1 -fill x -in .fr_astar_lbl25

pack .fr_201 .fr_202 .fr_203 .fr_205 .fr_206 .fr_astar_lbl20 .fr_astar_lbl21 .fr_astar_lbl22 .fr_astar_lbl23 .fr_astar_lbl24 .fr_astar_lbl25 -expand 0 -fill x -in .lf_autumn_asterism_star

.pw add .lf_autumn_asterism_star

################################################################

#
# See https://github.com/Stellarium/stellarium/blob/master/skycultures/modern/asterism_names.eng.fab
#

labelframe .lf_winter_asterism_star -text "Winter Asterism/Stars" -font $font_labelframe

frame .fr_301
label .l_301 -text "北斗七星"
create_button .b_3011 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~(Plough)')"
create_button .b_3010 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~(Plough)')"
pack .l_301 -in .fr_301
pack .b_3010 .b_3011 -side left -expand 1 -fill x -in .fr_301

frame .fr_302
label .l_302 -text "北斗七星5倍線"
create_button .b_3021 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Big~Dipper~5times')"
create_button .b_3020 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Big~Dipper~5times')"
pack .l_302 -in .fr_302
pack .b_3020 .b_3021 -side left -expand 1 -fill x -in .fr_302

frame .fr_303
label .l_303 -text "冬の大三角"
create_button .b_3031 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Winter~Triangle')"
create_button .b_3030 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Winter~Triangle')"
pack .l_303 -in .fr_303
pack .b_3030 .b_3031 -side left -expand 1 -fill x -in .fr_303

frame .fr_304
label .l_304 -text "冬のダイヤモンド"
create_button .b_3041 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Winter~Hexagon~(Winter~Circle)')"
create_button .b_3040 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Winter~Hexagon~(Winter~Circle)')"
pack .l_304 -in .fr_304
pack .b_3040 .b_3041 -side left -expand 1 -fill x -in .fr_304

#frame .fr_305
#label .l_305 -text "北極星"
#create_button .b_3051 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Polaris')"
#create_button .b_3050 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Polaris')"
#pack .l_305 -in .fr_305
#pack .b_3050 .b_3051 -side left -expand 1 -fill x -in .fr_305

#frame .fr_306
#label .l_306 -text "すばる" -font $font_small
#create_button_small .b_3061 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Preiades')"
#create_button_small .b_3060 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Preiades')"
#pack .l_306 -in .fr_306
#pack .b_3060 .b_3061 -side left -expand 1 -fill x -in .fr_306

frame .fr_307
label .l_307 -text "カシオペヤ座5倍線"
create_button .b_3071 " SET " "send_ssc_lines AsterismMgr.setFlagLines(true) AsterismMgr.selectAsterism('Cassio~5times')"
create_button .b_3070 "Unset" "send_ssc_lines AsterismMgr.deselectAsterism('Cassio~5times')"
pack .l_307 -in .fr_307
pack .b_3070 .b_3071 -side left -expand 1 -fill x -in .fr_307

#

frame .fr_astar_lbl30 -height [scaled_px 12]

#

frame .fr_astar_lbl31

label .l_astar_eachlbl32 -text "個別表示" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
create_button_small .b_astar_off32 "全Off" "unset_star_label Polaris M45 Betelgeuse Rigel Sirius Procyon Pollux Castor Capella Aldebaran"
pack .l_astar_eachlbl32 .b_astar_off32 -side left -expand 1 -fill both -in .fr_astar_lbl31

#

frame .fr_astar_lbl32

frame .fr_321
label .l_321 -text "北極星" -font $font_small
create_button_small .b_3211 "ON" "set_star_label Polaris"
create_button_small .b_3210 "Off" "unset_star_label Polaris"
pack .l_321 -in .fr_321
pack .b_3210 .b_3211 -side left -expand 1 -fill x -in .fr_321

frame .fr_322
label .l_322 -text "すばる" -font $font_small
create_button_small .b_3221 "ON" "set_star_label M45"
create_button_small .b_3220 "Off" "unset_star_label M45"
pack .l_322 -in .fr_322
pack .b_3220 .b_3221 -side left -expand 1 -fill x -in .fr_322

pack .fr_321 .fr_322 -side left -expand 1 -fill x -in .fr_astar_lbl32

#

frame .fr_astar_lbl33

frame .fr_323
label .l_323 -text "ベテルギウス" -font $font_small
create_button_small .b_3231 "ON" "set_star_label Betelgeuse"
create_button_small .b_3230 "Off" "unset_star_label Betelgeuse"
pack .l_323 -in .fr_323
pack .b_3230 .b_3231 -side left -expand 1 -fill x -in .fr_323

frame .fr_324
label .l_324 -text "リゲル" -font $font_small
create_button_small .b_3241 "ON" "set_star_label Rigel"
create_button_small .b_3240 "Off" "unset_star_label Rigel"
pack .l_324 -in .fr_324
pack .b_3240 .b_3241 -side left -expand 1 -fill x -in .fr_324

pack .fr_323 .fr_324 -side left -expand 1 -fill x -in .fr_astar_lbl33

#

frame .fr_astar_lbl34

frame .fr_325
label .l_325 -text "シリウス" -font $font_small
create_button_small .b_3251 "ON" "set_star_label Sirius"
create_button_small .b_3250 "Off" "unset_star_label Sirius"
pack .l_325 -in .fr_325
pack .b_3250 .b_3251 -side left -expand 1 -fill x -in .fr_325

frame .fr_326
label .l_326 -text "プロキオン" -font $font_small
create_button_small .b_3261 "ON" "set_star_label Procyon"
create_button_small .b_3260 "Off" "unset_star_label Procyon"
pack .l_326 -in .fr_326
pack .b_3260 .b_3261 -side left -expand 1 -fill x -in .fr_326

pack .fr_325 .fr_326 -side left -expand 1 -fill x -in .fr_astar_lbl34

#

frame .fr_astar_lbl35

frame .fr_327
label .l_327 -text "ポルックス" -font $font_small
create_button_small .b_3271 "ON" "set_star_label Pollux"
create_button_small .b_3270 "Off" "unset_star_label Pollux"
pack .l_327 -in .fr_327
pack .b_3270 .b_3271 -side left -expand 1 -fill x -in .fr_327

frame .fr_328
label .l_328 -text "カストル" -font $font_small
create_button_small .b_3281 "ON" "set_star_label Castor"
create_button_small .b_3280 "Off" "unset_star_label Castor"
pack .l_328 -in .fr_328
pack .b_3280 .b_3281 -side left -expand 1 -fill x -in .fr_328

pack .fr_327 .fr_328 -side left -expand 1 -fill x -in .fr_astar_lbl35

#

frame .fr_astar_lbl36

frame .fr_329
label .l_329 -text "カペラ" -font $font_small
create_button_small .b_3291 "ON" "set_star_label Capella"
create_button_small .b_3290 "Off" "unset_star_label Capella"
pack .l_329 -in .fr_329
pack .b_3290 .b_3291 -side left -expand 1 -fill x -in .fr_329

frame .fr_330
label .l_330 -text "アルデバラン" -font $font_small
create_button_small .b_3301 "ON" "set_star_label Aldebaran"
create_button_small .b_3300 "Off" "unset_star_label Aldebaran"
pack .l_330 -in .fr_330
pack .b_3300 .b_3301 -side left -expand 1 -fill x -in .fr_330

pack .fr_329 .fr_330 -side left -expand 1 -fill x -in .fr_astar_lbl36

#

pack .fr_301 .fr_302 .fr_303 .fr_304 .fr_307 .fr_astar_lbl30 .fr_astar_lbl31 .fr_astar_lbl32 .fr_astar_lbl33 .fr_astar_lbl34 .fr_astar_lbl35 .fr_astar_lbl36 -expand 0 -fill x -in .lf_winter_asterism_star

.pw add .lf_winter_asterism_star

################################################################

labelframe .lf_solarsystem -text "Solar System" -font $font_labelframe

frame .fr_sol_disp
label .l_sol_disp -text "太陽系 表示"
create_button .b_sol_disp0 "Off" "unset_solarsystem"
create_button .b_sol_disp1 "ON" "exec_ssc_on_server setSolarSystem.ssc updateLightMoonPollution.ssc"
pack .l_sol_disp -in .fr_sol_disp
pack .b_sol_disp0 .b_sol_disp1 -side left -expand 1 -fill x -in .fr_sol_disp

frame .fr_sol_label
label .l_sol_label -text "太陽系 天体名称"
create_button .b_sol_label0 "Unset" "send_ssc_of_solsyslabels -"
entry .e_sol_label -width 4 -textvariable buffer_solsyslabels
set buffer_solsyslabels "0.0"
create_button .b_sol_label1 "＜" "send_ssc_of_solsyslabels -0.5"
create_button .b_sol_label2 "＞" "send_ssc_of_solsyslabels +0.5"
bind .e_sol_label <Return> "send_ssc_of_solsyslabels"
pack .l_sol_label -in .fr_sol_label
pack .b_sol_label0 -side left -fill both -in .fr_sol_label
pack .e_sol_label -expand 1 -side left -fill both -in .fr_sol_label
pack .b_sol_label1 .b_sol_label2 -side left -fill both -in .fr_sol_label

frame .fr_sol_porbit
label .l_sol_porbit -text "惑星 軌道"
create_button .b_sol_porbit0 "Unset" "send_ssc_lines SolarSystem.setFlagOrbits(false)"
create_button .b_sol_porbit1 "SET" "send_or_exec_ssc SolarSystem.setOrbitColorStyle('one_color') setPlanetaryOrbits.ssc"
create_button .b_sol_porbit2 "色" "send_or_exec_ssc setPlanetaryOrbits.ssc SolarSystem.setOrbitColorStyle('major_planets_minor_types')"
create_button .b_sol_porbit3 "月" "send_or_exec_ssc SolarSystem.setOrbitColorStyle('major_planets_minor_types') setMoonPlanetaryOrbits.ssc"
pack .l_sol_porbit -in .fr_sol_porbit
pack .b_sol_porbit0 .b_sol_porbit1 .b_sol_porbit2 .b_sol_porbit3 -side left -expand 1 -fill x -in .fr_sol_porbit

frame .fr_sol_eorbit
label .l_sol_eorbit -text "地球 軌道"
create_button .b_sol_eorbit0 "Unset" "exec_ssc_on_server unsetEarthOrbitLabels.ssc unsetEarthOrbit.ssc"
create_button .b_sol_eorbit1 "SET" "exec_ssc_on_server unsetEarthOrbitLabels.ssc setEarthOrbit.ssc"
create_button .b_sol_eorbit2 "DATE" "exec_ssc_on_server setEarthOrbit.ssc setEarthOrbitLabels.ssc"
pack .l_sol_eorbit -in .fr_sol_eorbit
pack .b_sol_eorbit0 .b_sol_eorbit1 .b_sol_eorbit2 -side left -expand 1 -fill x -in .fr_sol_eorbit

frame .fr_sol_pscale
label .l_sol_pscale -text "天体 表示サイズ"
create_button .b_sol_pscale0 "惑星1x" "send_ssc_lines SolarSystem.setPlanetScale(1) SolarSystem.setFlagPlanetScale(false)"
create_button .b_sol_pscale1 "400x" "send_ssc_lines SolarSystem.setFlagPlanetScale(true) SolarSystem.setPlanetScale(400)"
create_button .b_sol_pscale2 "999x" "send_ssc_lines SolarSystem.setFlagPlanetScale(true) SolarSystem.setPlanetScale(999)"
pack .l_sol_pscale -in .fr_sol_pscale
pack .b_sol_pscale0 .b_sol_pscale1 .b_sol_pscale2 -side left -expand 1 -fill x -in .fr_sol_pscale

frame .fr_sol_mscale
#label .l_sol_mscale -text "月 表示サイズ" -font $font_small
create_button_small .b_sol_mscale0 "月1x" "send_ssc_lines SolarSystem.setMoonScale(1) SolarSystem.setFlagMoonScale(false)"
create_button_small .b_sol_mscale1 "初期値" "exec_ssc_on_server localInitMoonScale.ssc"
create_button_small .b_sol_mscale2 "32x" "send_ssc_lines SolarSystem.setFlagMoonScale(true) SolarSystem.setMoonScale(32)"
#pack .l_sol_mscale -in .fr_sol_mscale
pack .b_sol_mscale0 .b_sol_mscale1 .b_sol_mscale2 -side left -expand 1 -fill x -in .fr_sol_mscale

frame .fr_sol_sscale
#label .l_sol_sscale -text "太陽 表示サイズ" -font $font_small
create_button_small .b_sol_sscale0 "太陽1x" "send_ssc_lines SolarSystem.setSunScale(1) SolarSystem.setFlagSunScale(false)"
create_button_small .b_sol_sscale1 "初期値" "exec_ssc_on_server localInitSunScale.ssc"
create_button_small .b_sol_sscale2 "32x" "send_ssc_lines SolarSystem.setFlagSunScale(true) SolarSystem.setSunScale(32)"
#pack .l_sol_sscale -in .fr_sol_sscale
pack .b_sol_sscale0 .b_sol_sscale1 .b_sol_sscale2 -side left -expand 1 -fill x -in .fr_sol_sscale

#

frame .fr_sol_margin -height [scaled_px 4]

frame .fr_sol_lbl00

label .l_sol_eachlabel -text "名称 個別表示" -foreground $label_inverted_foreground -background $label_inverted_background
#create_button .b_sol_label000_0 "Off" "unset_star_label Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto"
pack .l_sol_eachlabel -side left -expand 1 -fill x -in .fr_sol_lbl00

#

frame .fr_sol_lbl01

frame .fr_sol_label000
label .l_sol_label000 -text "　全　" -font $font_small
create_button_small .b_sol_label000_1 "ON" "set_star_label Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto"
create_button_small .b_sol_label000_0 "Off" "unset_star_label Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto"
pack .l_sol_label000 -in .fr_sol_label000
pack .b_sol_label000_0 .b_sol_label000_1 -side left -expand 1 -fill x -in .fr_sol_label000

frame .fr_sol_label001
label .l_sol_label001 -text "太陽" -font $font_small
create_button_small .b_sol_label001_1 "ON" "set_star_label Sun"
create_button_small .b_sol_label001_0 "Off" "unset_star_label Sun"
pack .l_sol_label001 -in .fr_sol_label001
pack .b_sol_label001_0 .b_sol_label001_1 -side left -expand 1 -fill x -in .fr_sol_label001

pack .fr_sol_label000 .fr_sol_label001 -side left -expand 1 -fill x -in .fr_sol_lbl01

#

frame .fr_sol_lbl02

frame .fr_sol_label002
label .l_sol_label002 -text "月" -font $font_small
create_button_small .b_sol_label002_1 "ON" "set_star_label Moon"
create_button_small .b_sol_label002_0 "Off" "unset_star_label Moon"
pack .l_sol_label002 -in .fr_sol_label002
pack .b_sol_label002_0 .b_sol_label002_1 -side left -expand 1 -fill x -in .fr_sol_label002

frame .fr_sol_label003
label .l_sol_label003 -text "水星" -font $font_small
create_button_small .b_sol_label003_1 "ON" "set_star_label Mercury"
create_button_small .b_sol_label003_0 "Off" "unset_star_label Mercury"
pack .l_sol_label003 -in .fr_sol_label003
pack .b_sol_label003_0 .b_sol_label003_1 -side left -expand 1 -fill x -in .fr_sol_label003

pack .fr_sol_label002 .fr_sol_label003 -side left -expand 1 -fill x -in .fr_sol_lbl02

#

frame .fr_sol_lbl03

frame .fr_sol_label004
label .l_sol_label004 -text "金星" -font $font_small
create_button_small .b_sol_label004_1 "ON" "set_star_label Venus"
create_button_small .b_sol_label004_0 "Off" "unset_star_label Venus"
pack .l_sol_label004 -in .fr_sol_label004
pack .b_sol_label004_0 .b_sol_label004_1 -side left -expand 1 -fill x -in .fr_sol_label004

frame .fr_sol_label005
label .l_sol_label005 -text "火星" -font $font_small
create_button_small .b_sol_label005_1 "ON" "set_star_label Mars"
create_button_small .b_sol_label005_0 "Off" "unset_star_label Mars"
pack .l_sol_label005 -in .fr_sol_label005
pack .b_sol_label005_0 .b_sol_label005_1 -side left -expand 1 -fill x -in .fr_sol_label005

pack .fr_sol_label004 .fr_sol_label005 -side left -expand 1 -fill x -in .fr_sol_lbl03

#

frame .fr_sol_lbl04

frame .fr_sol_label006
label .l_sol_label006 -text "木星" -font $font_small
create_button_small .b_sol_label006_1 "ON" "set_star_label Jupiter"
create_button_small .b_sol_label006_0 "Off" "unset_star_label Jupiter"
pack .l_sol_label006 -in .fr_sol_label006
pack .b_sol_label006_0 .b_sol_label006_1 -side left -expand 1 -fill x -in .fr_sol_label006

frame .fr_sol_label007
label .l_sol_label007 -text "土星" -font $font_small
create_button_small .b_sol_label007_1 "ON" "set_star_label Saturn"
create_button_small .b_sol_label007_0 "Off" "unset_star_label Saturn"
pack .l_sol_label007 -in .fr_sol_label007
pack .b_sol_label007_0 .b_sol_label007_1 -side left -expand 1 -fill x -in .fr_sol_label007

pack .fr_sol_label006 .fr_sol_label007 -side left -expand 1 -fill x -in .fr_sol_lbl04

#

frame .fr_sol_lbl05

frame .fr_sol_label008
label .l_sol_label008 -text "天王星" -font $font_small
create_button_small .b_sol_label008_1 "ON" "set_star_label Uranus"
create_button_small .b_sol_label008_0 "Off" "unset_star_label Uranus"
pack .l_sol_label008 -in .fr_sol_label008
pack .b_sol_label008_0 .b_sol_label008_1 -side left -expand 1 -fill x -in .fr_sol_label008

frame .fr_sol_label009
label .l_sol_label009 -text "海王星" -font $font_small
create_button_small .b_sol_label009_1 "ON" "set_star_label Neptune"
create_button_small .b_sol_label009_0 "Off" "unset_star_label Neptune"
pack .l_sol_label009 -in .fr_sol_label009
pack .b_sol_label009_0 .b_sol_label009_1 -side left -expand 1 -fill x -in .fr_sol_label009

pack .fr_sol_label008 .fr_sol_label009 -side left -expand 1 -fill x -in .fr_sol_lbl05

#

pack .fr_sol_disp .fr_sol_label .fr_sol_porbit .fr_sol_eorbit .fr_sol_pscale .fr_sol_mscale .fr_sol_sscale -expand 0 -fill x -in .lf_solarsystem
pack .fr_sol_margin .fr_sol_lbl00 .fr_sol_lbl01 .fr_sol_lbl02 .fr_sol_lbl03 .fr_sol_lbl04 .fr_sol_lbl05 -expand 0 -fill x -in .lf_solarsystem

.pw add .lf_solarsystem


################################################################

labelframe .lf_dso_p0 -text "Objects" -font $font_labelframe

frame .fr_nebula
label .l_dso_list -text "一覧" -foreground $label_inverted_foreground -background $label_inverted_background
label .l_nebula -text "星雲・星団 表示"
create_button .b_nebula1 "ON" "send_ssc_lines NebulaMgr.setFlagHints(true) NebulaMgr.setFlagShow(true)"
create_button .b_nebula0 "Off" "send_ssc_lines NebulaMgr.setFlagHints(false) NebulaMgr.setFlagShow(false)"
pack .l_dso_list .l_nebula -fill x -in .fr_nebula
pack .b_nebula0 .b_nebula1 -side left -expand 1 -fill x -in .fr_nebula

frame .fr_dsotype
label .l_dsotype -text "カタログ選択"
create_button .b_dsotype1 "NGC" "send_ssc_lines NebulaMgr.setCatalogFilters(1)"
create_button .b_dsotype2 "IC" "send_ssc_lines NebulaMgr.setCatalogFilters(2)"
create_button .b_dsotype4 " M " "send_ssc_lines NebulaMgr.setCatalogFilters(4)"
pack .l_dsotype -in .fr_dsotype
pack .b_dsotype4 .b_dsotype1 .b_dsotype2 -side left -expand 1 -fill x -in .fr_dsotype

frame .fr_dsohint
label .l_dsohint -text "ヒント"
create_button .b_dsohint0 "０" "send_ssc_of_dsohints -"
entry .e_dsohint -width 5 -textvariable buffer_dsohints
set buffer_dsohints "0.0"
create_button .b_dsohint1 "＜" "send_ssc_of_dsohints -0.5"
create_button .b_dsohint2 "＞" "send_ssc_of_dsohints +0.5"
bind .e_dsohint <Return> "send_ssc_of_dsohints"
pack .l_dsohint -in .fr_dsohint
pack .b_dsohint0 -side left -fill both -in .fr_dsohint
pack .e_dsohint -expand 1 -side left -fill both -in .fr_dsohint
pack .b_dsohint1 .b_dsohint2 -side left -fill both -in .fr_dsohint

frame .fr_dsolbl
label .l_dsolbl -text "名称"
create_button .b_dsolbl0 "０" "send_ssc_of_dsolabels -"
entry .e_dsolbl -width 5 -textvariable buffer_dsolabels
set buffer_dsolabels "0.0"
create_button .b_dsolbl1 "＜" "send_ssc_of_dsolabels -0.5"
create_button .b_dsolbl2 "＞" "send_ssc_of_dsolabels +0.5"
bind .e_dsolbl <Return> "send_ssc_of_dsolabels"
pack .l_dsolbl -in .fr_dsolbl
pack .b_dsolbl0 -side left -fill both -in .fr_dsolbl
pack .e_dsolbl -expand 1 -side left -fill both -in .fr_dsolbl
pack .b_dsolbl1 .b_dsolbl2 -side left -fill both -in .fr_dsolbl

frame .fr_dsoeach_margin -height [scaled_px 6]

frame .fr_dsoinit_p0
label .l_dso_each -text "個別" -foreground $label_inverted_foreground -background $label_inverted_background
label .l_dsoinit_p0 -text "リセット"
create_button .b_dsoinit_p0m "マーカー" "send_ssc_lines core.setSelectedObjectMarkerVisible(false)"
#create_button .b_dsoinit_p0v "VIEW" "exec_ssc_on_server localInitView.ssc"
create_button .b_dsoinit_p0v "VIEW" "send_ssc_to_initialize_view 0 -"
pack .l_dso_each .l_dsoinit_p0 -fill x -in .fr_dsoinit_p0
if { $flag_dome_mode != 0 } {
  pack .b_dsoinit_p0m -side left -expand 1 -fill x -in .fr_dsoinit_p0
}
pack .b_dsoinit_p0v -side left -expand 1 -fill x -in .fr_dsoinit_p0

frame .fr_dsomarker
label .l_dsomarker -text "天体選択"
entry .e_dsomarker -width 8 -textvariable buffer_selected_dsoname
bind .e_dsomarker <Return> "select_a_dso"
create_button .b_dsomarker "SET" "select_a_dso"
pack .l_dsomarker -in .fr_dsomarker
pack .e_dsomarker .b_dsomarker -side left -expand 1 -fill both -in .fr_dsomarker

frame .fr_dsozoom
label .l_dsozoom -text "移動&ズーム(度)"
frame .fr_dsozoom_b0
create_button .b_dsozoom32 "32" "exec_ssc_on_server moveZoomToSelectedFov32.ssc"
create_button .b_dsozoom16 "16" "exec_ssc_on_server moveZoomToSelectedFov16.ssc"
create_button .b_dsozoom8 "8" "exec_ssc_on_server moveZoomToSelectedFov8.ssc"
create_button .b_dsozoom4 "4" "exec_ssc_on_server moveZoomToSelectedFov4.ssc"
create_button .b_dsozoom2 "2" "exec_ssc_on_server moveZoomToSelectedFov2.ssc"
create_button .b_dsozoom1 "1" "exec_ssc_on_server moveZoomToSelectedFov1.ssc"
pack .b_dsozoom32 .b_dsozoom16 .b_dsozoom8 .b_dsozoom4 .b_dsozoom2 .b_dsozoom1 -side left -expand 1 -fill x -in .fr_dsozoom_b0
frame .fr_dsozoom_b1
create_button_small .b_dsozoom18m "0.3" "exec_ssc_on_server moveZoomToSelectedFov0.3.ssc"
create_button_small .b_dsozoom6m "0.1" "exec_ssc_on_server moveZoomToSelectedFov0.1.ssc"
create_button_small .b_dsozoom2m "0.03" "exec_ssc_on_server moveZoomToSelectedFov0.03.ssc"
create_button_small .b_dsozoom06m "0.01" "exec_ssc_on_server moveZoomToSelectedFov0.01.ssc"
pack .b_dsozoom18m .b_dsozoom6m .b_dsozoom2m .b_dsozoom06m -side left -expand 1 -fill x -in .fr_dsozoom_b1
pack .l_dsozoom -in .fr_dsozoom
pack .fr_dsozoom_b0 .fr_dsozoom_b1 -expand 1 -fill x -in .fr_dsozoom

set buffer_dso_fov_circular_size "1.0000"
frame .fr_dsofovc

frame .fr_dsofovc_b0
label .l_dsofovc -text "視野円(度)"
create_button_small .b_dso_ed_fovc "編集" "open_text_editor .edt_fov list_fov-circular.txt"
pack .l_dsofovc -expand 1 -side left -fill both -in .fr_dsofovc_b0
pack .b_dso_ed_fovc -side left -fill both -in .fr_dsofovc_b0

frame .fr_dsofovc_b1
create_button .b_dsofovc0 "Off" "send_ssc_lines SpecialMarkersMgr.setFlagFOVCircularMarker(false)"
entry .e_dsofovc -width 6 -textvariable buffer_dso_fov_circular_size
create_button .b_dsofovc1 "ON" "send_ssc_of_dso_fov_circular_marker"
bind .e_dsofovc <Return> "send_ssc_of_dso_fov_circular_marker"
pack .b_dsofovc0 -side left -fill both -in .fr_dsofovc_b1
pack .e_dsofovc -expand 1 -side left -fill both -in .fr_dsofovc_b1
pack .b_dsofovc1 -side left -fill both -in .fr_dsofovc_b1

#frame .fr_dsofovc_b2
#create_button_small .b_dsofovc_t0 "6x" "send_ssc_of_dso_fov_circular_marker 8.0"
#create_button_small .b_dsofovc_t1 "45x" "send_ssc_of_dso_fov_circular_marker 1.0"
#create_button_small .b_dsofovc_t2 "100x" "send_ssc_of_dso_fov_circular_marker 0.45"
#create_button_small .b_dsofovc_t3 "200x" "send_ssc_of_dso_fov_circular_marker 0.225"
#pack .b_dsofovc_t0 .b_dsofovc_t1 .b_dsofovc_t2 .b_dsofovc_t3 -expand 1 -side left -fill x -in .fr_dsofovc_b2

pack .fr_dsofovc_b0 .fr_dsofovc_b1 -expand 1 -fill x -in .fr_dsofovc

pack .fr_nebula .fr_dsotype .fr_dsohint .fr_dsolbl .fr_dsoeach_margin .fr_dsoinit_p0 .fr_dsomarker .fr_dsozoom .fr_dsofovc -expand 0 -fill x -in .lf_dso_p0

build_dso_buttons "list_fov-circular.txt" "send_ssc_of_dso_fov_circular_marker" 4 "fovc" "p0"

.pw add .lf_dso_p0


################################################################

labelframe .lf_dso_p1 -text "Objects 1" -font $font_labelframe

frame .fr_dso_spring
label .l_dso_spring -text "春"
create_button_small .b_dso_ed_spring "編集" "open_text_editor .edt_spri objects_spring.txt"
pack .l_dso_spring -side left -expand 1 -fill x -in .fr_dso_spring
pack .b_dso_ed_spring -side left -expand 0 -fill x -in .fr_dso_spring
pack .fr_dso_spring -expand 0 -fill x -in .lf_dso_p1

build_dso_buttons "objects_spring.txt" "select_a_dso" 3 "spring" "p1"

frame .fr_dso_p1_margin00 -height [scaled_px 3]
pack .fr_dso_p1_margin00 -expand 0 -fill x -in .lf_dso_p1

frame .fr_dso_summer
label .l_dso_summer -text "夏"
create_button_small .b_dso_ed_summer "編集" "open_text_editor .edt_summ objects_summer.txt"
pack .l_dso_summer -side left -expand 1 -fill x -in .fr_dso_summer
pack .b_dso_ed_summer -side left -expand 0 -fill x -in .fr_dso_summer
pack .fr_dso_summer -expand 0 -fill x -in .lf_dso_p1

build_dso_buttons "objects_summer.txt" "select_a_dso" 3 "summer" "p1"

frame .fr_dso_p1_margin01 -height [scaled_px 3]
pack .fr_dso_p1_margin01 -expand 0 -fill x -in .lf_dso_p1

frame .fr_dso_etc1
label .l_dso_etc1 -text "etc.1"
create_button_small .b_dso_ed_etc1 "編集" "open_text_editor .edt_etc1 objects_etc1.txt"
pack .l_dso_etc1 -side left -expand 1 -fill x -in .fr_dso_etc1
pack .b_dso_ed_etc1 -side left -expand 0 -fill x -in .fr_dso_etc1
pack .fr_dso_etc1 -expand 0 -fill x -in .lf_dso_p1

build_dso_buttons "objects_etc1.txt" "select_a_dso" 3 "etc1_" "p1"

.pw add .lf_dso_p1


################################################################

labelframe .lf_dso_p2 -text "Objects 2" -font $font_labelframe

frame .fr_dso_autumn
label .l_dso_autumn -text "秋"
create_button_small .b_dso_ed_autumn "編集" "open_text_editor .edt_autu objects_autumn.txt"
pack .l_dso_autumn -side left -expand 1 -fill x -in .fr_dso_autumn
pack .b_dso_ed_autumn -side left -expand 0 -fill x -in .fr_dso_autumn
pack .fr_dso_autumn -expand 0 -fill x -in .lf_dso_p2

build_dso_buttons "objects_autumn.txt" "select_a_dso" 3 "autumn" "p2"

frame .fr_dso_p2_margin00 -height [scaled_px 3]
pack .fr_dso_p2_margin00 -expand 0 -fill x -in .lf_dso_p2

frame .fr_dso_winter
label .l_dso_winter -text "冬"
create_button_small .b_dso_ed_winter "編集" "open_text_editor .edt_wint objects_winter.txt"
pack .l_dso_winter -side left -expand 1 -fill x -in .fr_dso_winter
pack .b_dso_ed_winter -side left -expand 0 -fill x -in .fr_dso_winter
pack .fr_dso_winter -expand 0 -fill x -in .lf_dso_p2

build_dso_buttons "objects_winter.txt" "select_a_dso" 3 "winter" "p2"

frame .fr_dso_p2_margin01 -height [scaled_px 3]
pack .fr_dso_p2_margin01 -expand 0 -fill x -in .lf_dso_p2

frame .fr_dso_etc2
label .l_dso_etc2 -text "etc.2"
create_button_small .b_dso_ed_etc2 "編集" "open_text_editor .edt_etc2 objects_etc2.txt"
pack .l_dso_etc2 -side left -expand 1 -fill x -in .fr_dso_etc2
pack .b_dso_ed_etc2 -side left -expand 0 -fill x -in .fr_dso_etc2
pack .fr_dso_etc2 -expand 0 -fill x -in .lf_dso_p2

build_dso_buttons "objects_etc2.txt" "select_a_dso" 3 "etc2_" "p2"

.pw add .lf_dso_p2


################################################################

labelframe .lf_gridlines -text "Grid Lines" -font $font_labelframe

frame .fr_grid_disp
label .l_grid_disp -text "座標線 表示"
create_button .b_grid_disp1 "ON" "exec_ssc_on_server setFlagGridLines.ssc"
create_button .b_grid_disp0 "Off" "exec_ssc_on_server unsetFlagGridLines.ssc"
pack .l_grid_disp -in .fr_grid_disp
pack .b_grid_disp0 .b_grid_disp1 -side left -expand 1 -fill x -in .fr_grid_disp

frame .fr_grid_azim
label .l_grid_azim -text "高度方位"
create_button .b_grid_azim1 " SET " "send_ssc_lines GridLinesMgr.setFlagAzimuthalGrid(true)"
create_button .b_grid_azim0 "Unset" "send_ssc_lines GridLinesMgr.setFlagAzimuthalGrid(false)"
pack .l_grid_azim -in .fr_grid_azim
pack .b_grid_azim0 .b_grid_azim1 -side left -expand 1 -fill x -in .fr_grid_azim

frame .fr_grid_eclip
label .l_grid_eclip -text "黄経/黄緯"
create_button .b_grid_eclip1 " SET " "send_ssc_lines GridLinesMgr.setFlagEclipticGrid(true)"
create_button .b_grid_eclip0 "Unset" "send_ssc_lines GridLinesMgr.setFlagEclipticGrid(false)"
pack .l_grid_eclip -in .fr_grid_eclip
pack .b_grid_eclip0 .b_grid_eclip1 -side left -expand 1 -fill x -in .fr_grid_eclip

frame .fr_grid_eclipl
label .l_grid_eclipl -text "黄道"
create_button .b_grid_eclipl2 "DATE" "send_or_exec_ssc GridLinesMgr.setFlagEclipticLine(true) setEclipticDateLabels.ssc"
create_button .b_grid_eclipl1 "SET" "send_or_exec_ssc unsetEclipticDateLabels.ssc GridLinesMgr.setFlagEclipticLine(true)"
create_button .b_grid_eclipl0 "Unset" "send_or_exec_ssc unsetEclipticDateLabels.ssc GridLinesMgr.setFlagEclipticLine(false)"
pack .l_grid_eclipl -in .fr_grid_eclipl
pack .b_grid_eclipl0 .b_grid_eclipl1 .b_grid_eclipl2 -side left -expand 1 -fill x -in .fr_grid_eclipl

frame .fr_grid_equat
label .l_grid_equat -text "赤経/赤緯"
create_button .b_grid_equat1 " SET " "send_ssc_lines GridLinesMgr.setFlagEquatorGrid(true)"
create_button .b_grid_equat0 "Unset" "send_ssc_lines GridLinesMgr.setFlagEquatorGrid(false)"
pack .l_grid_equat -in .fr_grid_equat
pack .b_grid_equat0 .b_grid_equat1 -side left -expand 1 -fill x -in .fr_grid_equat

frame .fr_grid_equatl
label .l_grid_equatl -text "天の赤道"
create_button .b_grid_equatl1 " SET " "send_ssc_lines GridLinesMgr.setFlagEquatorLine(true)"
create_button .b_grid_equatl0 "Unset" "send_ssc_lines GridLinesMgr.setFlagEquatorLine(false)"
pack .l_grid_equatl -in .fr_grid_equatl
pack .b_grid_equatl0 .b_grid_equatl1 -side left -expand 1 -fill x -in .fr_grid_equatl

frame .fr_grid_sundnl
label .l_grid_sundnl -text "太陽 日周運動"
create_button .b_grid_sundnl1 " SET " "exec_ssc_on_server setSunDiurnalMotion.ssc"
create_button .b_grid_sundnl0 "Unset" "exec_ssc_on_server unsetSunDiurnalMotion.ssc"
pack .l_grid_sundnl -in .fr_grid_sundnl
pack .b_grid_sundnl0 .b_grid_sundnl1 -side left -expand 1 -fill x -in .fr_grid_sundnl

frame .fr_grid_galact
label .l_grid_galact -text "銀経/銀緯"
create_button .b_grid_galact1 " SET " "send_ssc_lines GridLinesMgr.setFlagGalacticGrid(true)"
create_button .b_grid_galact0 "Unset" "send_ssc_lines GridLinesMgr.setFlagGalacticGrid(false)"
pack .l_grid_galact -in .fr_grid_galact
pack .b_grid_galact0 .b_grid_galact1 -side left -expand 1 -fill x -in .fr_grid_galact

frame .fr_grid_galactl
label .l_grid_galactl -text "銀河面"
create_button .b_grid_galactl1 " SET " "send_ssc_lines GridLinesMgr.setFlagGalacticEquatorLine(true)"
create_button .b_grid_galactl0 "Unset" "send_ssc_lines GridLinesMgr.setFlagGalacticEquatorLine(false)"
pack .l_grid_galactl -in .fr_grid_galactl
pack .b_grid_galactl0 .b_grid_galactl1 -side left -expand 1 -fill x -in .fr_grid_galactl

pack .fr_grid_disp .fr_grid_azim .fr_grid_eclip .fr_grid_eclipl .fr_grid_equat .fr_grid_equatl .fr_grid_sundnl .fr_grid_galact .fr_grid_galactl -expand 0 -fill x -in .lf_gridlines

.pw add .lf_gridlines


################################################################

labelframe .lf_config -text "Configuration" -font $font_labelframe

set buffer_projection_mode $asarr_initial_settings(ProjectionMode)
set pjmod {
  "ProjectionPerspective"
  "ProjectionEqualArea"
  "ProjectionStereographic"
  "ProjectionFisheye"
  "ProjectionHammer"
  "ProjectionCylinder"
  "ProjectionMercator"
  "ProjectionOrthographic"
  "ProjectionSinusoidal"
  "ProjectionMiller"
}
frame .fr_mnt_pjmod
label .l_mnt_pjmod -text "Projection Mode" -font $font_small -pady 0
ttk::combobox .cb_mnt_pjmod -textvariable buffer_projection_mode -value $pjmod -font $font_small -state readonly
bind .cb_mnt_pjmod <<ComboboxSelected>> { send_ssc_of_projection_mode }
pack .l_mnt_pjmod -in .fr_mnt_pjmod
pack .cb_mnt_pjmod -side left -expand 1 -fill x -in .fr_mnt_pjmod

frame .fr_mnt_gui
label .l_mnt_gui -text "GUI of Main Window" -font $font_small -pady 0
create_button_small .b_mnt_gui0 "Off" "send_ssc_lines StelMovementMgr.setFlagEnableMouseZooming(false) StelMovementMgr.setFlagEnableMouseNavigation(false) core.setSelectedObjectMarkerVisible(false) core.setGuiVisible(false) core.setSelectedObjectInfo('None')"
create_button_small .b_mnt_gui1 "ON" "send_ssc_lines StelMovementMgr.setFlagEnableMouseZooming(true) StelMovementMgr.setFlagEnableMouseNavigation(true) core.setSelectedObjectMarkerVisible(true) core.setGuiVisible(true) core.setSelectedObjectInfo('AllInfo')"
pack .l_mnt_gui -in .fr_mnt_gui
pack .b_mnt_gui0 .b_mnt_gui1 -side left -expand 1 -fill x -in .fr_mnt_gui

#frame .fr_mnt_msz
#label .l_mnt_msz -text "Mouse Zoom & Navi" -font $font_small
#create_button_small .b_mnt_msz0 "Off" "send_ssc_lines StelMovementMgr.setFlagEnableMouseZooming(false) StelMovementMgr.setFlagEnableMouseNavigation(false)"
#create_button_small .b_mnt_msz1 "ON" "send_ssc_lines StelMovementMgr.setFlagEnableMouseZooming(true) StelMovementMgr.setFlagEnableMouseNavigation(true)"
#pack .l_mnt_msz -in .fr_mnt_msz
#pack .b_mnt_msz0 .b_mnt_msz1 -side left -expand 1 -fill x -in .fr_mnt_msz

set buffer_display_gamma $asarr_initial_settings(DisplayGamma)
frame .fr_mnt_gamma
label .l_mnt_gamma -text "Display Gamma" -font $font_small -pady 0
create_button_small .b_mnt_gamma0 "初" "send_ssc_of_display_gamma -"
entry .e_mnt_gamma -width 6 -textvariable buffer_display_gamma -font $font_small
#create_button_small .b_mnt_gamma1 "SET" "send_ssc_of_display_gamma"
create_button_small .b_mnt_gamma1 "≪" "send_ssc_of_display_gamma -0.1"
create_button_small .b_mnt_gamma2 "<" "send_ssc_of_display_gamma -0.01"
create_button_small .b_mnt_gamma3 ">" "send_ssc_of_display_gamma +0.01"
create_button_small .b_mnt_gamma4 "≫" "send_ssc_of_display_gamma +0.1"
bind .e_mnt_gamma <Return> "send_ssc_of_display_gamma"
pack .l_mnt_gamma -in .fr_mnt_gamma
pack .b_mnt_gamma0 -side left -fill both -in .fr_mnt_gamma
pack .e_mnt_gamma -expand 1 -side left -fill both -in .fr_mnt_gamma
pack .b_mnt_gamma1 .b_mnt_gamma2 .b_mnt_gamma3 .b_mnt_gamma4 -side left -fill both -in .fr_mnt_gamma

set buffer_disk_viewport $asarr_initial_settings(DiskViewport)
frame .fr_mnt_diskview
label .l_mnt_diskview -text "Disk Viewport" -font $font_small -pady 0
create_button_small .b_mnt_diskview0 "初期値" "send_ssc_of_disk_viewport -"
create_button_small .b_mnt_diskview1 "Off" "send_ssc_of_disk_viewport false"
create_button_small .b_mnt_diskview2 "ON" "send_ssc_of_disk_viewport true"
pack .l_mnt_diskview -in .fr_mnt_diskview
pack .b_mnt_diskview0 -side left -fill x -in .fr_mnt_diskview
pack .b_mnt_diskview1 .b_mnt_diskview2 -expand 1 -side left -fill x -in .fr_mnt_diskview

set buffer_view_fov $asarr_initial_settings(ViewFov)
frame .fr_mnt_fov
label .l_mnt_fov -text "View FOV" -font $font_small -pady 0
create_button_small .b_mnt_fov0 "初" "send_ssc_of_view_fov -"
entry .e_mnt_fov -width 6 -textvariable buffer_view_fov -font $font_small
#create_button_small .b_mnt_fov1 "SET" "send_ssc_of_view_fov"
create_button_small .b_mnt_fov1 "≪" "send_ssc_of_view_fov -5.0"
create_button_small .b_mnt_fov2 "<" "send_ssc_of_view_fov -0.5"
create_button_small .b_mnt_fov3 ">" "send_ssc_of_view_fov +0.5"
create_button_small .b_mnt_fov4 "≫" "send_ssc_of_view_fov +5.0"
bind .e_mnt_fov <Return> "send_ssc_of_view_fov"
pack .l_mnt_fov -in .fr_mnt_fov
pack .b_mnt_fov0 -side left -fill both -in .fr_mnt_fov
pack .e_mnt_fov -expand 1 -side left -fill both -in .fr_mnt_fov
pack .b_mnt_fov1 .b_mnt_fov2 .b_mnt_fov3 .b_mnt_fov4 -side left -fill both -in .fr_mnt_fov

set buffer_viewport_x $asarr_initial_settings(ViewportX)
frame .fr_mnt_vpx
label .l_mnt_vpx -text "Viewport X" -font $font_small -pady 0
create_button_small .b_mnt_vpx0 "初" "send_ssc_of_viewport_x -"
entry .e_mnt_vpx -width 6 -textvariable buffer_viewport_x -font $font_small
#create_button_small .b_mnt_vpx1 "SET" "send_ssc_of_viewport_x"
create_button_small .b_mnt_vpx1 "≪" "send_ssc_of_viewport_x -0.1"
create_button_small .b_mnt_vpx2 "<" "send_ssc_of_viewport_x -0.01"
create_button_small .b_mnt_vpx3 ">" "send_ssc_of_viewport_x +0.01"
create_button_small .b_mnt_vpx4 "≫" "send_ssc_of_viewport_x +0.1"
bind .e_mnt_vpx <Return> "send_ssc_of_viewport_x"
pack .l_mnt_vpx -in .fr_mnt_vpx
pack .b_mnt_vpx0 -side left -fill both -in .fr_mnt_vpx
pack .e_mnt_vpx -expand 1 -side left -fill both -in .fr_mnt_vpx
pack .b_mnt_vpx1 .b_mnt_vpx2 .b_mnt_vpx3 .b_mnt_vpx4 -side left -fill both -in .fr_mnt_vpx

set buffer_viewport_y $asarr_initial_settings(ViewportY)
frame .fr_mnt_vpy
label .l_mnt_vpy -text "Viewport Y" -font $font_small -pady 0
create_button_small .b_mnt_vpy0 "初" "send_ssc_of_viewport_y -"
entry .e_mnt_vpy -width 6 -textvariable buffer_viewport_y -font $font_small
#create_button_small .b_mnt_vpy1 "SET" "send_ssc_of_viewport_y"
create_button_small .b_mnt_vpy1 "≪" "send_ssc_of_viewport_y -0.1"
create_button_small .b_mnt_vpy2 "<" "send_ssc_of_viewport_y -0.01"
create_button_small .b_mnt_vpy3 ">" "send_ssc_of_viewport_y +0.01"
create_button_small .b_mnt_vpy4 "≫" "send_ssc_of_viewport_y +0.1"
bind .e_mnt_vpy <Return> "send_ssc_of_viewport_y"
pack .l_mnt_vpy -in .fr_mnt_vpy
pack .b_mnt_vpy0 -side left -fill both -in .fr_mnt_vpy
pack .e_mnt_vpy -expand 1 -side left -fill both -in .fr_mnt_vpy
pack .b_mnt_vpy1 .b_mnt_vpy2 .b_mnt_vpy3 .b_mnt_vpy4 -side left -fill both -in .fr_mnt_vpy

set buffer_view_altitude $asarr_initial_settings(ViewAltitude)
frame .fr_mnt_altitude
label .l_mnt_altitude -text "View Altitude" -font $font_small -pady 0
create_button_small .b_mnt_altitude0 "初" "send_ssc_of_view_altitude -"
entry .e_mnt_altitude -width 6 -textvariable buffer_view_altitude -font $font_small
#create_button_small .b_mnt_altitude1 "SET" "send_ssc_of_view_altitude"
create_button_small .b_mnt_altitude1 "≪" "send_ssc_of_view_altitude -5.0"
create_button_small .b_mnt_altitude2 "<" "send_ssc_of_view_altitude -0.5"
create_button_small .b_mnt_altitude3 ">" "send_ssc_of_view_altitude +0.5"
create_button_small .b_mnt_altitude4 "≫" "send_ssc_of_view_altitude +5.0"
bind .e_mnt_altitude <Return> "send_ssc_of_view_altitude"
pack .l_mnt_altitude -in .fr_mnt_altitude
pack .b_mnt_altitude0 -side left -fill both -in .fr_mnt_altitude
pack .e_mnt_altitude -expand 1 -side left -fill both -in .fr_mnt_altitude
pack .b_mnt_altitude1 .b_mnt_altitude2 .b_mnt_altitude3 .b_mnt_altitude4 -side left -fill both -in .fr_mnt_altitude

set buffer_view_azimuth $asarr_initial_settings(ViewAzimuth)
frame .fr_mnt_azimuth
label .l_mnt_azimuth -text "View Azimuth" -font $font_small -pady 0
create_button_small .b_mnt_azimuth0 "初" "send_ssc_of_view_azimuth -"
entry .e_mnt_azimuth -width 6 -textvariable buffer_view_azimuth -font $font_small
#create_button_small .b_mnt_azimuth1 "SET" "send_ssc_of_view_azimuth"
create_button_small .b_mnt_azimuth1 "≪" "send_ssc_of_view_azimuth -10.0"
create_button_small .b_mnt_azimuth2 "<" "send_ssc_of_view_azimuth -1.0"
create_button_small .b_mnt_azimuth3 ">" "send_ssc_of_view_azimuth +1.0"
create_button_small .b_mnt_azimuth4 "≫" "send_ssc_of_view_azimuth +10.0"
bind .e_mnt_azimuth <Return> "send_ssc_of_view_azimuth"
pack .l_mnt_azimuth -in .fr_mnt_azimuth
pack .b_mnt_azimuth0 -side left -fill both -in .fr_mnt_azimuth
pack .e_mnt_azimuth -expand 1 -side left -fill both -in .fr_mnt_azimuth
pack .b_mnt_azimuth1 .b_mnt_azimuth2 .b_mnt_azimuth3 .b_mnt_azimuth4 -side left -fill both -in .fr_mnt_azimuth

pack .fr_mnt_pjmod .fr_mnt_gui .fr_mnt_gamma .fr_mnt_diskview .fr_mnt_fov .fr_mnt_vpx .fr_mnt_vpy .fr_mnt_altitude .fr_mnt_azimuth -expand 0 -fill x -in .lf_config

####

if { $flag_host_gui != 0 } {

  frame .fr_mnt_margin -height [scaled_px 4]

  frame .fr_mnt_edlbl
  label .l_mnt_edlbl -text "設定ファイル編集" -font $font_small -foreground $label_inverted_foreground -background $label_inverted_background
  pack .l_mnt_edlbl -side left -expand 1 -fill x -in .fr_mnt_edlbl

  pack .fr_mnt_margin .fr_mnt_edlbl -expand 0 -fill x -in .lf_config

  create_button_small .b_mnt_ed00 "localConfig.ssc" "open_text_editor .edt_cfg0 localConfig.ssc"
  create_button_small .b_mnt_ed01 "localInitView.ssc" "open_text_editor .edt_cfg1 localInitView.ssc"
  create_button_small .b_mnt_ed02 "localInitLocation.ssc" "open_text_editor .edt_cfg2 localInitLocation.ssc"
  create_button_small .b_mnt_ed03 "localInitSettings.ssc" "open_text_editor .edt_cfg3 localInitSettings.ssc"
  create_button_small .b_mnt_ed04 "localInitSunScale.ssc" "open_text_editor .edt_cfg4 localInitSunScale.ssc"
  create_button_small .b_mnt_ed05 "localInitMoonScale.ssc" "open_text_editor .edt_cfg5 localInitMoonScale.ssc"

  pack .b_mnt_ed00 .b_mnt_ed01 .b_mnt_ed02 .b_mnt_ed03 .b_mnt_ed04 .b_mnt_ed05 -expand 0 -fill x -in .lf_config

  if { $flag_dome_mode != 0 } {
    create_button_small .b_mnt_ed06 "exec_date-monitor.bat" "open_text_editor .edt_cfg6 exec_date-monitor.bat"
    pack .b_mnt_ed06 -expand 0 -fill x -in .lf_config
  }

}

.pw add .lf_config


################################################################

labelframe .lf_objinfo -text "Object Info" -font $font_labelframe

set buffer_object_info ""

#

frame .fr_objinfo_1

# GUI のサイズごとに横幅を調整する
if { $flag_large_gui == 0 } {
  # Normal GUI
  text .t_objinfo -font $font_small -width 68 -yscrollcommand ".sb_objinfo set" -state disabled -wrap char
} else {
  # Large GUI
  text .t_objinfo -font $font_small -width 69 -yscrollcommand ".sb_objinfo set" -state disabled -wrap char
}
scrollbar .sb_objinfo -command ".t_objinfo yview"

pack .t_objinfo -side left -expand 1 -fill both -in .fr_objinfo_1
pack .sb_objinfo -side left -expand 0 -fill y -in .fr_objinfo_1

pack .fr_objinfo_1 -expand 1 -fill both -in .lf_objinfo

#

frame .fr_objinfo_margin -height [scaled_px 2]
pack .fr_objinfo_margin -in .lf_objinfo

frame .fr_objinfo_2

create_button_small .b_objinfo_reload "Reload" "get_object_data_from_stellarium"
label .l_objinfo_spc -text "" -font $font_small
create_button_small .b_objinfo_copy "クリップボードへコピー" "set_object_text_to_clipboard"
pack .b_objinfo_reload  -side left -in .fr_objinfo_2
pack .l_objinfo_spc -expand 1 -side left -in .fr_objinfo_2
pack .b_objinfo_copy -side left -in .fr_objinfo_2

pack .fr_objinfo_2 -fill x -in .lf_objinfo

#

.pw add .lf_objinfo


################################################################

set list_documents {
  "https://stellarium.org/doc/head/scripting.html"
  "http://www.nct9.ne.jp/m_hiroi/tcl_tk_doc/"
}
set counter_list_documents 0

proc show_one_in_list_documents {} {
  global list_documents
  global counter_list_documents
  global buffer_codefield
  set len_list [llength $list_documents]
  if { $len_list <= $counter_list_documents } {
    set counter_list_documents 0
  }
  set buffer_codefield [lindex $list_documents $counter_list_documents]
  incr counter_list_documents
}

labelframe .lf_codefield -text "Log / Send or Exec an SSC" -font $font_labelframe

entry .e_code -textvariable buffer_codefield -font $font_codefield
bind .e_code <Return> "send_ssc_from_codefield"
create_button_code .b_code0 "Clr"  "set buffer_codefield \"\""
create_button_code .b_code1 "Date" "set_buffer_codefield_from_buffer_datearg"
create_button_code .b_code2 "Doc"  "show_one_in_list_documents"
create_button_code .b_code3 "EXEC" "send_ssc_from_codefield"
pack .b_code0 .b_code1 .b_code2 -side left -in .lf_codefield
pack .e_code -side left -expand 1 -fill both -in .lf_codefield
pack .b_code3 -side left -in .lf_codefield

pack .lf_codefield -side left -expand 1 -fill both -padx [scaled_px 4] -pady [scaled_px 3] -in .fr_rootwin

labelframe .lf_options -text "Options" -font $font_labelframe
ttk::checkbutton .cb_editor -text "外部Editor" -variable flag_external_editor -style "Small.TCheckbutton"
pack .cb_editor -in .lf_options

pack .lf_options -expand 0 -fill both -padx [scaled_px 4] -pady [scaled_px 3] -in .fr_rootwin



########################################################################

pack .fr_rootwin

########################################################################

deselect_all
select_general


####################################################################
##                                                                ##
##                          M A I N (2)                           ##
##                                                                ##
####################################################################

#
# 日付時刻ウィンドウを起動
#

exec_date_gui

#
# Host用 GUIの場合、設定ファイルをISC Serverのrootにコピーしてすぐ終了する
#
if { $flag_host_gui != 0 } {
  exec_media_gui -copy-only
}

#######################################################################

if { $flag_initialize != 0 } {

  if { $flag_host_gui != 0 } {

    set iscserv_root_directory ""

    # ISC Server の root ディレクトリを取得する
    do_http_get $iscserv_url ret buffer_codefield
    set lst [split [string trim $ret] ","]

    if { 1 < [llength $lst] } {
      set iscserv_root_directory [string trim [lindex $lst 1]]
    }

    # 設定ファイルをリセットする
    if { $iscserv_root_directory ne "" } {
      reset_a_file $iscserv_root_directory/_tmp_starLabels.isc.txt
      reset_a_file $iscserv_root_directory/_tmp_initialSettings.isc.txt
    } else {
      wm state . normal
      tk_messageBox -type ok -icon info -title "ERROR" -message "Cannot get iscserv_root_directory."
      exit
    }

  }

  send_ssc_to_initialize_all 0

} else {

  # ISC Serverに保存された初期値を取得
  get_initial_settings_to asarr_initial_settings

  set flag_initialized 0

  if { $flag_host_gui != 0 } {

    # 異なるモードの場合は全初期化

    set current_is_dome_mode [get_flag_dome_mode_on_server]

    if { $flag_dome_mode == 0 } {
      if { $current_is_dome_mode != 0 } {
        #exec_ssc_on_server unsetCardinal.ssc
        #exec_ssc_on_server unsetFlagDomeMode.ssc
        #exec_ssc_on_server setCardinal.ssc
        send_ssc_to_initialize_all 0
        set flag_initialized 1
      }
    } else {
      if { $current_is_dome_mode == 0 } {
        #exec_ssc_on_server unsetCardinal.ssc
        #exec_ssc_on_server setFlagDomeMode.ssc
        #exec_ssc_on_server setCardinal.ssc
        #send_ssc_to_initialize_view 0 -
        send_ssc_to_initialize_all 0
        set flag_initialized 1
      }
    }

  }

  if { $flag_initialized == 0 } {
    # GUIパラメータのリロード
    reload_all_gui_params
  }

}

#######################################################################

# display main window
wm state . normal


